/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ts.utils;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterators;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.n4JS.ArrowFunction;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.LocalArgumentsVariable;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.SuperLiteral;
import org.eclipse.n4js.n4JS.ThisArgProvider;
import org.eclipse.n4js.n4JS.ThisLiteral;
import org.eclipse.n4js.n4JS.ThisTarget;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Utility methods added to support ES6 arrow functions, aka lambdas.
 */
@SuppressWarnings("all")
public class LambdaUtils {
  /**
   * All {@link ThisLiteral}s occurring in <code>body</code> that refer to the same this-value;
   * ie without delving into functions (unless arrow functions) or into a declaration that introduces
   * a this context, ie any {@link ThisTarget} in general.
   */
  public static Iterator<EObject> thisLiterals(final Block body) {
    final Predicate<EObject> _function = (EObject it) -> {
      boolean _introducesThisContext = LambdaUtils.introducesThisContext(it);
      return (!_introducesThisContext);
    };
    final Function1<EObject, Boolean> _function_1 = (EObject it) -> {
      return Boolean.valueOf((it instanceof ThisLiteral));
    };
    return IteratorExtensions.<EObject>filter(EcoreUtilN4.getAllContentsFiltered(body, _function), _function_1);
  }
  
  private static boolean introducesThisContext(final EObject node) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (node instanceof FunctionExpression) {
      _matched=true;
      boolean _isArrowFunction = ((FunctionExpression)node).isArrowFunction();
      _switchResult = (!_isArrowFunction);
    }
    if (!_matched) {
      if (node instanceof FunctionDefinition) {
        _matched=true;
        _switchResult = true;
      }
    }
    if (!_matched) {
      if (node instanceof ThisArgProvider) {
        _matched=true;
        _switchResult = true;
      }
    }
    if (!_matched) {
      _switchResult = false;
    }
    return _switchResult;
  }
  
  /**
   * All references occurring in <code>body</code> that target the given {@link LocalArgumentsVariable}.
   * 
   * For better speed early pruning is performed (ie, without delving into functions, unless arrow functions,
   * or into a declaration that introduces a this context, ie without delving into any {@link ThisTarget} in general).
   */
  public static Iterator<? extends IdentifierRef> argumentsOccurrences(final Block body, final LocalArgumentsVariable target) {
    if ((target == null)) {
      return CollectionLiterals.<IdentifierRef>emptyList().iterator();
    }
    final Predicate<EObject> _function = (EObject it) -> {
      boolean _introducesThisContext = LambdaUtils.introducesThisContext(it);
      return (!_introducesThisContext);
    };
    final Function1<IdentifierRef, Boolean> _function_1 = (IdentifierRef it) -> {
      IdentifiableElement _id = it.getId();
      return Boolean.valueOf((_id == target));
    };
    return IteratorExtensions.<IdentifierRef>filter(Iterators.<IdentifierRef>filter(EcoreUtilN4.getAllContentsFiltered(body, _function), IdentifierRef.class), _function_1);
  }
  
  /**
   * TODO For now, usages of 'arguments' inside an arrow function are resolved to the {@link LocalArgumentsVariable}
   * owned by that arrow function. In fact, they should be resolved (together with similar usages in
   * nested arrow functions, without delving into this-introducing contexts) to whatever 'arguments' is in effect.
   * This method is used to patch up that resolution.
   * <p>
   * In detail, this method collects usages of 'arguments', walking down nested arrow-functions,
   * stopping at arguments-introducing functions (ie, non-arrow ones). The result contains
   * uses that point to different {@link LocalArgumentsVariable} instances. That's intended.
   * For the purposes of alpha-renaming, all those uses denote the same arguments-capture,
   * and thus will all be rewritten to the same identifier (a fresh name) in the xpiler's output.
   */
  public static Iterator<? extends IdentifierRef> allArgumentsUsages(final FunctionOrFieldAccessor argumentsBindingConstruct) {
    Iterator<? extends IdentifierRef> result = LambdaUtils.argumentsOccurrences(argumentsBindingConstruct.getBody(), argumentsBindingConstruct.get_lok());
    final Function1<FunctionExpression, Iterator<? extends IdentifierRef>> _function = (FunctionExpression it) -> {
      return LambdaUtils.allArgumentsUsages(it);
    };
    List<Iterator<? extends IdentifierRef>> _map = ListExtensions.<FunctionExpression, Iterator<? extends IdentifierRef>>map(LambdaUtils.directlyEnclosedLambdas(argumentsBindingConstruct.getBody()), _function);
    for (final Iterator<? extends IdentifierRef> nestedResult : _map) {
      result = Iterators.<IdentifierRef>concat(result, nestedResult);
    }
    boolean _isEmpty = IteratorExtensions.isEmpty(result);
    if (_isEmpty) {
      return Collections.<IdentifierRef>emptyIterator();
    } else {
      return result;
    }
  }
  
  /**
   * All {@link SuperLiteral}s occurring in <code>body</code> delving all the way.
   */
  public static Iterator<SuperLiteral> superLiterals(final Block body) {
    return Iterators.<SuperLiteral>filter(body.eAllContents(), SuperLiteral.class);
  }
  
  /**
   * All arrow functions directly enclosed in <code>body</code> ie, without delving into functions, not even into arrow functions,
   * or into a declaration that introduces a this context, ie without delving into any {@link ThisTarget} in general).
   */
  public static List<FunctionExpression> directlyEnclosedLambdas(final Block body) {
    final HashSet<FunctionExpression> result = new HashSet<FunctionExpression>();
    final Set<ArrowFunction> candidates = IteratorExtensions.<ArrowFunction>toSet(LambdaUtils.allLambdas(body));
    result.addAll(candidates);
    for (final ArrowFunction candidate : candidates) {
      {
        final Function1<FunctionExpression, Boolean> _function = (FunctionExpression it) -> {
          return Boolean.valueOf(((candidate != it) && LambdaUtils.isEnclosedBy(candidate, it)));
        };
        final boolean isCandidateItselfNested = IterableExtensions.<FunctionExpression>exists(result, _function);
        if (isCandidateItselfNested) {
          result.remove(candidate);
        }
      }
    }
    return IterableExtensions.<FunctionExpression>toList(result);
  }
  
  /**
   * All arrow functions, be they directly or transitively enclosed in <code>body</code>. The search doesn't delve into non-arrow functions,
   * same goes for declarations that introduce a this context, ie without delving into any {@link ThisTarget} in general).
   */
  public static Iterator<ArrowFunction> allLambdas(final Block body) {
    final Predicate<EObject> _function = (EObject it) -> {
      boolean _introducesThisContext = LambdaUtils.introducesThisContext(it);
      return (!_introducesThisContext);
    };
    return Iterators.<ArrowFunction>filter(EcoreUtilN4.getAllContentsFiltered(body, _function), ArrowFunction.class);
  }
  
  public static boolean isEnclosedBy(final EObject inner, final EObject outer) {
    return EcoreUtil.isAncestor(outer, inner);
  }
  
  public static boolean isLambda(final EObject funDef) {
    return (funDef instanceof ArrowFunction);
  }
  
  public static boolean isSingleExprImplicitReturn(final FunctionOrFieldAccessor funDef) {
    boolean _xifexpression = false;
    boolean _isLambda = LambdaUtils.isLambda(funDef);
    if (_isLambda) {
      _xifexpression = ((ArrowFunction) funDef).isSingleExprImplicitReturn();
    } else {
      _xifexpression = false;
    }
    return _xifexpression;
  }
  
  /**
   * A this-binder is a non-lambda {@link FunctionDefinition}.
   * This method looks up the nearest such construct that encloses the argument.
   * In case the argument itself is a this-binder, it is returned.
   * In case no this-binder exists (for example, the argument is enclosed in a top-level arrow-function) then null is returned.
   * <p>
   * Note: unlike {@link N4JSASTUtils#getContainingFunctionOrAccessor(EObject)} this method
   * regards {@link N4FieldDeclaration} as valid answer (ie, a "nearest enclosing this-binder").
   */
  public static ThisArgProvider nearestEnclosingThisBinder(final EObject expr) {
    if ((expr == null)) {
      return null;
    }
    final ThisArgProvider enclosingThisBinder = EcoreUtil2.<ThisArgProvider>getContainerOfType(expr, ThisArgProvider.class);
    if ((enclosingThisBinder == null)) {
      return null;
    }
    boolean _isLambda = LambdaUtils.isLambda(enclosingThisBinder);
    if (_isLambda) {
      return LambdaUtils.nearestEnclosingThisBinder(enclosingThisBinder.eContainer());
    }
    return enclosingThisBinder;
  }
  
  /**
   * A top-level lambda has no enclosing lexical context that provides bindings for 'this' and 'arguments'.
   */
  public static boolean isTopLevelLambda(final FunctionExpression funExpr) {
    boolean _and = false;
    boolean _isLambda = LambdaUtils.isLambda(funExpr);
    if (!_isLambda) {
      _and = false;
    } else {
      boolean _xblockexpression = false;
      {
        final EObject enclosing = funExpr.eContainer();
        ThisArgProvider _nearestEnclosingThisBinder = LambdaUtils.nearestEnclosingThisBinder(enclosing);
        _xblockexpression = (null == _nearestEnclosingThisBinder);
      }
      _and = _xblockexpression;
    }
    return _and;
  }
}
