/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.organize.imports;

import com.google.common.base.Objects;
import com.google.common.base.Optional;
import com.google.common.collect.Iterables;
import com.google.common.collect.LinkedHashMultimap;
import com.google.common.collect.Multimap;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.MemberAccess;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.organize.imports.ImportStateCalculator;
import org.eclipse.n4js.organize.imports.RecordingImportState;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.scoping.accessModifiers.TypeVisibilityChecker;
import org.eclipse.n4js.scoping.accessModifiers.VariableVisibilityChecker;
import org.eclipse.n4js.scoping.utils.ImportSpecifierUtil;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ui.contentassist.N4JSCandidateFilter;
import org.eclipse.n4js.ui.organize.imports.BreakException;
import org.eclipse.n4js.ui.organize.imports.DisambiguateUtil;
import org.eclipse.n4js.ui.organize.imports.ImportDeclarationTextHelper;
import org.eclipse.n4js.ui.organize.imports.ImportProvidedElementLabelprovider;
import org.eclipse.n4js.ui.organize.imports.ImportableObject;
import org.eclipse.n4js.ui.organize.imports.ImportsFactory;
import org.eclipse.n4js.ui.organize.imports.ImportsSorter;
import org.eclipse.n4js.ui.organize.imports.Interaction;
import org.eclipse.n4js.ui.organize.imports.ReferenceProxyInfo;
import org.eclipse.n4js.ui.organize.imports.UnresolveProxyCrossRefHelper;
import org.eclipse.n4js.ui.organize.imports.XtextResourceUtils;
import org.eclipse.n4js.utils.Log;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescription;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.MapExtensions;

/**
 * Computes imports required by the given resource. In principle removes unused imports, adds missing imports, sorts imports - all in one go.
 * Does not actually change the resource, but prepares final state that callers can apply to the resource.
 */
@Log
@SuppressWarnings("all")
public class ImportsComputer {
  @Inject
  private ImportStateCalculator importStateCalculator;
  
  @Inject
  private ImportsFactory importsFactory;
  
  @Inject
  private ImportProvidedElementLabelprovider importProvidedElementLabelprovider;
  
  @Inject
  private N4JSCandidateFilter candidateFilter;
  
  @Inject
  private IN4JSCore core;
  
  @Inject
  private ImportDeclarationTextHelper declarationTextHelper;
  
  @Inject
  private TypeVisibilityChecker typeVisibilityChecker;
  
  @Inject
  private VariableVisibilityChecker varVisibilityChecker;
  
  @Inject
  private UnresolveProxyCrossRefHelper crossRef;
  
  /**
   * Adapter used to mark programmatically created AST-Elements without a corresponding parse tree node.
   */
  private final Adapter nodelessMarker = new AdapterImpl();
  
  /**
   * Calculate the real content of the new import section in the file header.
   * 
   * @param xtextResource the resource to organize
   * @param lineEnding current active line ending in file
   * @param interaction mode of handling ambiguous situations
   * @return new import section, might be an empty string
   * @throws BreakException when import resolution is ambiguous and mode is {@link Interaction#breakBuild}
   */
  public String getOrganizedImportSection(final XtextResource xtextResource, final String lineEnding, final Interaction interaction) throws BreakException {
    final StringBuilder sb = new StringBuilder();
    final Script script = XtextResourceUtils.getScript(xtextResource);
    final List<ImportDeclaration> resultingImports = IterableExtensions.<ImportDeclaration>toList(Iterables.<ImportDeclaration>filter(script.getScriptElements(), ImportDeclaration.class));
    final RecordingImportState reg = this.importStateCalculator.calculateImportstate(script);
    reg.removeDuplicatedImportDeclarations(resultingImports);
    reg.removeLocalNameCollisions(resultingImports, this.nodelessMarker);
    reg.removeDuplicatedImportsOfSameelement(resultingImports, this.nodelessMarker);
    reg.removeBrokenImports(resultingImports, this.nodelessMarker);
    reg.removeUnusedImports(resultingImports, this.nodelessMarker);
    final Set<String> brokenNames = reg.calcRemovedImportedNames();
    ArrayList<ImportDeclaration> _resolveMissingImports = this.resolveMissingImports(script, brokenNames, interaction);
    Iterables.<ImportDeclaration>addAll(resultingImports, _resolveMissingImports);
    final Consumer<ImportDeclaration> _function = (ImportDeclaration it) -> {
      EcoreUtil.resolveAll(it);
    };
    resultingImports.forEach(_function);
    ImportsSorter.sortByImport(resultingImports);
    final Consumer<ImportDeclaration> _function_1 = (ImportDeclaration it) -> {
      final String text = this.declarationTextHelper.extractPureText(it, xtextResource, this.nodelessMarker);
      sb.append(text).append(lineEnding);
    };
    resultingImports.forEach(_function_1);
    final int length = sb.length();
    int _length = lineEnding.length();
    boolean _greaterThan = (length > _length);
    if (_greaterThan) {
      int _length_1 = lineEnding.length();
      int _minus = (length - _length_1);
      sb.delete(_minus, length);
    }
    return sb.toString();
  }
  
  /**
   * Calculate new Imports.
   */
  private ArrayList<ImportDeclaration> resolveMissingImports(final Script script, final Set<String> namesThatWeBroke, final Interaction interaction) throws BreakException {
    final IN4JSProject contextProject = this.core.findProject(script.eResource().getURI()).orNull();
    final Multimap<String, ImportableObject> resolutions = this.createResolutionsForBrokenNames(script, contextProject, namesThatWeBroke);
    final Function2<String, Collection<ImportableObject>, Boolean> _function = (String p1, Collection<ImportableObject> p2) -> {
      int _size = p2.size();
      return Boolean.valueOf((_size == 1));
    };
    final Map<String, Collection<ImportableObject>> solutions = MapExtensions.<String, Collection<ImportableObject>>filter(resolutions.asMap(), _function);
    final ArrayList<ImportDeclaration> ret = CollectionLiterals.<ImportDeclaration>newArrayList();
    final BiConsumer<String, Collection<ImportableObject>> _function_1 = (String name, Collection<ImportableObject> importable) -> {
      final ImportableObject imp = IterableExtensions.<ImportableObject>head(importable);
      ret.add(this.importsFactory.createImport(imp, contextProject, this.nodelessMarker));
    };
    solutions.forEach(_function_1);
    namesThatWeBroke.removeAll(solutions.keySet());
    final Function2<String, Collection<ImportableObject>, Boolean> _function_2 = (String p1, Collection<ImportableObject> p2) -> {
      int _size = p2.size();
      return Boolean.valueOf((_size > 1));
    };
    final Map<String, Collection<ImportableObject>> ambiguousSolution = MapExtensions.<String, Collection<ImportableObject>>filter(resolutions.asMap(), _function_2);
    final Multimap<String, ImportableObject> forDisambiguation = LinkedHashMultimap.<String, ImportableObject>create();
    final BiConsumer<String, Collection<ImportableObject>> _function_3 = (String p1, Collection<ImportableObject> p2) -> {
      forDisambiguation.putAll(p1, p2);
    };
    ambiguousSolution.forEach(_function_3);
    final HashSet<String> resolved = new HashSet<String>();
    final Consumer<String> _function_4 = (String key) -> {
      final Collection<ImportableObject> multiSolutions = forDisambiguation.get(key);
      final Function1<ImportableObject, Boolean> _function_5 = (ImportableObject it) -> {
        boolean _isAsNamespace = it.isAsNamespace();
        return Boolean.valueOf((!_isAsNamespace));
      };
      final boolean containsOnlyNamespaces = IterableExtensions.isEmpty(IterableExtensions.<ImportableObject>filter(multiSolutions, _function_5));
      if (containsOnlyNamespaces) {
        ret.add(this.importsFactory.createImport(IterableExtensions.<ImportableObject>head(multiSolutions), contextProject, this.nodelessMarker));
        resolved.add(key);
      }
    };
    forDisambiguation.keySet().forEach(_function_4);
    final Consumer<String> _function_5 = (String key) -> {
      forDisambiguation.removeAll(key);
    };
    resolved.forEach(_function_5);
    final List<ImportableObject> chosenSolutions = DisambiguateUtil.<ImportableObject>disambiguate(forDisambiguation, interaction, 
      this.importProvidedElementLabelprovider);
    final Consumer<ImportableObject> _function_6 = (ImportableObject it) -> {
      ret.add(this.importsFactory.createImport(it, contextProject, this.nodelessMarker));
    };
    chosenSolutions.forEach(_function_6);
    return ret;
  }
  
  /**
   * Finds unresolved cross references in this script and combines them with provided broken names.
   * @returns list of resolutions for all broken names.
   */
  private Multimap<String, ImportableObject> createResolutionsForBrokenNames(final Script script, final IN4JSProject contextProject, final Set<String> namesThatWeBroke) {
    final Multimap<String, ImportableObject> resolutions = LinkedHashMultimap.<String, ImportableObject>create();
    final Function1<ReferenceProxyInfo, Boolean> _function = (ReferenceProxyInfo it) -> {
      return Boolean.valueOf((Boolean.valueOf((it.eobject instanceof MemberAccess)) == Boolean.valueOf(false)));
    };
    final Iterable<ReferenceProxyInfo> unresolved = IterableExtensions.<ReferenceProxyInfo>filter(this.crossRef.findProxyCrossRefInfo(script), _function);
    final HashSet<String> brokenNames = new HashSet<String>();
    brokenNames.addAll(namesThatWeBroke);
    final Function1<ReferenceProxyInfo, String> _function_1 = (ReferenceProxyInfo it) -> {
      return it.name;
    };
    Iterables.<String>addAll(brokenNames, IterableExtensions.<ReferenceProxyInfo, String>map(unresolved, _function_1));
    this.addResolutionsFromIndex(resolutions, contextProject, brokenNames, script.eResource());
    return resolutions;
  }
  
  /**
   * Obtains index based on the provided resource. Matches all broken names against object descriptions in the index.
   * Those that pass checks are added to the resolutions map as potential solutions.
   * <p>
   * Note that similar results can be achieved by querying scopes (i.e. TypeRef scope), as it was before. Unfortunately
   * browsing the scope is much slower than index. For performance reasons we are using index, even though we need to
   * duplicate some scope semantics, e.g. check for visibility. Still performance does not allow us to use scopes here.
   */
  private void addResolutionsFromIndex(final Multimap<String, ImportableObject> resolution, final IN4JSProject contextProject, final Iterable<String> brokenNames, final Resource contextResource) {
    final ResourceSet resourceSet = this.core.createResourceSet(Optional.<IN4JSProject>fromNullable(contextProject));
    final Iterable<IResourceDescription> resources = this.core.getXtextIndex(resourceSet).getAllResourceDescriptions();
    final Consumer<IResourceDescription> _function = (IResourceDescription res) -> {
      final IN4JSProject candidateProject = this.core.findProject(res.getURI()).orNull();
      if ((candidateProject != null)) {
        IN4JSProject _dependencyWithID = ImportSpecifierUtil.getDependencyWithID(candidateProject.getProjectName(), contextProject);
        boolean _tripleNotEquals = (_dependencyWithID != null);
        if (_tripleNotEquals) {
          final Iterator<IEObjectDescription> exportedIEODs = res.getExportedObjects().iterator();
          while (exportedIEODs.hasNext()) {
            {
              final IEObjectDescription ieod = exportedIEODs.next();
              boolean _isNotModule = this.isNotModule(ieod.getEObjectURI());
              if (_isNotModule) {
                final Consumer<String> _function_1 = (String usedName) -> {
                  if (((this.candidateFilter.apply(ieod) && 
                    this.isCandidate(ieod, usedName)) && 
                    this.isVisible(ieod, contextResource))) {
                    this.add(resolution, usedName, ieod, contextResource);
                  }
                };
                brokenNames.forEach(_function_1);
              }
            }
          }
        }
      }
    };
    resources.forEach(_function);
  }
  
  /**
   * Creates {@link ImportableObject} from provided name and object description. Result is added to the collection.
   */
  private boolean add(final Multimap<String, ImportableObject> resolution, final String usedName, final IEObjectDescription ieoDescription, final Resource contextResource) {
    boolean _contains = usedName.contains(".");
    if (_contains) {
      final String[] segments = usedName.split("\\.");
      int _size = ((List<String>)Conversions.doWrapArray(segments)).size();
      boolean _notEquals = (_size != 2);
      if (_notEquals) {
        return false;
      }
      final EObject eo = this.tryGetEObjectOrProxy(ieoDescription.getEObjectOrProxy(), contextResource);
      if ((Boolean.valueOf((eo instanceof TExportableElement)) == Boolean.valueOf(false))) {
        return false;
      }
      String _exportedName = ((TExportableElement) eo).getExportedName();
      Object _last = IterableExtensions.<Object>last(((Iterable<Object>)Conversions.doWrapArray(segments)));
      boolean _notEquals_1 = (!Objects.equal(_exportedName, _last));
      if (_notEquals_1) {
        return false;
      }
      String _head = IterableExtensions.<String>head(((Iterable<String>)Conversions.doWrapArray(segments)));
      String _head_1 = IterableExtensions.<String>head(((Iterable<String>)Conversions.doWrapArray(segments)));
      boolean _isDefaultExport = this.isDefaultExport(ieoDescription, contextResource);
      ImportableObject _importableObject = new ImportableObject(_head_1, ((TExportableElement) eo), _isDefaultExport, true);
      return resolution.put(_head, _importableObject);
    } else {
      final EObject eo_1 = this.tryGetEObjectOrProxy(ieoDescription.getEObjectOrProxy(), contextResource);
      boolean _isDefaultExport_1 = this.isDefaultExport(ieoDescription, contextResource);
      ImportableObject _importableObject_1 = new ImportableObject(usedName, ((TExportableElement) eo_1), _isDefaultExport_1);
      return resolution.put(usedName, _importableObject_1);
    }
  }
  
  private boolean isDefaultExport(final IEObjectDescription description, final Resource contextResource) {
    final EObject eo = this.tryGetEObjectOrProxy(description.getEObjectOrProxy(), contextResource);
    if ((eo instanceof TExportableElement)) {
      String _exportedName = ((TExportableElement)eo).getExportedName();
      return Objects.equal(_exportedName, N4JSLanguageConstants.EXPORT_DEFAULT_NAME);
    }
    return false;
  }
  
  /**
   * return true if {@code description} is a possible candidate for an element with name {@code usedName}.
   */
  private boolean isCandidate(final IEObjectDescription description, final String usedName) {
    final QualifiedName qName = description.getName();
    String _lastSegment = qName.getLastSegment();
    boolean _equals = Objects.equal(_lastSegment, usedName);
    if (_equals) {
      return true;
    }
    boolean _contains = usedName.contains(".");
    if (_contains) {
      final String[] segments = usedName.split("\\.");
      if (((((List<String>)Conversions.doWrapArray(segments)).size() == 2) && Objects.equal(IterableExtensions.<String>last(((Iterable<String>)Conversions.doWrapArray(segments))), qName.getLastSegment()))) {
        return true;
      }
    }
    if ((Objects.equal(qName.getLastSegment(), N4JSLanguageConstants.EXPORT_DEFAULT_NAME) && (qName.getSegmentCount() > 1))) {
      int _segmentCount = qName.getSegmentCount();
      int _minus = (_segmentCount - 2);
      final String moduleName = qName.getSegment(_minus).toLowerCase();
      final String usedSimpleName = usedName.toLowerCase();
      boolean _equals_1 = Objects.equal(usedSimpleName, moduleName);
      if (_equals_1) {
        return true;
      }
      final String simplifiedModuleName = moduleName.replaceAll("-|_", "");
      boolean _equals_2 = Objects.equal(usedSimpleName, simplifiedModuleName);
      if (_equals_2) {
        return true;
      }
    }
    return false;
  }
  
  /**
   * In most cases we cannot compute imports from proxy objects, as essential information like declared name, exported name etc.
   * are not available. If the provided object is a proxy, tries to resolve it. If object is still a proxy logs a warning. In either case returns result of resolving a proxy
   * @returns resolved EObject or proxy
   */
  private EObject tryGetEObjectOrProxy(final EObject eobject, final Resource contextResource) {
    EObject eo = eobject;
    boolean _eIsProxy = eo.eIsProxy();
    if (_eIsProxy) {
      eo = EcoreUtil.resolve(eo, contextResource);
      String data = "";
      boolean _eIsProxy_1 = eo.eIsProxy();
      if (_eIsProxy_1) {
        if ((eo instanceof TClass)) {
          String _name = ((TClass)eo).getName();
          String _plus = (_name + " ");
          data = _plus;
        }
        ImportsComputer.logger.warn((("Cannot resolve proxy " + data) + eo));
      }
    }
    return eo;
  }
  
  /**
   * Checks if the provided uri does not point to a module. For {@link IEObjectDescription} instances
   * the URI for modules (files) will be in form {@code some/path/1} while for the AST elements inside the module
   * it will be {@code some/path/1/@topLevelTypes.4} or {@code some/path/1/@variables.7}
   */
  private boolean isNotModule(final URI uri) {
    String _fragment = null;
    if (uri!=null) {
      _fragment=uri.fragment();
    }
    final boolean res = _fragment.endsWith("/1");
    return (!res);
  }
  
  private boolean isVisible(final IEObjectDescription ieod, final Resource contextResource) {
    final EObject eo = this.tryGetEObjectOrProxy(ieod.getEObjectOrProxy(), contextResource);
    if (((eo instanceof Type) && this.isSubtypeOfType(eo.eClass()))) {
      return this.typeVisibilityChecker.isVisible(contextResource, ((Type) eo)).visibility;
    }
    if (((eo instanceof TVariable) && this.isSubtypeOfIdentifiable(eo.eClass()))) {
      return this.varVisibilityChecker.isVisible(contextResource, ((TVariable) eo)).visibility;
    }
    return false;
  }
  
  /**
   * Returns <code>true</code> if the given {@code type} is a subtype of {@link IdentifiableElement}.
   * @see org.eclipse.n4js.scoping.N4JSGlobalScopeProvider.isSubtypeOfIdentifiable(EClass)
   */
  protected boolean isSubtypeOfIdentifiable(final EClass type) {
    return ((type == TypesPackage.Literals.IDENTIFIABLE_ELEMENT) || ((type.getEPackage() == TypesPackage.eINSTANCE) && 
      TypesPackage.Literals.IDENTIFIABLE_ELEMENT.isSuperTypeOf(type)));
  }
  
  /**
   * Returns <code>true</code> if the given {@code type} is a subtype of {@link Type}.
   * @see org.eclipse.n4js.ts.scoping.builtin.DefaultN4GlobalScopeProvider.isSubtypeOfType(EClass)
   */
  protected boolean isSubtypeOfType(final EClass type) {
    return ((type == TypesPackage.Literals.TYPE) || 
      ((type.getEPackage() == TypesPackage.eINSTANCE) && TypesPackage.Literals.TYPE.isSuperTypeOf(type)));
  }
  
  private final static Logger logger = Logger.getLogger(ImportsComputer.class);
}
