/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.organize.imports;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.ts.types.TExportableElement;

/**
 * Provides methods for import declarations sorting.
 */
@SuppressWarnings("all")
public class ImportsSorter {
  /**
   * Sorting a List of import declarations (mixed content Named / Namespace)
   * Order is: First all Named imports, then all Namespace imports.
   */
  public static final void sortByImport(final List<ImportDeclaration> declarations) {
    declarations.sort(new Comparator<ImportDeclaration>() {
      @Override
      public int compare(final ImportDeclaration o1, final ImportDeclaration o2) {
        int _switchResult = (int) 0;
        ImportSpecifier _get = o1.getImportSpecifiers().get(0);
        boolean _matched = false;
        if (_get instanceof NamespaceImportSpecifier) {
          _matched=true;
          int _xifexpression = (int) 0;
          ImportSpecifier _get_1 = o2.getImportSpecifiers().get(0);
          if ((_get_1 instanceof NamespaceImportSpecifier)) {
            _xifexpression = ImportsSorter.compModules(o1, o2);
          } else {
            _xifexpression = 1;
          }
          _switchResult = _xifexpression;
        }
        if (!_matched) {
          if (_get instanceof NamedImportSpecifier) {
            _matched=true;
            int _xifexpression = (int) 0;
            ImportSpecifier _get_1 = o2.getImportSpecifiers().get(0);
            if ((_get_1 instanceof NamespaceImportSpecifier)) {
              _xifexpression = (-1);
            } else {
              int _xblockexpression = (int) 0;
              {
                int cmp = ImportsSorter.compModules(o1, o2);
                int _xifexpression_1 = (int) 0;
                if ((cmp == 0)) {
                  _xifexpression_1 = ImportsSorter.compNamedImports(o1.getImportSpecifiers(), o2.getImportSpecifiers());
                } else {
                  _xifexpression_1 = cmp;
                }
                _xblockexpression = _xifexpression_1;
              }
              _xifexpression = _xblockexpression;
            }
            _switchResult = _xifexpression;
          }
        }
        if (!_matched) {
          throw new UnsupportedOperationException("Unknown ImportSpecifier");
        }
        return _switchResult;
      }
    });
  }
  
  /**
   * Sorting a List of import specifiers based on used names.
   * Order is: First all Named imports, then all Namespace imports.
   */
  public static final void sortByName(final List<ImportSpecifier> list) {
    Collections.<ImportSpecifier>sort(list, new Comparator<ImportSpecifier>() {
      @Override
      public int compare(final ImportSpecifier o1, final ImportSpecifier o2) {
        if ((o1 instanceof NamespaceImportSpecifier)) {
          return 1;
        } else {
          if ((o2 instanceof NamespaceImportSpecifier)) {
            return (-1);
          } else {
            return ImportsSorter.compNamedImport(((NamedImportSpecifier) o1), ((NamedImportSpecifier) o2));
          }
        }
      }
    });
  }
  
  /**
   * Compare list of NamedImports.
   *  Comparing elements in sequential order until inequality is found.
   */
  private static final int compNamedImports(final EList<ImportSpecifier> l1, final EList<ImportSpecifier> l2) {
    final int comparable_elements = Math.min(l1.size(), l2.size());
    for (int i = 0; (i < comparable_elements); i++) {
      {
        ImportSpecifier _get = l1.get(i);
        ImportSpecifier _get_1 = l2.get(i);
        final int cmp = ImportsSorter.compNamedImport(((NamedImportSpecifier) _get), ((NamedImportSpecifier) _get_1));
        if ((cmp != 0)) {
          return cmp;
        }
      }
    }
    int _size = l1.size();
    int _size_1 = l2.size();
    return (_size - _size_1);
  }
  
  /**
   * Compares two NamedImport specifier: e.g. "Z as A" <--> "X as B"
   */
  private static final int compNamedImport(final NamedImportSpecifier o1, final NamedImportSpecifier o2) {
    int _xblockexpression = (int) 0;
    {
      TExportableElement _importedElement = null;
      if (o1!=null) {
        _importedElement=o1.getImportedElement();
      }
      String _name = null;
      if (_importedElement!=null) {
        _name=_importedElement.getName();
      }
      String name1 = _name;
      TExportableElement _importedElement_1 = null;
      if (o2!=null) {
        _importedElement_1=o2.getImportedElement();
      }
      String _name_1 = null;
      if (_importedElement_1!=null) {
        _name_1=_importedElement_1.getName();
      }
      String name2 = _name_1;
      String _elvis = null;
      if (name1 != null) {
        _elvis = name1;
      } else {
        _elvis = "";
      }
      String _elvis_1 = null;
      if (name2 != null) {
        _elvis_1 = name2;
      } else {
        _elvis_1 = "";
      }
      final int cmp1 = _elvis.compareTo(_elvis_1);
      int _xifexpression = (int) 0;
      if ((cmp1 == 0)) {
        String _elvis_2 = null;
        String _alias = o1.getAlias();
        if (_alias != null) {
          _elvis_2 = _alias;
        } else {
          _elvis_2 = "";
        }
        String _elvis_3 = null;
        String _alias_1 = o2.getAlias();
        if (_alias_1 != null) {
          _elvis_3 = _alias_1;
        } else {
          _elvis_3 = "";
        }
        _xifexpression = _elvis_2.compareTo(_elvis_3);
      } else {
        _xifexpression = cmp1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * compare based on Qualified name
   */
  private static final int compModules(final ImportDeclaration o1, final ImportDeclaration o2) {
    String _elvis = null;
    String _qualifiedName = o1.getModule().getQualifiedName();
    if (_qualifiedName != null) {
      _elvis = _qualifiedName;
    } else {
      _elvis = "";
    }
    String _elvis_1 = null;
    String _qualifiedName_1 = o2.getModule().getQualifiedName();
    if (_qualifiedName_1 != null) {
      _elvis_1 = _qualifiedName_1;
    } else {
      _elvis_1 = "";
    }
    return _elvis.compareTo(_elvis_1);
  }
}
