/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.xpect.config;

import java.util.Collections;
import java.util.List;
import org.apache.log4j.Logger;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.ts.typeRefs.TypeRefsPackage;
import org.eclipse.n4js.ts.types.MemberAccessModifier;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.xpect.config.ValueList;
import org.eclipse.xpect.setup.XpectSetupComponent;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.scoping.IScopeProvider;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This Component generates a visibility-filtered list of feature names from a type. The type is loaded when evaluating
 * this component using type-information from the currently loaded script.
 * <p>
 * This Component can be part of {@link VarDef}
 * <p>
 * The MemberList takes three String-arguments in the constructor:
 * <code>MemberList "myType" "myFeatureType" "myVisibility" { }</code>
 * <p>
 * Example:<br>
 * <code>MemberList  "Object" "methods" "public" {}</code> <br/>
 * 
 * will evaluate to <br>
 * {@code "constructor","hasOwnPorperty","isPrototypeOf",...}<br>
 * when evaluated in a ECMAScript 5 execution environment
 * <p>
 * See {@link StringList} {@link VarDef} {@link Config}
 * 
 * @param type
 *            - type to be resolved against the defined execution environment, e.g. "Object".
 * @param featureType
 *            - what features of the type should be queried, e.g. "methods"
 * @param visibility
 *            - visibility filter, e.g. "public","publicInternal","protected","protectedInternal","project","private".
 */
@XpectSetupComponent
@SuppressWarnings("all")
public class MemberList implements ValueList {
  private static Logger logger = Logger.getLogger(MemberList.class);
  
  public String typename;
  
  public String featureType;
  
  public String visibility;
  
  /**
   * @param type - type to be resolved against the defined execution environment.
   * @param featureType - what features of the type should be queried, e.g. 'methods'
   * @param visibility - visibility filter.
   */
  public MemberList(final String type, final String featureType, final String visibility) {
    this.typename = type;
    this.featureType = featureType;
    this.visibility = visibility;
  }
  
  @Override
  public String toString() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Memberlist(o=\'");
    _builder.append(this.typename);
    _builder.append("\',f=\'");
    _builder.append(this.featureType);
    _builder.append("\',v=\'");
    _builder.append(this.visibility);
    _builder.append("\')");
    return _builder.toString();
  }
  
  /**
   * Evaluate this description against a current environment.
   */
  @Override
  public List<String> evaluate(final XtextResource res) {
    MemberList.logger.warn("just a dummy list in most cases. No impl yet.");
    final IScopeProvider scopeProv = res.getResourceServiceProvider().<IScopeProvider>get(IScopeProvider.class);
    final IScope scope2 = scopeProv.getScope(IterableExtensions.<EObject>head(res.getContents()), TypeRefsPackage.Literals.PARAMETERIZED_TYPE_REF__DECLARED_TYPE);
    EObject theType = scope2.getSingleElement(QualifiedName.create(this.typename)).getEObjectOrProxy();
    theType = EcoreUtil.resolve(theType, res);
    final TClassifier tClass = ((TClassifier) theType);
    MemberAccessModifier _elvis = null;
    MemberAccessModifier _byName = MemberAccessModifier.getByName(this.visibility);
    if (_byName != null) {
      _elvis = _byName;
    } else {
      _elvis = MemberAccessModifier.UNDEFINED;
    }
    final MemberAccessModifier modVisibility = _elvis;
    List<String> _switchResult = null;
    final String featureType = this.featureType;
    if (featureType != null) {
      switch (featureType) {
        case "methods":
          final Function1<TMember, Boolean> _function = (TMember it) -> {
            return Boolean.valueOf(it.isMethod());
          };
          final Function1<TMember, TMethod> _function_1 = (TMember it) -> {
            return ((TMethod) it);
          };
          final Function1<TMethod, Boolean> _function_2 = (TMethod it) -> {
            boolean _isStatic = it.isStatic();
            return Boolean.valueOf((!_isStatic));
          };
          final Function1<TMethod, Boolean> _function_3 = (TMethod it) -> {
            int _compareTo = it.getMemberAccessModifier().compareTo(modVisibility);
            return Boolean.valueOf((_compareTo >= 0));
          };
          final Function1<TMethod, String> _function_4 = (TMethod it) -> {
            return it.getName();
          };
          _switchResult = IterableExtensions.<String>toList(IterableExtensions.<TMethod, String>map(IterableExtensions.<TMethod>filter(IterableExtensions.<TMethod>filter(IterableExtensions.<TMember, TMethod>map(IterableExtensions.<TMember>filter(tClass.getOwnedMembers(), _function), _function_1), _function_2), _function_3), _function_4));
          break;
        default:
          _switchResult = Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("toString", "toLocaleString", "prototype"));
          break;
      }
    } else {
      _switchResult = Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("toString", "toLocaleString", "prototype"));
    }
    final List<String> list = _switchResult;
    return list;
  }
}
