/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.xpect.ui.common;

import com.google.common.collect.Multimap;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import junit.framework.AssertionFailedError;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.nodemodel.BidiTreeIterator;
import org.eclipse.xtext.nodemodel.ILeafNode;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.ui.editor.quickfix.IssueResolution;
import org.eclipse.xtext.validation.Issue;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pure;

@SuppressWarnings("all")
public class QuickFixTestHelper {
  /**
   * Container to track changed lines in multiline strings.
   */
  public static class ChangeInfo {
    @Accessors
    public static class ChangedLine {
      private final int lineNumber;
      
      private final int beforeOffset;
      
      private final String before;
      
      private final int afterOffset;
      
      private final String after;
      
      @Override
      public String toString() {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("L:");
        _builder.append(this.lineNumber);
        _builder.append("[");
        _builder.append(this.before);
        _builder.append("|");
        _builder.append(this.after);
        _builder.append("]");
        return _builder.toString();
      }
      
      public ChangedLine(final int lineNumber, final int beforeOffset, final String before, final int afterOffset, final String after) {
        super();
        this.lineNumber = lineNumber;
        this.beforeOffset = beforeOffset;
        this.before = before;
        this.afterOffset = afterOffset;
        this.after = after;
      }
      
      @Pure
      public int getLineNumber() {
        return this.lineNumber;
      }
      
      @Pure
      public int getBeforeOffset() {
        return this.beforeOffset;
      }
      
      @Pure
      public String getBefore() {
        return this.before;
      }
      
      @Pure
      public int getAfterOffset() {
        return this.afterOffset;
      }
      
      @Pure
      public String getAfter() {
        return this.after;
      }
    }
    
    private final List<QuickFixTestHelper.ChangeInfo.ChangedLine> changes = CollectionLiterals.<QuickFixTestHelper.ChangeInfo.ChangedLine>newArrayList();
    
    public ChangeInfo() {
    }
    
    public boolean add(final int n, final int beforeOffset, final String b, final int afterOffset, final String a) {
      QuickFixTestHelper.ChangeInfo.ChangedLine _changedLine = new QuickFixTestHelper.ChangeInfo.ChangedLine(n, beforeOffset, b, afterOffset, a);
      return this.changes.add(_changedLine);
    }
    
    public String asString() {
      final Function1<QuickFixTestHelper.ChangeInfo.ChangedLine, String> _function = (QuickFixTestHelper.ChangeInfo.ChangedLine it) -> {
        return (((("L:" + Integer.valueOf(it.lineNumber)) + " \'") + it.after) + "\'\n");
      };
      final Function2<String, String, String> _function_1 = (String p1, String p2) -> {
        return (p1 + p2);
      };
      return IterableExtensions.<String>reduce(ListExtensions.<QuickFixTestHelper.ChangeInfo.ChangedLine, String>map(this.changes, _function), _function_1);
    }
    
    public boolean isMoreThanOne() {
      int _size = this.changes.size();
      return (_size > 1);
    }
    
    public boolean isEmpty() {
      return this.changes.isEmpty();
    }
    
    public QuickFixTestHelper.ChangeInfo.ChangedLine first() {
      return this.changes.get(0);
    }
    
    public QuickFixTestHelper.ChangeInfo.ChangedLine get(final int i) {
      return this.changes.get(i);
    }
    
    public int size() {
      return this.changes.size();
    }
  }
  
  /**
   * Select all issues from {@link #offset2issue} concerning the {@link #offStartLine}
   * 
   * @param offStartLine line number
   * @param line2issue map of issues
   * @return sorted list of Issues
   */
  public static List<Issue> extractAllIssuesInLine(final int offStartLine, final Multimap<Integer, Issue> offset2issue) {
    final Function1<Issue, Boolean> _function = (Issue it) -> {
      Integer _lineNumber = it.getLineNumber();
      return Boolean.valueOf(((_lineNumber).intValue() == offStartLine));
    };
    final ArrayList<Issue> result = CollectionLiterals.<Issue>newArrayList(((Issue[])Conversions.unwrapArray(IterableExtensions.<Issue>toList(IterableExtensions.<Issue>filter(offset2issue.values(), _function)), Issue.class)));
    final Comparator<Issue> _function_1 = (Issue a, Issue b) -> {
      int _xblockexpression = (int) 0;
      {
        Integer _offset = b.getOffset();
        Integer _offset_1 = a.getOffset();
        final int first = ((_offset).intValue() - (_offset_1).intValue());
        int _xifexpression = (int) 0;
        if ((first != 0)) {
          _xifexpression = first;
        } else {
          _xifexpression = a.getMessage().compareToIgnoreCase(b.getMessage());
        }
        _xblockexpression = _xifexpression;
      }
      return _xblockexpression;
    };
    ListExtensions.<Issue>sortInplace(result, _function_1);
    return result;
  }
  
  /**
   * Find all EObjects beginning or in same line. Goes over all LeafNodes in same line and resolved the associated
   * EObjects.
   * 
   * @param offsetNode
   *            Node at cursor position
   * @return list of EObjects directly associated with this line. (but no wrapping Elements)
   */
  public static List<EObject> elementsInSameLine(final ILeafNode offsetNode) {
    final ArrayList<EObject> result = CollectionLiterals.<EObject>newArrayList();
    final int line = offsetNode.getStartLine();
    INode firstNode = null;
    final ArrayList<INode> nodesInLine = CollectionLiterals.<INode>newArrayList();
    final BidiTreeIterator<INode> iter = offsetNode.getRootNode().getAsTreeIterable().iterator();
    while (((firstNode == null) && iter.hasNext())) {
      {
        final INode next = iter.next();
        int _endLine = next.getEndLine();
        boolean _lessThan = (_endLine < line);
        if (_lessThan) {
          iter.prune();
        } else {
          if (((next.getStartLine() >= line) && (next instanceof ILeafNode))) {
            firstNode = next;
            nodesInLine.add(firstNode);
          }
        }
      }
    }
    while (iter.hasNext()) {
      {
        final INode next = iter.next();
        int _startLine = next.getStartLine();
        boolean _greaterThan = (_startLine > line);
        if (_greaterThan) {
          iter.prune();
        } else {
          if ((next instanceof ILeafNode)) {
            nodesInLine.add(next);
          }
        }
      }
    }
    boolean _isEmpty = nodesInLine.isEmpty();
    if (_isEmpty) {
      return CollectionLiterals.<EObject>emptyList();
    }
    for (final INode ln : nodesInLine) {
      {
        final EObject eo = NodeModelUtils.findActualSemanticObjectFor(ln);
        if (((eo != null) && (!result.contains(eo)))) {
          result.add(eo);
        }
      }
    }
    return result;
  }
  
  /**
   * Breaks lines into strings on comma-positions and remove outer quotes.
   * Beware: not quote-safe (Commas in matching quotes are <b>not</b> treated specifically.)
   * @param line to break up
   */
  public static List<String> separateOnCommaAndQuote(final CharSequence line) {
    return QuickFixTestHelper.separateOnComma(line, true);
  }
  
  /**
   * Breaks up a CharSequence into a List of Strings on comma-characters.
   * Beware: not quote-safe (Commas in matching quotes are <b>not</b> treated specifically.)
   * @param line to break up
   * @param removeQuote - if true, remove outer quotes on  all strings in list
   */
  public static List<String> separateOnComma(final CharSequence line, final boolean removeQuote) {
    final Function1<String, String> _function = (String it) -> {
      return it.trim();
    };
    List<String> result = ListExtensions.<String, String>map(((List<String>)Conversions.doWrapArray(line.toString().split(","))), _function);
    if (removeQuote) {
      final Function1<String, String> _function_1 = (String it) -> {
        String _xifexpression = null;
        if (((it.startsWith("\"") && it.endsWith("\"")) || (it.startsWith("\'") && it.endsWith("\'")))) {
          int _length = it.length();
          int _minus = (_length - 1);
          _xifexpression = it.substring(1, _minus);
        } else {
          _xifexpression = it;
        }
        return _xifexpression;
      };
      result = ListExtensions.<String, String>map(result, _function_1);
    }
    return IterableExtensions.<String>toList(result);
  }
  
  /**
   * Create a single line of comma-separated & single-quoted label strings.
   * @param list of resolutions
   */
  public static String asString(final List<IssueResolution> resolutions) {
    final Function1<IssueResolution, String> _function = (IssueResolution it) -> {
      return it.getLabel();
    };
    final Function2<String, String, String> _function_1 = (String p1, String p2) -> {
      return (((("\'" + p1) + "\', \'") + p2) + "\'");
    };
    return IterableExtensions.<String>reduce(ListExtensions.<IssueResolution, String>map(resolutions, _function), _function_1);
  }
  
  /**
   * Create a single line of comma-separated & single-quoted display strings.
   * @param s list of completion proposals
   */
  public static String asString2(final List<ICompletionProposal> s) {
    final Function1<ICompletionProposal, String> _function = (ICompletionProposal it) -> {
      String _displayString = it.getDisplayString();
      String _plus = ("\'" + _displayString);
      return (_plus + "\'");
    };
    final Function2<String, String, String> _function_1 = (String p1, String p2) -> {
      return ((p1 + ", ") + p2);
    };
    return IterableExtensions.<String>reduce(ListExtensions.<ICompletionProposal, String>map(s, _function), _function_1);
  }
  
  /**
   * Return a specific IssueResolution out of resolutions.
   * @param resolutions candidates
   * @param labelPart matcher
   * @return a resolution with a label containing labelPart as substring. Never null.
   * @throws AssertionFailedError - if more then one or none resolution contains labelPart in its label
   */
  public static IssueResolution selectSingleOrFail(final List<IssueResolution> resolutions, final String labelPart) {
    IssueResolution _xblockexpression = null;
    {
      final Function1<IssueResolution, Boolean> _function = (IssueResolution it) -> {
        return Boolean.valueOf(it.getLabel().contains(labelPart));
      };
      final List<IssueResolution> matched = IterableExtensions.<IssueResolution>toList(IterableExtensions.<IssueResolution>filter(resolutions, _function));
      int _size = matched.size();
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        int _size_1 = matched.size();
        String _plus = ((("The selected issue resolution \'" + labelPart) + "\' is matched by more then one (") + Integer.valueOf(_size_1));
        String _plus_1 = (_plus + 
          ") resolutions ");
        String _asString = QuickFixTestHelper.asString(matched);
        String _plus_2 = (_plus_1 + _asString);
        throw new AssertionFailedError(_plus_2);
      } else {
        int _size_2 = matched.size();
        boolean _lessThan = (_size_2 < 1);
        if (_lessThan) {
          throw new AssertionFailedError((("No issue-resolution found with name containing \'" + labelPart) + "\'"));
        }
      }
      _xblockexpression = IterableExtensions.<IssueResolution>head(matched);
    }
    return _xblockexpression;
  }
  
  /**
   * This simple algorithm can detect changes between the two multiline texts before & after assuming
   * a single line has been changed, inserted or deleted.
   * 
   * If more than one line is effected the result is probably useless.
   * 
   * Query this with {@ChangeInfo#isMoreThanOne()}
   * 
   * @param before first text
   * @param after second text
   * 
   * @return List of changes. If more than one line is effected the result is probably useless.
   */
  public static QuickFixTestHelper.ChangeInfo extractSingleChangedLine(final String before, final String after) {
    final String delim = "\n";
    final String[] bLines = before.split(delim);
    final String[] aLines = after.split(delim);
    final QuickFixTestHelper.ChangeInfo ci = new QuickFixTestHelper.ChangeInfo();
    int bo = 0;
    int ao = 0;
    {
      int bi = 0;
      int ai = 0;
      boolean _while = ((bi < bLines.length) && (ai < aLines.length));
      while (_while) {
        {
          final String b = bLines[bi];
          final String a = aLines[ai];
          boolean _equals = b.trim().equals(a.trim());
          boolean _not = (!_equals);
          if (_not) {
            if ((((ai + 1) < aLines.length) && b.trim().equals(aLines[(ai + 1)].trim()))) {
              ci.add(bi, bo, "", ao, aLines[ai]);
              ai++;
              int _ao = ao;
              int _length = aLines[ai].length();
              int _length_1 = delim.length();
              int _plus = (_length + _length_1);
              ao = (_ao + _plus);
            } else {
              if ((((bi + 1) < bLines.length) && bLines[(bi + 1)].trim().equals(a.trim()))) {
                ci.add(bi, bo, bLines[bi], ao, "");
                bi++;
                int _bo = bo;
                int _length_2 = aLines[bi].length();
                int _length_3 = delim.length();
                int _plus_1 = (_length_2 + _length_3);
                bo = (_bo + _plus_1);
              } else {
                ci.add(bi, bo, bLines[bi], ao, aLines[ai]);
              }
            }
          } else {
          }
          int _ao_1 = ao;
          int _length_4 = aLines[ai].length();
          int _length_5 = delim.length();
          int _plus_2 = (_length_4 + _length_5);
          ao = (_ao_1 + _plus_2);
          int _bo_1 = bo;
          int _length_6 = bLines[bi].length();
          int _length_7 = delim.length();
          int _plus_3 = (_length_6 + _length_7);
          bo = (_bo_1 + _plus_3);
        }
        bi++;
        ai++;
        _while = ((bi < bLines.length) && (ai < aLines.length));
      }
    }
    return ci;
  }
}
