/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.n4JS;

import com.google.common.collect.Iterables;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.n4js.n4JS.ArrayBindingPattern;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.BindingElement;
import org.eclipse.n4js.n4JS.BindingPattern;
import org.eclipse.n4js.n4JS.BindingProperty;
import org.eclipse.n4js.n4JS.ControlFlowElement;
import org.eclipse.n4js.n4JS.DestructureUtils;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ForStatement;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.LiteralOrComputedPropertyName;
import org.eclipse.n4js.n4JS.N4JSFactory;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.ObjectBindingPattern;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.PropertyAssignmentAnnotationList;
import org.eclipse.n4js.n4JS.PropertyGetterDeclaration;
import org.eclipse.n4js.n4JS.PropertyMethodDeclaration;
import org.eclipse.n4js.n4JS.PropertyNameOwner;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.PropertyNameValuePairSingleName;
import org.eclipse.n4js.n4JS.PropertySetterDeclaration;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.StringLiteral;
import org.eclipse.n4js.n4JS.VariableBinding;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TypableElement;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * Destructuring patterns can appear in very different forms within the AST and in different contexts.
 * This helper class is used to transform those heterogeneous representations into a single, uniform
 * structure, that can be traversed more easily.
 * 
 * All fields are optional, i.e. may be 'null'. At most one of 'varRef', 'varDecl' and 'nestedPattern'
 * may be non-null; if all three are 'null' the node is a padding node.
 * 
 * <h2>Overview of Destructuring Patterns in the AST</h2>
 * Different forms:
 * <ol>
 * <li>as a {@link BindingPattern} (may contain nested {@code BindingPattern}s).
 * <li>as an {@link ArrayLiteral} (may contain nested patterns in form of {@code ArrayLiteral}s or {@code ObjectLiteral}s).
 * <li>as an {@link ObjectLiteral} (may contain nested patterns in form of {@code ArrayLiteral}s or {@code ObjectLiteral}s).
 * </ol>
 * Different contexts:
 * <ol>
 * <li>within a {@link VariableStatement} (then contained in a {@link VariableBinding}, which is an alternative
 *     to a {@link VariableDeclaration}).
 * <li>within an {@link AssignmentExpression} (then it appears as the left-hand side expression)
 * <li>within a {@link ForStatement} (only in for..in and for..of, because if used in plain for it is a use case
 *     of a variable statement or assignment expression inside the for statement).
 * <li><b>NOT SUPPORTED YET</b>: in the context of lists of formal parameters or function argument lists
 * </ol>
 * The above 6 use cases have several special characteristics and constraints, most of which are unified in this class.
 * It might be possible to generate more unified patterns in the parser, but the above situation is more in line with
 * terminology in the ES6 specification.
 * <p>
 */
@Data
@SuppressWarnings("all")
public class DestructNode {
  private final EObject astElement;
  
  private final String propName;
  
  private final IdentifierRef varRef;
  
  private final VariableDeclaration varDecl;
  
  private final DestructNode[] nestedNodes;
  
  private final Expression defaultExpr;
  
  private final EObject assignedElem;
  
  private final boolean rest;
  
  /**
   * Tells if receiving node belongs to a positional destructuring pattern
   * (i.e. an array destructuring pattern).
   */
  public boolean isPositional() {
    return (this.propName == null);
  }
  
  /**
   * Tells if the given nodes belong to a positional destructuring pattern
   * (i.e. an array destructuring pattern).
   */
  public static boolean arePositional(final DestructNode[] nodes) {
    return ((nodes != null) && IterableExtensions.<DestructNode>exists(((Iterable<DestructNode>)Conversions.doWrapArray(nodes)), ((Function1<DestructNode, Boolean>) (DestructNode it) -> {
      return Boolean.valueOf(it.isPositional());
    })));
  }
  
  /**
   * Tells if this is a padding node.
   */
  public boolean isPadding() {
    return (((this.varRef == null) && (this.varDecl == null)) && (this.nestedNodes == null));
  }
  
  /**
   * If this node has a reference to a variable or a variable declaration,
   * returns the variable's name, <code>null</code> otherwise.
   */
  public String varName() {
    String _xifexpression = null;
    if ((this.varRef != null)) {
      IdentifiableElement _id = this.varRef.getId();
      String _name = null;
      if (_id!=null) {
        _name=_id.getName();
      }
      _xifexpression = _name;
    } else {
      String _xifexpression_1 = null;
      if ((this.varDecl != null)) {
        _xifexpression_1 = this.varDecl.getName();
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  /**
   * Returns the variable declaration contained in this node's astElement or <code>null</code>.
   */
  public VariableDeclaration getVariableDeclaration() {
    VariableDeclaration _switchResult = null;
    final EObject astElement = this.astElement;
    boolean _matched = false;
    if (astElement instanceof BindingElement) {
      _matched=true;
      _switchResult = ((BindingElement)this.astElement).getVarDecl();
    }
    if (!_matched) {
      if (astElement instanceof BindingProperty) {
        BindingElement _value = ((BindingProperty)this.astElement).getValue();
        boolean _tripleNotEquals = (_value != null);
        if (_tripleNotEquals) {
          _matched=true;
          _switchResult = ((BindingProperty)this.astElement).getValue().getVarDecl();
        }
      }
    }
    return _switchResult;
  }
  
  /**
   * Returns the AST node and EStructuralFeature to be used when showing an error message
   * on the receiving node's propName attribute. Intended for issue generation in validations.
   */
  public Pair<EObject, EStructuralFeature> getEObjectAndFeatureForPropName() {
    Pair<EObject, EStructuralFeature> _xifexpression = null;
    if ((this.propName != null)) {
      Pair<EObject, EStructuralFeature> _switchResult = null;
      final EObject astElement = this.astElement;
      boolean _matched = false;
      if (astElement instanceof PropertyNameValuePairSingleName) {
        _matched=true;
        EReference _propertyNameValuePairSingleName_IdentifierRef = N4JSPackage.eINSTANCE.getPropertyNameValuePairSingleName_IdentifierRef();
        _switchResult = Pair.<EObject, EStructuralFeature>of(this.astElement, _propertyNameValuePairSingleName_IdentifierRef);
      }
      if (!_matched) {
        if (astElement instanceof BindingProperty) {
          LiteralOrComputedPropertyName _declaredName = ((BindingProperty)this.astElement).getDeclaredName();
          boolean _tripleNotEquals = (_declaredName != null);
          if (_tripleNotEquals) {
            _matched=true;
            EReference _propertyNameOwner_DeclaredName = N4JSPackage.eINSTANCE.getPropertyNameOwner_DeclaredName();
            _switchResult = Pair.<EObject, EStructuralFeature>of(this.astElement, _propertyNameOwner_DeclaredName);
          }
        }
      }
      if (!_matched) {
        if (astElement instanceof BindingProperty) {
          BindingElement _value = ((BindingProperty)this.astElement).getValue();
          VariableDeclaration _varDecl = null;
          if (_value!=null) {
            _varDecl=_value.getVarDecl();
          }
          String _name = null;
          if (_varDecl!=null) {
            _name=_varDecl.getName();
          }
          boolean _tripleNotEquals = (_name != null);
          if (_tripleNotEquals) {
            _matched=true;
            VariableDeclaration _varDecl_1 = ((BindingProperty)this.astElement).getValue().getVarDecl();
            EAttribute _identifiableElement_Name = TypesPackage.eINSTANCE.getIdentifiableElement_Name();
            _switchResult = Pair.<EObject, EStructuralFeature>of(_varDecl_1, _identifiableElement_Name);
          }
        }
      }
      if (!_matched) {
        if (astElement instanceof PropertyNameOwner) {
          _matched=true;
          EReference _propertyNameOwner_DeclaredName = N4JSPackage.eINSTANCE.getPropertyNameOwner_DeclaredName();
          _switchResult = Pair.<EObject, EStructuralFeature>of(this.astElement, _propertyNameOwner_DeclaredName);
        }
      }
      if (!_matched) {
        _switchResult = Pair.<EObject, EStructuralFeature>of(this.astElement, null);
      }
      _xifexpression = _switchResult;
    } else {
      _xifexpression = Pair.<EObject, EStructuralFeature>of(this.astElement, null);
    }
    return _xifexpression;
  }
  
  /**
   * Returns the node with the given <code>astElement</code>.
   */
  public DestructNode findNodeForElement(final EObject astElement) {
    final Predicate<DestructNode> _function = (DestructNode it) -> {
      return (it.astElement == astElement);
    };
    return this.stream().filter(_function).findFirst().orElse(null);
  }
  
  /**
   * Returns stream of this node and all its descendants, i.e. directly and indirectly nested nodes.
   */
  public Stream<DestructNode> stream() {
    Stream<DestructNode> _xifexpression = null;
    if (((this.nestedNodes == null) || ((List<DestructNode>)Conversions.doWrapArray(this.nestedNodes)).isEmpty())) {
      _xifexpression = Stream.<DestructNode>of(this);
    } else {
      final Function<DestructNode, Stream<DestructNode>> _function = (DestructNode it) -> {
        return it.stream();
      };
      _xifexpression = Stream.<DestructNode>concat(Stream.<DestructNode>of(this), Stream.<DestructNode>of(this.nestedNodes).<DestructNode>flatMap(_function));
    }
    return _xifexpression;
  }
  
  public static DestructNode unify(final EObject eobj) {
    DestructNode _switchResult = null;
    boolean _matched = false;
    if (eobj instanceof VariableBinding) {
      _matched=true;
      _switchResult = DestructNode.unify(((VariableBinding)eobj));
    }
    if (!_matched) {
      if (eobj instanceof AssignmentExpression) {
        _matched=true;
        _switchResult = DestructNode.unify(((AssignmentExpression)eobj));
      }
    }
    if (!_matched) {
      if (eobj instanceof ForStatement) {
        _matched=true;
        _switchResult = DestructNode.unify(((ForStatement)eobj));
      }
    }
    if (!_matched) {
      _switchResult = null;
    }
    return _switchResult;
  }
  
  /**
   * Returns a unified copy of the given destructuring pattern or <code>null</code> if it is invalid.
   * This is helpful because these patterns can appear in very different forms and locations within the AST.
   */
  public static DestructNode unify(final EObject lhs, final Expression rhs) {
    DestructNode[] _entries = DestructNode.toEntries(lhs, rhs);
    return new DestructNode(lhs, 
      null, 
      null, 
      null, _entries, rhs, rhs, 
      false);
  }
  
  /**
   * Returns a unified copy of the given destructuring pattern or <code>null</code> if it is invalid.
   * This is helpful because these patterns can appear in very different forms and locations within the AST.
   */
  public static DestructNode unify(final VariableBinding binding) {
    DestructNode _xifexpression = null;
    if ((((binding != null) && (binding.getPattern() != null)) && ((binding.getExpression() != null) || (binding.eContainer() instanceof ForStatement)))) {
      _xifexpression = DestructNode.unify(binding.getPattern(), binding.getExpression());
    }
    return _xifexpression;
  }
  
  /**
   * Returns a unified copy of the given destructuring pattern or <code>null</code> if it is invalid.
   * This is helpful because these patterns can appear in very different forms and locations within the AST.
   */
  public static DestructNode unify(final AssignmentExpression expr) {
    DestructNode _xifexpression = null;
    if (((((expr != null) && (expr.getLhs() != null)) && (expr.getRhs() != null)) && DestructureUtils.isTopOfDestructuringAssignment(expr))) {
      _xifexpression = DestructNode.unify(expr.getLhs(), expr.getRhs());
    }
    return _xifexpression;
  }
  
  /**
   * Returns a unified copy of the given destructuring pattern or <code>null</code> if it is invalid.
   * This is helpful because these patterns can appear in very different forms and locations within the AST.
   */
  public static DestructNode unify(final ForStatement stmnt) {
    DestructNode _xifexpression = null;
    if (((stmnt != null) && DestructureUtils.isTopOfDestructuringForStatement(stmnt))) {
      DestructNode _xblockexpression = null;
      {
        Expression _xifexpression_1 = null;
        boolean _isForOf = stmnt.isForOf();
        if (_isForOf) {
          _xifexpression_1 = stmnt.getExpression();
        } else {
          StringLiteral _xifexpression_2 = null;
          boolean _isForIn = stmnt.isForIn();
          if (_isForIn) {
            _xifexpression_2 = N4JSFactory.eINSTANCE.createStringLiteral();
          } else {
            throw new IllegalStateException();
          }
          _xifexpression_1 = _xifexpression_2;
        }
        final Expression valueToBeDestructured = _xifexpression_1;
        DestructNode _xifexpression_3 = null;
        boolean _containsDestructuringPattern = DestructureUtils.containsDestructuringPattern(stmnt);
        if (_containsDestructuringPattern) {
          DestructNode _xblockexpression_1 = null;
          {
            final VariableBinding binding = IterableExtensions.<VariableBinding>head(Iterables.<VariableBinding>filter(stmnt.getVarDeclsOrBindings(), VariableBinding.class));
            DestructNode[] _entries = DestructNode.toEntries(binding.getPattern(), stmnt.getExpression());
            _xblockexpression_1 = new DestructNode(binding, 
              null, 
              null, 
              null, _entries, valueToBeDestructured, valueToBeDestructured, 
              false);
          }
          _xifexpression_3 = _xblockexpression_1;
        } else {
          DestructNode _xifexpression_4 = null;
          boolean _isObjectOrArrayLiteral = DestructureUtils.isObjectOrArrayLiteral(stmnt.getInitExpr());
          if (_isObjectOrArrayLiteral) {
            Expression _initExpr = stmnt.getInitExpr();
            DestructNode[] _entries = DestructNode.toEntries(stmnt.getInitExpr(), null);
            _xifexpression_4 = new DestructNode(_initExpr, 
              null, 
              null, 
              null, _entries, valueToBeDestructured, valueToBeDestructured, 
              false);
          }
          _xifexpression_3 = _xifexpression_4;
        }
        _xblockexpression = _xifexpression_3;
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  private static DestructNode[] toEntries(final EObject pattern, final EObject rhs) {
    Iterator<? extends EObject> _switchResult = null;
    boolean _matched = false;
    if (pattern instanceof ArrayLiteral) {
      _matched=true;
      _switchResult = ((ArrayLiteral)pattern).getElements().iterator();
    }
    if (!_matched) {
      if (pattern instanceof ObjectLiteral) {
        _matched=true;
        _switchResult = ((ObjectLiteral)pattern).getPropertyAssignments().iterator();
      }
    }
    if (!_matched) {
      if (pattern instanceof ArrayBindingPattern) {
        _matched=true;
        _switchResult = ((ArrayBindingPattern)pattern).getElements().iterator();
      }
    }
    if (!_matched) {
      if (pattern instanceof ObjectBindingPattern) {
        _matched=true;
        _switchResult = ((ObjectBindingPattern)pattern).getProperties().iterator();
      }
    }
    final Iterator<? extends EObject> patElemIter = _switchResult;
    Iterator<? extends TypableElement> _switchResult_1 = null;
    boolean _matched_1 = false;
    if (rhs instanceof ArrayLiteral) {
      _matched_1=true;
      _switchResult_1 = ((ArrayLiteral)rhs).getElements().iterator();
    }
    if (!_matched_1) {
      if (rhs instanceof ObjectLiteral) {
        _matched_1=true;
        _switchResult_1 = ((ObjectLiteral)rhs).getPropertyAssignments().iterator();
      }
    }
    Iterator<? extends EObject> rhsElemIter = _switchResult_1;
    final BasicEList<DestructNode> nestedDNs = new BasicEList<DestructNode>();
    while (patElemIter.hasNext()) {
      {
        final EObject patElem = patElemIter.next();
        EObject _xifexpression = null;
        if ((rhsElemIter == null)) {
          _xifexpression = rhs;
        } else {
          EObject _xifexpression_1 = null;
          boolean _hasNext = rhsElemIter.hasNext();
          if (_hasNext) {
            _xifexpression_1 = rhsElemIter.next();
          } else {
            _xifexpression_1 = null;
          }
          _xifexpression = _xifexpression_1;
        }
        final EObject litElem = _xifexpression;
        DestructNode _switchResult_2 = null;
        boolean _matched_2 = false;
        if (patElem instanceof ArrayElement) {
          _matched_2=true;
          _switchResult_2 = DestructNode.toEntry(((ArrayElement)patElem), litElem);
        }
        if (!_matched_2) {
          if (patElem instanceof PropertyNameValuePair) {
            _matched_2=true;
            _switchResult_2 = DestructNode.toEntry(((PropertyNameValuePair)patElem), litElem);
          }
        }
        if (!_matched_2) {
          if (patElem instanceof BindingElement) {
            _matched_2=true;
            _switchResult_2 = DestructNode.toEntry(((BindingElement)patElem), litElem);
          }
        }
        if (!_matched_2) {
          if (patElem instanceof BindingProperty) {
            _matched_2=true;
            _switchResult_2 = DestructNode.toEntry(((BindingProperty)patElem), litElem);
          }
        }
        final DestructNode nestedNode = _switchResult_2;
        if ((nestedNode != null)) {
          nestedDNs.add(nestedNode);
        }
      }
    }
    return ((DestructNode[])Conversions.unwrapArray(nestedDNs, DestructNode.class));
  }
  
  private static DestructNode toEntry(final ArrayElement elem, final EObject rhs) {
    DestructNode _xblockexpression = null;
    {
      EObject _xifexpression = null;
      if ((rhs instanceof ArrayElement)) {
        _xifexpression = ((ArrayElement)rhs).getExpression();
      } else {
        _xifexpression = rhs;
      }
      final EObject rhsExpr = _xifexpression;
      final Expression expr = elem.getExpression();
      DestructNode _xifexpression_1 = null;
      if ((expr instanceof AssignmentExpression)) {
        _xifexpression_1 = DestructNode.toEntry(elem, null, ((AssignmentExpression)expr).getLhs(), ((AssignmentExpression)expr).getRhs(), elem.isSpread(), rhsExpr);
      } else {
        _xifexpression_1 = DestructNode.toEntry(elem, null, expr, null, elem.isSpread(), rhsExpr);
      }
      _xblockexpression = _xifexpression_1;
    }
    return _xblockexpression;
  }
  
  private static DestructNode toEntry(final PropertyNameValuePair pa, final EObject rhs) {
    DestructNode _xblockexpression = null;
    {
      EObject _xifexpression = null;
      if ((rhs instanceof PropertyNameValuePair)) {
        _xifexpression = ((PropertyNameValuePair)rhs).getExpression();
      } else {
        _xifexpression = rhs;
      }
      final EObject rhsExpr = _xifexpression;
      DestructNode _xifexpression_1 = null;
      if ((pa instanceof PropertyNameValuePairSingleName)) {
        _xifexpression_1 = DestructNode.toEntry(pa, ((PropertyNameValuePairSingleName)pa).getName(), ((PropertyNameValuePairSingleName)pa).getIdentifierRef(), ((PropertyNameValuePairSingleName)pa).getExpression(), false, rhsExpr);
      } else {
        DestructNode _xblockexpression_1 = null;
        {
          final Expression expr = pa.getExpression();
          DestructNode _xifexpression_2 = null;
          if ((expr instanceof AssignmentExpression)) {
            _xifexpression_2 = DestructNode.toEntry(pa, pa.getName(), ((AssignmentExpression)expr).getLhs(), ((AssignmentExpression)expr).getRhs(), false, rhsExpr);
          } else {
            _xifexpression_2 = DestructNode.toEntry(pa, pa.getName(), expr, null, false, rhsExpr);
          }
          _xblockexpression_1 = _xifexpression_2;
        }
        _xifexpression_1 = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression_1;
    }
    return _xblockexpression;
  }
  
  private static DestructNode toEntry(final BindingElement elem, final EObject rhs) {
    DestructNode _xblockexpression = null;
    {
      EObject _xifexpression = null;
      if ((rhs instanceof ArrayElement)) {
        _xifexpression = ((ArrayElement)rhs).getExpression();
      } else {
        _xifexpression = rhs;
      }
      final EObject expr = _xifexpression;
      DestructNode _xifexpression_1 = null;
      VariableDeclaration _varDecl = elem.getVarDecl();
      boolean _tripleNotEquals = (_varDecl != null);
      if (_tripleNotEquals) {
        _xifexpression_1 = DestructNode.toEntry(elem, null, elem.getVarDecl(), elem.getVarDecl().getExpression(), elem.isRest(), expr);
      } else {
        DestructNode _xifexpression_2 = null;
        BindingPattern _nestedPattern = elem.getNestedPattern();
        boolean _tripleNotEquals_1 = (_nestedPattern != null);
        if (_tripleNotEquals_1) {
          _xifexpression_2 = DestructNode.toEntry(elem, null, elem.getNestedPattern(), elem.getExpression(), elem.isRest(), expr);
        } else {
          _xifexpression_2 = DestructNode.toEntry(elem, null, null, null, false, expr);
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xblockexpression = _xifexpression_1;
    }
    return _xblockexpression;
  }
  
  private static DestructNode toEntry(final BindingProperty prop, final EObject rhs) {
    DestructNode _xifexpression = null;
    BindingElement _value = prop.getValue();
    VariableDeclaration _varDecl = null;
    if (_value!=null) {
      _varDecl=_value.getVarDecl();
    }
    boolean _tripleNotEquals = (_varDecl != null);
    if (_tripleNotEquals) {
      DestructNode _xblockexpression = null;
      {
        final EObject expr = DestructNode.getPropertyAssignmentExpression(rhs);
        _xblockexpression = DestructNode.toEntry(prop, prop.getName(), prop.getValue().getVarDecl(), prop.getValue().getVarDecl().getExpression(), false, expr);
      }
      _xifexpression = _xblockexpression;
    } else {
      DestructNode _xifexpression_1 = null;
      BindingElement _value_1 = prop.getValue();
      BindingPattern _nestedPattern = null;
      if (_value_1!=null) {
        _nestedPattern=_value_1.getNestedPattern();
      }
      boolean _tripleNotEquals_1 = (_nestedPattern != null);
      if (_tripleNotEquals_1) {
        DestructNode _xblockexpression_1 = null;
        {
          final EObject expr = DestructNode.getPropertyAssignmentExpression(rhs);
          _xblockexpression_1 = DestructNode.toEntry(prop, prop.getName(), prop.getValue().getNestedPattern(), prop.getValue().getExpression(), false, expr);
        }
        _xifexpression_1 = _xblockexpression_1;
      } else {
        _xifexpression_1 = DestructNode.toEntry(prop, null, null, null, false, rhs);
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  /**
   * @param bindingTarget
   *              an IdentifierRef/VariableDeclaration or a nested pattern (which may be
   *              a BindingPattern, ArrayLiteral, or ObjectLiteral)
   */
  private static DestructNode toEntry(final EObject astElement, final String propName, final EObject bindingTarget, final Expression defaultExpr, final boolean rest, final EObject rhs) {
    DestructNode _xifexpression = null;
    if ((bindingTarget == null)) {
      _xifexpression = new DestructNode(astElement, propName, null, null, null, defaultExpr, null, rest);
    } else {
      DestructNode _xifexpression_1 = null;
      if ((bindingTarget instanceof IdentifierRef)) {
        _xifexpression_1 = new DestructNode(astElement, propName, ((IdentifierRef)bindingTarget), null, null, defaultExpr, rhs, rest);
      } else {
        DestructNode _xifexpression_2 = null;
        if ((bindingTarget instanceof VariableDeclaration)) {
          _xifexpression_2 = new DestructNode(astElement, propName, null, ((VariableDeclaration)bindingTarget), null, defaultExpr, rhs, rest);
        } else {
          DestructNode _xifexpression_3 = null;
          if ((((bindingTarget instanceof ArrayLiteral) || (bindingTarget instanceof ObjectLiteral)) || 
            (bindingTarget instanceof BindingPattern))) {
            DestructNode[] _entries = DestructNode.toEntries(bindingTarget, rhs);
            _xifexpression_3 = new DestructNode(astElement, propName, null, null, _entries, defaultExpr, rhs, rest);
          } else {
            _xifexpression_3 = new DestructNode(astElement, propName, null, null, null, defaultExpr, null, rest);
          }
          _xifexpression_2 = _xifexpression_3;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  /**
   * @return the expression or function of the given PropertyAssignment
   */
  private static EObject getPropertyAssignmentExpression(final EObject rhs) {
    boolean _matched = false;
    if (rhs instanceof PropertyGetterDeclaration) {
      _matched=true;
      return ((PropertyGetterDeclaration)rhs).getDefinedFunctionOrAccessor();
    }
    if (!_matched) {
      if (rhs instanceof PropertySetterDeclaration) {
        _matched=true;
        return ((PropertySetterDeclaration)rhs).getDefinedFunctionOrAccessor();
      }
    }
    if (!_matched) {
      if (rhs instanceof PropertyMethodDeclaration) {
        _matched=true;
        return ((PropertyMethodDeclaration)rhs).getDefinedFunctionOrAccessor();
      }
    }
    if (!_matched) {
      if (rhs instanceof PropertyNameValuePair) {
        _matched=true;
        return ((PropertyNameValuePair)rhs).getExpression();
      }
    }
    if (!_matched) {
      if (rhs instanceof PropertyAssignmentAnnotationList) {
        _matched=true;
        return null;
      }
    }
    return rhs;
  }
  
  /**
   * @return all {@link IdentifierRef} of variables that are written in the given assignment
   */
  public List<Expression> getAllDeclaredIdRefs() {
    final List<Expression> idRefs = new LinkedList<Expression>();
    final Iterator<DestructNode> allNestedNodes = this.stream().iterator();
    while (allNestedNodes.hasNext()) {
      {
        final EObject eobj = allNestedNodes.next().getAstElement();
        if ((eobj instanceof ArrayElement)) {
          final Expression expr = ((ArrayElement)eobj).getExpression();
          if ((expr instanceof AssignmentExpression)) {
            idRefs.add(((AssignmentExpression)expr).getLhs());
          } else {
            idRefs.add(expr);
          }
        } else {
          if ((eobj instanceof PropertyNameValuePairSingleName)) {
            idRefs.add(((PropertyNameValuePairSingleName)eobj).getIdentifierRef());
          } else {
            if ((eobj instanceof PropertyNameValuePair)) {
              final Expression expr_1 = ((PropertyNameValuePair)eobj).getExpression();
              if ((expr_1 instanceof AssignmentExpression)) {
                idRefs.add(((AssignmentExpression)expr_1).getLhs());
              } else {
                idRefs.add(expr_1);
              }
            }
          }
        }
      }
    }
    return idRefs;
  }
  
  /**
   * @return a pair where its key is the assigned EObject and its value is the default EObject to the given lhs AST element
   */
  public static Pair<EObject, EObject> getValueFromDestructuring(final EObject nodeElem) {
    EObject node = nodeElem;
    EObject topNode = null;
    EObject dNodeElem = null;
    boolean breakSearch = false;
    while ((!breakSearch)) {
      {
        EObject parent = node.eContainer();
        dNodeElem = DestructNode.getDNodeElem(dNodeElem, parent, node);
        topNode = DestructNode.getTopElem(topNode, parent);
        breakSearch = (parent instanceof Statement);
        node = parent;
      }
    }
    DestructNode _xifexpression = null;
    if ((topNode instanceof AssignmentExpression)) {
      _xifexpression = DestructNode.unify(((AssignmentExpression)topNode));
    } else {
      DestructNode _xifexpression_1 = null;
      if ((topNode instanceof VariableBinding)) {
        _xifexpression_1 = DestructNode.unify(((VariableBinding)topNode));
      } else {
        DestructNode _xifexpression_2 = null;
        if ((topNode instanceof ForStatement)) {
          _xifexpression_2 = DestructNode.unify(((ForStatement)topNode));
        } else {
          _xifexpression_2 = null;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    DestructNode dNode = _xifexpression;
    if ((dNode != null)) {
      dNode = dNode.findNodeForElement(dNodeElem);
      if ((dNode != null)) {
        EObject assgnValue = dNode.getAssignedElem();
        EObject defaultValue = dNode.getDefaultExpr();
        return Pair.<EObject, EObject>of(assgnValue, defaultValue);
      }
    }
    return null;
  }
  
  private static EObject getDNodeElem(final EObject dNodeElem, final EObject parent, final EObject node) {
    if ((dNodeElem != null)) {
      return dNodeElem;
    }
    if (((node instanceof BindingElement) && (parent instanceof BindingProperty))) {
      return parent;
    }
    if ((((node instanceof BindingElement) || (node instanceof ArrayElement)) || (node instanceof PropertyAssignment))) {
      return node;
    }
    return null;
  }
  
  private static EObject getTopElem(final EObject oldTopNode, final EObject parent) {
    EObject _xblockexpression = null;
    {
      ControlFlowElement _switchResult = null;
      boolean _matched = false;
      if (parent instanceof ForStatement) {
        _matched=true;
        _switchResult = ((ControlFlowElement)parent);
      }
      if (!_matched) {
        if (parent instanceof AssignmentExpression) {
          _matched=true;
          _switchResult = ((ControlFlowElement)parent);
        }
      }
      if (!_matched) {
        if (parent instanceof VariableBinding) {
          _matched=true;
          _switchResult = ((ControlFlowElement)parent);
        }
      }
      if (!_matched) {
        _switchResult = null;
      }
      final EObject newTopNode = _switchResult;
      EObject _xifexpression = null;
      if ((newTopNode != null)) {
        return newTopNode;
      } else {
        _xifexpression = oldTopNode;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public static List<Expression> getAllDeclaredIdRefs(final EObject eobj) {
    DestructNode _switchResult = null;
    boolean _matched = false;
    if (eobj instanceof ForStatement) {
      _matched=true;
      _switchResult = DestructNode.unify(((ForStatement)eobj));
    }
    if (!_matched) {
      if (eobj instanceof VariableBinding) {
        _matched=true;
        _switchResult = DestructNode.unify(((VariableBinding)eobj));
      }
    }
    if (!_matched) {
      if (eobj instanceof AssignmentExpression) {
        _matched=true;
        _switchResult = DestructNode.unify(((AssignmentExpression)eobj));
      }
    }
    if (!_matched) {
      _switchResult = null;
    }
    final DestructNode dnode = _switchResult;
    if ((dnode == null)) {
      return Collections.<Expression>unmodifiableList(CollectionLiterals.<Expression>newArrayList());
    }
    return dnode.getAllDeclaredIdRefs();
  }
  
  public DestructNode(final EObject astElement, final String propName, final IdentifierRef varRef, final VariableDeclaration varDecl, final DestructNode[] nestedNodes, final Expression defaultExpr, final EObject assignedElem, final boolean rest) {
    super();
    this.astElement = astElement;
    this.propName = propName;
    this.varRef = varRef;
    this.varDecl = varDecl;
    this.nestedNodes = nestedNodes;
    this.defaultExpr = defaultExpr;
    this.assignedElem = assignedElem;
    this.rest = rest;
  }
  
  @Override
  @Pure
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.astElement== null) ? 0 : this.astElement.hashCode());
    result = prime * result + ((this.propName== null) ? 0 : this.propName.hashCode());
    result = prime * result + ((this.varRef== null) ? 0 : this.varRef.hashCode());
    result = prime * result + ((this.varDecl== null) ? 0 : this.varDecl.hashCode());
    result = prime * result + ((this.nestedNodes== null) ? 0 : Arrays.deepHashCode(this.nestedNodes));
    result = prime * result + ((this.defaultExpr== null) ? 0 : this.defaultExpr.hashCode());
    result = prime * result + ((this.assignedElem== null) ? 0 : this.assignedElem.hashCode());
    return prime * result + (this.rest ? 1231 : 1237);
  }
  
  @Override
  @Pure
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    DestructNode other = (DestructNode) obj;
    if (this.astElement == null) {
      if (other.astElement != null)
        return false;
    } else if (!this.astElement.equals(other.astElement))
      return false;
    if (this.propName == null) {
      if (other.propName != null)
        return false;
    } else if (!this.propName.equals(other.propName))
      return false;
    if (this.varRef == null) {
      if (other.varRef != null)
        return false;
    } else if (!this.varRef.equals(other.varRef))
      return false;
    if (this.varDecl == null) {
      if (other.varDecl != null)
        return false;
    } else if (!this.varDecl.equals(other.varDecl))
      return false;
    if (this.nestedNodes == null) {
      if (other.nestedNodes != null)
        return false;
    } else if (!Arrays.deepEquals(this.nestedNodes, other.nestedNodes))
      return false;
    if (this.defaultExpr == null) {
      if (other.defaultExpr != null)
        return false;
    } else if (!this.defaultExpr.equals(other.defaultExpr))
      return false;
    if (this.assignedElem == null) {
      if (other.assignedElem != null)
        return false;
    } else if (!this.assignedElem.equals(other.assignedElem))
      return false;
    if (other.rest != this.rest)
      return false;
    return true;
  }
  
  @Override
  @Pure
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("astElement", this.astElement);
    b.add("propName", this.propName);
    b.add("varRef", this.varRef);
    b.add("varDecl", this.varDecl);
    b.add("nestedNodes", this.nestedNodes);
    b.add("defaultExpr", this.defaultExpr);
    b.add("assignedElem", this.assignedElem);
    b.add("rest", this.rest);
    return b.toString();
  }
  
  @Pure
  public EObject getAstElement() {
    return this.astElement;
  }
  
  @Pure
  public String getPropName() {
    return this.propName;
  }
  
  @Pure
  public IdentifierRef getVarRef() {
    return this.varRef;
  }
  
  @Pure
  public VariableDeclaration getVarDecl() {
    return this.varDecl;
  }
  
  @Pure
  public DestructNode[] getNestedNodes() {
    return this.nestedNodes;
  }
  
  @Pure
  public Expression getDefaultExpr() {
    return this.defaultExpr;
  }
  
  @Pure
  public EObject getAssignedElem() {
    return this.assignedElem;
  }
  
  @Pure
  public boolean isRest() {
    return this.rest;
  }
}
