/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.n4idl.versioning;

import com.google.inject.Inject;
import java.math.BigDecimal;
import java.util.List;
import java.util.ListIterator;
import org.eclipse.n4js.n4JS.VersionedElement;
import org.eclipse.n4js.n4idl.versioning.VersionHelper;
import org.eclipse.n4js.ts.typeRefs.ComposedTypeRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRefStructural;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsPackage;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.typeRefs.VersionedFunctionTypeRef;
import org.eclipse.n4js.ts.typeRefs.VersionedParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.VersionedParameterizedTypeRefStructural;
import org.eclipse.n4js.ts.typeRefs.VersionedReference;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.utils.TypeUtils;

/**
 * Resolver for the actual version of a type referenced by a given type reference.
 * 
 * Generally, when resolving an actual version of a versioned type, this implementation will delegate to
 * {@link VersionHelper#findClassifierWithVersion(TClassifier, int)}.
 * 
 * This resolver does not resolve any non-versionable/non-versioned elements and
 * thus has no effect, when applied to any non-N4IDL-specific AST elements.
 */
@SuppressWarnings("all")
public class N4IDLVersionResolver {
  @Inject
  private VersionHelper versionHelper;
  
  /**
   * Returns a type reference referencing the requested version of the type referenced by the given type reference.
   * 
   * @param typeRef
   *            the type reference to resolve
   * @param versionedReference
   *            the versioned reference to obtain the context version from
   * @return a reference to the resolved version of the type
   */
  public <T extends TypeArgument, S extends Object> T resolveVersion(final T typeRef, final S versionedReference) {
    if ((versionedReference instanceof VersionedReference)) {
      return this.<T>resolveVersion(typeRef, ((VersionedReference) versionedReference));
    } else {
      return typeRef;
    }
  }
  
  private <T extends TypeArgument> T resolveVersion(final T typeRef, final VersionedReference versionedReference) {
    boolean _hasRequestedVersion = versionedReference.hasRequestedVersion();
    if (_hasRequestedVersion) {
      return this.<T>resolveVersion(typeRef, versionedReference.getRequestedVersionOrZero());
    } else {
      return typeRef;
    }
  }
  
  /**
   * Returns a type reference referencing the requested version of the type referenced by the given type reference.
   * May return the given type argument or a different instance that is assignable to <code>T</code>. Type arguments
   * are also allowed to allow easier use of this method.
   * 
   * @param typeRef
   *            the type reference to resolve
   * @param contextVersion
   *            the context version to use when determining the actual version of the type referenced by the actual
   *            parameter
   * @return a reference to the resolved version of the type
   */
  public <T extends TypeArgument> T resolveVersion(final T typeArg, final int contextVersion) {
    T _xblockexpression = null;
    {
      if ((contextVersion == 0)) {
        return typeArg;
      }
      T _switchResult = null;
      boolean _matched = false;
      if (typeArg instanceof VersionedReference) {
        boolean _hasRequestedVersion = ((VersionedReference)typeArg).hasRequestedVersion();
        if (_hasRequestedVersion) {
          _matched=true;
          _switchResult = typeArg;
        }
      }
      if (!_matched) {
        if (typeArg instanceof VersionedElement) {
          boolean _hasDeclaredVersion = ((VersionedElement)typeArg).hasDeclaredVersion();
          if (_hasDeclaredVersion) {
            _matched=true;
            _switchResult = typeArg;
          }
        }
      }
      if (!_matched) {
        if (typeArg instanceof ComposedTypeRef) {
          _matched=true;
          ComposedTypeRef _resolveVersionOfComposedTypeRef = this.resolveVersionOfComposedTypeRef(((ComposedTypeRef)typeArg), contextVersion);
          _switchResult = ((T) _resolveVersionOfComposedTypeRef);
        }
      }
      if (!_matched) {
        if (typeArg instanceof TypeTypeRef) {
          _matched=true;
          TypeTypeRef _resolveVersionOfTypeTypeRef = this.resolveVersionOfTypeTypeRef(((TypeTypeRef)typeArg), contextVersion);
          _switchResult = ((T) _resolveVersionOfTypeTypeRef);
        }
      }
      if (!_matched) {
        if (typeArg instanceof FunctionTypeRef) {
          _matched=true;
          FunctionTypeRef _resolveVersionOfFunctionTypeRef = this.resolveVersionOfFunctionTypeRef(((FunctionTypeRef)typeArg), contextVersion);
          _switchResult = ((T) _resolveVersionOfFunctionTypeRef);
        }
      }
      if (!_matched) {
        if (typeArg instanceof FunctionTypeExpression) {
          _matched=true;
          FunctionTypeExpression _resolveVersionOfFunctionTypeExpression = this.resolveVersionOfFunctionTypeExpression(((FunctionTypeExpression)typeArg), contextVersion);
          _switchResult = ((T) _resolveVersionOfFunctionTypeExpression);
        }
      }
      if (!_matched) {
        if (typeArg instanceof ParameterizedTypeRef) {
          _matched=true;
          ParameterizedTypeRef _resolveVersionOfParameterizedTypeRef = this.resolveVersionOfParameterizedTypeRef(((ParameterizedTypeRef)typeArg), contextVersion);
          _switchResult = ((T) _resolveVersionOfParameterizedTypeRef);
        }
      }
      if (!_matched) {
        _switchResult = typeArg;
      }
      _xblockexpression = _switchResult;
    }
    return _xblockexpression;
  }
  
  private ComposedTypeRef resolveVersionOfComposedTypeRef(final ComposedTypeRef typeRef, final int contextVersion) {
    final ComposedTypeRef composedTypeRefWithVersion = TypeUtils.<ComposedTypeRef>copy(typeRef);
    this.<TypeArgument>resolveTypeArguments(composedTypeRefWithVersion.getTypeArgs(), contextVersion);
    return composedTypeRefWithVersion;
  }
  
  private TypeTypeRef resolveVersionOfTypeTypeRef(final TypeTypeRef typeRef, final int contextVersion) {
    final TypeTypeRef typeTypeRefWithVersion = TypeUtils.<TypeTypeRef>copy(typeRef);
    typeTypeRefWithVersion.setTypeArg(this.<TypeArgument>resolveVersion(typeTypeRefWithVersion.getTypeArg(), contextVersion));
    return typeTypeRefWithVersion;
  }
  
  private FunctionTypeRef resolveVersionOfFunctionTypeRef(final FunctionTypeRef typeRef, final int contextVersion) {
    Type _declaredType = typeRef.getDeclaredType();
    final TFunction function = ((TFunction) _declaredType);
    if ((function instanceof TMethod)) {
      final ContainerType<?> container = ((TMethod)function).getContainingType();
      if ((container instanceof TClassifier)) {
        VersionedParameterizedTypeRef _copyParameterizedTypeRef = this.copyParameterizedTypeRef(typeRef, contextVersion);
        final VersionedFunctionTypeRef functionTypeRefWithVersion = ((VersionedFunctionTypeRef) _copyParameterizedTypeRef);
        functionTypeRefWithVersion.setDeclaredType(this.versionHelper.<TMethod>findMemberWithVersion(((TMethod)function), contextVersion));
        return functionTypeRefWithVersion;
      }
    }
    return null;
  }
  
  private FunctionTypeExpression resolveVersionOfFunctionTypeExpression(final FunctionTypeExpression typeRef, final int contextVersion) {
    final FunctionTypeExpression typeRefWithVersion = TypeUtils.<FunctionTypeExpression>copy(typeRef);
    typeRefWithVersion.setReturnTypeRef(this.<TypeRef>resolveVersion(typeRefWithVersion.getReturnTypeRef(), contextVersion));
    typeRefWithVersion.setDeclaredThisType(this.<TypeRef>resolveVersion(typeRefWithVersion.getDeclaredThisType(), contextVersion));
    this.resolveFormalParameters(typeRefWithVersion.getFpars(), contextVersion);
    this.<TypeRef>resolveTypeArguments(typeRefWithVersion.getUnboundTypeVarsUpperBounds(), contextVersion);
    return typeRefWithVersion;
  }
  
  private ParameterizedTypeRef resolveVersionOfParameterizedTypeRef(final ParameterizedTypeRef typeRef, final int contextVersion) {
    final VersionedParameterizedTypeRef propTypeRefWithVersion = this.copyParameterizedTypeRef(typeRef, contextVersion);
    final Type declaredType = typeRef.getDeclaredType();
    if ((declaredType instanceof TClassifier)) {
      propTypeRefWithVersion.setDeclaredType(this.versionHelper.<TClassifier>findTypeWithVersion(((TClassifier)declaredType), contextVersion));
    }
    this.<TypeArgument>resolveTypeArguments(propTypeRefWithVersion.getTypeArgs(), contextVersion);
    return propTypeRefWithVersion;
  }
  
  private VersionedParameterizedTypeRef copyParameterizedTypeRef(final ParameterizedTypeRef typeRef, final int requestedVersion) {
    VersionedParameterizedTypeRef _switchResult = null;
    boolean _matched = false;
    if (typeRef instanceof FunctionTypeRef) {
      _matched=true;
      FunctionTypeRef _copy = TypeUtils.<FunctionTypeRef>copy(((FunctionTypeRef)typeRef), TypeRefsPackage.eINSTANCE.getVersionedParameterizedTypeRef());
      _switchResult = ((VersionedFunctionTypeRef) _copy);
    }
    if (!_matched) {
      if (typeRef instanceof ParameterizedTypeRefStructural) {
        _matched=true;
        ParameterizedTypeRefStructural _copy = TypeUtils.<ParameterizedTypeRefStructural>copy(((ParameterizedTypeRefStructural)typeRef), TypeRefsPackage.eINSTANCE.getVersionedParameterizedTypeRefStructural());
        _switchResult = ((VersionedParameterizedTypeRefStructural) _copy);
      }
    }
    if (!_matched) {
      ParameterizedTypeRef _copy = TypeUtils.<ParameterizedTypeRef>copy(typeRef, TypeRefsPackage.eINSTANCE.getVersionedParameterizedTypeRef());
      _switchResult = ((VersionedParameterizedTypeRef) _copy);
    }
    final VersionedParameterizedTypeRef result = _switchResult;
    BigDecimal _bigDecimal = new BigDecimal(requestedVersion);
    result.setRequestedVersion(_bigDecimal);
    return result;
  }
  
  private <T extends TypeArgument> void resolveTypeArguments(final List<T> typeArgs, final int contextVersion) {
    final ListIterator<T> i = typeArgs.listIterator();
    while (i.hasNext()) {
      {
        final TypeArgument current = i.next();
        if ((current != null)) {
          TypeArgument _resolveVersion = this.<TypeArgument>resolveVersion(current, contextVersion);
          i.set(((T) _resolveVersion));
        }
      }
    }
  }
  
  private void resolveFormalParameters(final List<TFormalParameter> parameters, final int contextVersion) {
    final ListIterator<TFormalParameter> i = parameters.listIterator();
    while (i.hasNext()) {
      {
        final TFormalParameter par = i.next();
        par.setTypeRef(this.<TypeRef>resolveVersion(par.getTypeRef(), contextVersion));
      }
    }
  }
}
