/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.ArrayList;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.CatchBlock;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ForStatement;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.LiteralOrComputedPropertyName;
import org.eclipse.n4js.n4JS.LocalArgumentsVariable;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.PropertyGetterDeclaration;
import org.eclipse.n4js.n4JS.PropertyMethodDeclaration;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.PropertySetterDeclaration;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.SetterDeclaration;
import org.eclipse.n4js.n4JS.ThisLiteral;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.YieldExpression;
import org.eclipse.n4js.n4idl.versioning.MigrationUtils;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.postprocessing.AbstractProcessor;
import org.eclipse.n4js.postprocessing.CompileTimeExpressionProcessor;
import org.eclipse.n4js.postprocessing.ComputedNameProcessor;
import org.eclipse.n4js.postprocessing.TypeDeferredProcessor;
import org.eclipse.n4js.postprocessing.TypeProcessor;
import org.eclipse.n4js.resource.N4JSResource;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TMigratable;
import org.eclipse.n4js.ts.types.TMigration;
import org.eclipse.n4js.ts.types.TypableElement;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.languages.N4LanguageUtils;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.util.CancelIndicator;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure0;

/**
 * Main processor used during {@link N4JSPostProcessor post-processing} of N4JS resources. It controls the overall
 * work flow of processing the AST, but does not do any actual work; instead, it delegates to the other processors:
 * <ul>
 * <li>{@link TypeProcessor}, which delegates further to
 *     <ul>
 *     <li>{@link PolyProcessor}, which delegates further to
 *         <ul>
 *         <li>{@link PolyProcessor_ArrayLiteral}
 *         <li>{@link PolyProcessor_ObjectLiteral}
 *         <li>{@link PolyProcessor_FunctionExpression}
 *         <li>{@link PolyProcessor_CallExpression}
 *         </ul>
 *     <li>{@link DestructureProcessor}
 *     </ul>
 * <li>{@code TypeExpectedProcessor} (coming soon!)
 * <li>{@link TypeDeferredProcessor}
 * </ul>
 */
@Singleton
@SuppressWarnings("all")
public class ASTProcessor extends AbstractProcessor {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private ComputedNameProcessor computedNameProcessor;
  
  @Inject
  private TypeProcessor typeProcessor;
  
  @Inject
  private TypeDeferredProcessor typeDeferredProcessor;
  
  @Inject
  private CompileTimeExpressionProcessor compileTimeExpressionProcessor;
  
  @Inject
  private JavaScriptVariantHelper variantHelper;
  
  /**
   * Entry point for processing of the entire AST of the given resource.
   * Will throw IllegalStateException if called more than once per N4JSResource.
   * <p>
   * This method performs some preparatory tasks (e.g., creating an instance of {@link ASTMetaInfoCache}) and ensures
   * consistency by tracking the 'isProcessing' state with try/finally; for actual processing, this method delegates
   * to method {@link #processAST(RuleEnvironment, Script, ASTMetaInfoCache)}.
   * 
   * @param resource  may not be null.
   * @param cancelIndicator  may be null.
   */
  public void processAST(final N4JSResource resource, final CancelIndicator cancelIndicator) {
    if ((resource == null)) {
      throw new IllegalArgumentException("resource may not be null");
    }
    resource.clearResolving();
    URI _uRI = resource.getURI();
    String _plus = ("### processing resource: " + _uRI);
    AbstractProcessor.log(0, _plus);
    final Script script = resource.getScript();
    final ASTMetaInfoCache cache = resource.getASTMetaInfoCacheVerifyContext();
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(resource);
    RuleEnvironmentExtensions.addCancelIndicator(G, cancelIndicator);
    try {
      this.processAST(G, script, cache);
    } finally {
      boolean _isCanceled = RuleEnvironmentExtensions.isCanceled(G);
      if (_isCanceled) {
        AbstractProcessor.log(0, "CANCELED by cancelIndicator");
      }
      boolean _isDEBUG_LOG_RESULT = AbstractProcessor.isDEBUG_LOG_RESULT();
      if (_isDEBUG_LOG_RESULT) {
        URI _uRI_1 = resource.getURI();
        String _plus_1 = ("### result for " + _uRI_1);
        AbstractProcessor.log(0, _plus_1);
        AbstractProcessor.log(4, resource.getScript(), cache);
      }
      URI _uRI_2 = resource.getURI();
      String _plus_2 = ("### done: " + _uRI_2);
      AbstractProcessor.log(0, _plus_2);
    }
  }
  
  /**
   * First method to actually perform processing of the AST. This method defines the various processing phases.
   * <p>
   * There exists a single "main phase" where 95% of processing happens (entry point for this main phase is method
   * {@link #processSubtree(RuleEnvironment, EObject, ASTMetaInfoCache, int)}), plus a number of smaller phases before
   * and after that where some special handling is performed.
   * 
   * @param resource  may not be null.
   * @param cancelIndicator  may be null.
   */
  private void processAST(final RuleEnvironment G, final Script script, final ASTMetaInfoCache cache) {
    Iterable<Expression> _iterable = IteratorExtensions.<Expression>toIterable(Iterators.<Expression>filter(script.eAllContents(), Expression.class));
    for (final Expression node : _iterable) {
      this.compileTimeExpressionProcessor.evaluateCompileTimeExpression(G, node, cache, 0);
    }
    Iterable<LiteralOrComputedPropertyName> _iterable_1 = IteratorExtensions.<LiteralOrComputedPropertyName>toIterable(Iterators.<LiteralOrComputedPropertyName>filter(script.eAllContents(), LiteralOrComputedPropertyName.class));
    for (final LiteralOrComputedPropertyName node_1 : _iterable_1) {
      this.computedNameProcessor.processComputedPropertyName(G, node_1, cache, 0);
    }
    this.processSubtree(G, script, cache, 0);
    EObject eObj = null;
    while (((eObj = cache.postponedSubTrees.poll()) != null)) {
      this.processSubtree(G, eObj, cache, 0);
    }
    for (final FunctionOrFieldAccessor potentialContainer : cache.potentialContainersOfLocalArgumentsVariable) {
      {
        final LocalArgumentsVariable lav = potentialContainer.get_lok();
        if ((lav != null)) {
          TypeRef _typeFailSafe = cache.getTypeFailSafe(lav);
          boolean _tripleEquals = (_typeFailSafe == null);
          if (_tripleEquals) {
            this.processSubtree(G, lav, cache, 0);
          }
        }
      }
    }
  }
  
  /**
   * Process given node and all of its direct and indirect children.
   * 
   * @param node  the root of the subtree to process; must be an AST node.
   */
  void processSubtree(final RuleEnvironment G, final EObject node, final ASTMetaInfoCache cache, final int indentLevel) {
    AbstractProcessor.assertTrueIfRigid(cache, "argument \'node\' must be an AST node", N4JSLanguageUtils.isASTNode(node));
    String _objectInfo = AbstractProcessor.getObjectInfo(node);
    String _plus = ("processing: " + _objectInfo);
    AbstractProcessor.log(indentLevel, _plus);
    this.checkCanceled(G);
    boolean _contains = cache.forwardProcessedSubTrees.contains(node);
    if (_contains) {
      boolean _isDEBUG_LOG = AbstractProcessor.isDEBUG_LOG();
      if (_isDEBUG_LOG) {
        AbstractProcessor.log(indentLevel, "(subtree already processed as a forward reference)");
        if ((node instanceof TypableElement)) {
          AbstractProcessor.log(indentLevel, cache.getTypeFailSafe(((TypableElement)node)));
        }
      }
      return;
    }
    boolean _contains_1 = cache.postponedSubTrees.contains(node);
    if (_contains_1) {
      throw new IllegalStateException("eager processing of postponed subtree");
    }
    boolean _add = cache.astNodesCurrentlyBeingTyped.add(node);
    boolean _not = (!_add);
    if (_not) {
      boolean _isDEBUG_LOG_1 = AbstractProcessor.isDEBUG_LOG();
      if (_isDEBUG_LOG_1) {
        AbstractProcessor.log(indentLevel, "(subtree currently in progress - skipping)");
      }
      return;
    }
    try {
      this.processNode_preChildren(G, node, cache, indentLevel);
      final List<EObject> children = this.childrenToBeProcessed(G, node);
      for (final EObject child : children) {
        boolean _isPostponedNode = this.isPostponedNode(child);
        if (_isPostponedNode) {
          cache.postponedSubTrees.add(child);
        } else {
          this.processSubtree(G, child, cache, (indentLevel + 1));
          this.checkCanceled(G);
        }
      }
      this.processNode_postChildren(G, node, cache, indentLevel);
      this.resolveAndProcessReferencesInNode(node, cache);
    } finally {
      cache.astNodesCurrentlyBeingTyped.remove(node);
    }
  }
  
  private boolean isPostponedNode(final EObject node) {
    return (this.isPostponedInitializer(node) || N4JSASTUtils.isBodyOfFunctionOrFieldAccessor(node));
  }
  
  /**
   * Initializers are postponed iff:
   * <ul>
   * <li>Node is an initializer of a FormalParameter p,</li>
   * <li>and p is part of a Poly FunctionExpression f,</li>
   * <li>and p contains references to other FormalParameters of f, or f itself.</li>
   * </ul>
   */
  private boolean isPostponedInitializer(final EObject node) {
    boolean isPostponedInitializer = false;
    final EObject fpar = node.eContainer();
    if ((fpar instanceof FormalParameter)) {
      if ((node instanceof Expression)) {
        boolean _isHasInitializerAssignment = ((FormalParameter)fpar).isHasInitializerAssignment();
        if (_isHasInitializerAssignment) {
          final EObject funDef = ((FormalParameter)fpar).eContainer();
          if ((funDef instanceof FunctionExpression)) {
            final EList<FormalParameter> allFPars = ((FunctionExpression)funDef).getFpars();
            final List<IdentifierRef> allRefs = EcoreUtilN4.<IdentifierRef>getAllContentsOfTypeStopAt(fpar, IdentifierRef.class, N4JSPackage.Literals.FUNCTION_OR_FIELD_ACCESSOR__BODY);
            for (final IdentifierRef ir : allRefs) {
              {
                final IdentifiableElement id = ir.getId();
                final boolean idRefCausesCyclDep = (allFPars.contains(id) || ((id instanceof VariableDeclaration) && (((VariableDeclaration) id).getExpression() == funDef)));
                if (idRefCausesCyclDep) {
                  isPostponedInitializer = true;
                }
              }
            }
          }
          final boolean thisLiteralCausesCyclDep = ((funDef instanceof PropertyMethodDeclaration) || ((funDef instanceof FunctionExpression) && (funDef.eContainer() instanceof PropertyNameValuePair)));
          if (thisLiteralCausesCyclDep) {
            final boolean containsThisLiteral = EcoreUtilN4.<ThisLiteral>containsContentsOfTypeStopAt(fpar, ThisLiteral.class, N4JSPackage.Literals.FUNCTION_OR_FIELD_ACCESSOR__BODY);
            if (containsThisLiteral) {
              isPostponedInitializer = true;
            }
          }
        }
      }
    }
    return isPostponedInitializer;
  }
  
  /**
   * Forward-process given node and all of its direct and indirect children.
   * <p>
   * Via this method, other processors can request a forward processing of some subtree. Does nothing if the given
   * node was processed already, either as part of a forward reference or during normal processing.
   * 
   * @return <code>true</code> iff the forward processing is legal, <code>false</code> otherwise.
   */
  boolean processSubtree_forwardReference(final RuleEnvironment G, final TypableElement node, final ASTMetaInfoCache cache) {
    AbstractProcessor.assertTrueIfRigid(cache, "argument \'node\' must be an AST node", N4JSLanguageUtils.isASTNode(node));
    final boolean valid = (N4JSLanguageUtils.isIdentifiableSubtree(node) || ASTProcessor.isExceptionCaseOfForwardReferencableSubtree(node));
    if ((!valid)) {
      Resource _eResource = node.eResource();
      final XtextResource resource = ((XtextResource) _eResource);
      if ((resource != null)) {
        String _text = resource.getParseResult().getRootNode().getText();
        String _plus = ((("forward reference only allowed to identifiable subtrees; but was: " + node) + " in\n") + _text);
        AbstractProcessor.assertTrueIfRigid(cache, _plus, valid);
      } else {
        AbstractProcessor.assertTrueIfRigid(cache, ("forward reference only allowed to identifiable subtrees; but was: " + node), valid);
      }
    }
    final TypeRef fromCache = cache.getTypeFailSafe(node);
    if ((fromCache != null)) {
      return true;
    }
    boolean _contains = cache.astNodesCurrentlyBeingTyped.contains(node);
    if (_contains) {
      final boolean isCyclicForwardReference = cache.astNodesCurrentlyBeingTyped.contains(node);
      if ((isCyclicForwardReference && 
        (((((((node instanceof VariableDeclaration) || (node instanceof N4ClassifierDeclaration)) || (node instanceof N4FieldDeclaration)) || ((node instanceof PropertyNameValuePair) && (((PropertyNameValuePair) node).getExpression() instanceof FunctionExpression))) || (node instanceof PropertyGetterDeclaration)) || (node instanceof PropertySetterDeclaration)) || ((node instanceof Expression) && (node.eContainer() instanceof YieldExpression))))) {
        return true;
      }
      String _objectInfo = AbstractProcessor.getObjectInfo(node);
      String _plus_1 = ("*#*#*#*#*#* illegal cyclic forward reference to " + _objectInfo);
      String _plus_2 = (_plus_1 + " (resource: ");
      Resource _eResource_1 = node.eResource();
      URI _uRI = null;
      if (_eResource_1!=null) {
        _uRI=_eResource_1.getURI();
      }
      String _plus_3 = (_plus_2 + _uRI);
      final String msg = (_plus_3 + ")");
      AbstractProcessor.logErr(msg);
      return false;
    } else {
      boolean _isSemiCyclicForwardReferenceInForLoop = this.isSemiCyclicForwardReferenceInForLoop(node, cache);
      if (_isSemiCyclicForwardReferenceInForLoop) {
        return true;
      }
    }
    boolean _contains_1 = cache.forwardProcessedSubTrees.contains(node);
    if (_contains_1) {
      throw new IllegalStateException();
    }
    String _objectInfo_1 = AbstractProcessor.getObjectInfo(node);
    String _plus_4 = ("===START of identifiable sub-tree below " + _objectInfo_1);
    AbstractProcessor.log(0, _plus_4);
    final RuleEnvironment G_fresh = RuleEnvironmentExtensions.newRuleEnvironment(G);
    this.processSubtree(G_fresh, node, cache, 0);
    cache.forwardProcessedSubTrees.add(node);
    String _objectInfo_2 = AbstractProcessor.getObjectInfo(node);
    String _plus_5 = ("===END of identifiable sub-tree below " + _objectInfo_2);
    AbstractProcessor.log(0, _plus_5);
    return true;
  }
  
  /**
   * Top-down processing of AST nodes happens here, i.e. this method will see all AST nodes in a top-down order.
   */
  private void processNode_preChildren(final RuleEnvironment G, final EObject node, final ASTMetaInfoCache cache, final int indentLevel) {
    if ((node instanceof FunctionOrFieldAccessor)) {
      cache.potentialContainersOfLocalArgumentsVariable.add(((FunctionOrFieldAccessor)node));
    }
    if ((node instanceof FunctionDefinition)) {
      this.handleAsyncFunctionDefinition(G, ((FunctionDefinition)node), cache);
      this.handleGeneratorFunctionDefinition(G, ((FunctionDefinition)node), cache);
    }
    this.typeDeferredProcessor.handleDeferredTypeRefs_preChildren(G, node, cache);
  }
  
  /**
   * Bottom-up processing of AST nodes happens here, i.e. this method will see all AST nodes in a bottom-up order.
   */
  private void processNode_postChildren(final RuleEnvironment G, final EObject node, final ASTMetaInfoCache cache, final int indentLevel) {
    this.typeDeferredProcessor.handleDeferredTypeRefs_postChildren(G, node, cache);
    this.typeProcessor.typeNode(G, node, cache, indentLevel);
    if ((node instanceof NamedImportSpecifier)) {
      final TExportableElement elem = ((NamedImportSpecifier)node).getImportedElement();
      if ((elem != null)) {
        final N4JSTypeSystem tsCorrect = N4LanguageUtils.<N4JSTypeSystem>getServiceForContext(elem, N4JSTypeSystem.class).orElse(this.ts);
        tsCorrect.type(G, elem);
      }
    }
    if (((node instanceof FunctionDeclaration) && 
      AnnotationDefinition.MIGRATION.hasAnnotation(((FunctionDeclaration) node)))) {
      TFunction _definedFunction = ((FunctionDeclaration) node).getDefinedFunction();
      this.registerMigrationWithTypes(((TMigration) _definedFunction));
    }
  }
  
  /**
   * This method returns the direct children of 'obj' that are to be processed, <em>in the order in which they are to
   * be processed</em>. By default, all direct children must be processed and the order is insignificant, so in the
   * default case this method simply returns {@link EObject#eContents()}. However, this method implements special
   * handling for some exception cases where the processing order is significant.
   */
  private List<EObject> childrenToBeProcessed(final RuleEnvironment G, final EObject obj) {
    List<EObject> _switchResult = null;
    boolean _matched = false;
    if (obj instanceof SetterDeclaration) {
      _matched=true;
      _switchResult = this.<EObject>bringToFront(((SetterDeclaration)obj).eContents(), ((SetterDeclaration)obj).getFpar());
    }
    if (!_matched) {
      if (obj instanceof FunctionDefinition) {
        _matched=true;
        _switchResult = this.<EObject>bringToFront(((FunctionDefinition)obj).eContents(), ((EObject[])Conversions.unwrapArray(((FunctionDefinition)obj).getFpars(), EObject.class)));
      }
    }
    if (!_matched) {
      if (obj instanceof CatchBlock) {
        _matched=true;
        _switchResult = this.<EObject>bringToFront(((CatchBlock)obj).eContents(), ((CatchBlock)obj).getCatchVariable());
      }
    }
    if (!_matched) {
      if (obj instanceof ForStatement) {
        _matched=true;
        _switchResult = this.<EObject>bringToFront(((ForStatement)obj).eContents(), ((ForStatement)obj).getExpression());
      }
    }
    if (!_matched) {
      if (obj instanceof ParameterizedCallExpression) {
        boolean _isMigrateCall = MigrationUtils.isMigrateCall(obj);
        if (_isMigrateCall) {
          _matched=true;
          _switchResult = this.<EObject>bringToFront(((ParameterizedCallExpression)obj).eContents(), ((EObject[])Conversions.unwrapArray(((ParameterizedCallExpression)obj).getArguments(), EObject.class)));
        }
      }
    }
    if (!_matched) {
      if (obj instanceof Script) {
        boolean _allowVersionedTypes = this.variantHelper.allowVersionedTypes(obj);
        if (_allowVersionedTypes) {
          _matched=true;
          final Function1<FunctionDeclaration, Boolean> _function = (FunctionDeclaration it) -> {
            return Boolean.valueOf(MigrationUtils.isMigrationDefinition(it));
          };
          _switchResult = this.<EObject>bringToFront(((Script)obj).eContents(), 
            ((EObject[])Conversions.unwrapArray(IterableExtensions.<FunctionDeclaration>filter(Iterables.<FunctionDeclaration>filter(((Script)obj).getScriptElements(), FunctionDeclaration.class), _function), EObject.class)));
        }
      }
    }
    if (!_matched) {
      _switchResult = obj.eContents();
    }
    return _switchResult;
  }
  
  /**
   * Normally, forward references are allowed only to {@link N4JSLanguageUtils#isIdentifiableSubtree(EObject)
   * identifiable subtrees}. However, there are exception cases that are also allowed and this method returns
   * <code>true</code> for those cases.
   */
  private static boolean isExceptionCaseOfForwardReferencableSubtree(final EObject astNode) {
    return ASTProcessor.isExpressionInForOf(astNode);
  }
  
  private static boolean isExpressionInForOf(final EObject astNode) {
    return ((((astNode instanceof Expression) && (astNode.eContainer() instanceof ForStatement)) && ((ForStatement) astNode.eContainer()).isForOf()) && (astNode.eContainingFeature() == N4JSPackage.eINSTANCE.getIterationStatement_Expression()));
  }
  
  /**
   * Returns true if we have a semi-cyclic reference to a variable declaration in a for in/of loop.
   * For example:
   * <pre>
   * for(var x of foo(x)) {}
   * </pre>
   */
  boolean isSemiCyclicForwardReferenceInForLoop(final EObject node, final ASTMetaInfoCache cache) {
    if ((node instanceof VariableDeclaration)) {
      final EObject parent = ((VariableDeclaration)node).eContainer();
      if ((parent instanceof ForStatement)) {
        return ((((ForStatement)parent).isForIn() || ((ForStatement)parent).isForOf()) && cache.astNodesCurrentlyBeingTyped.contains(((ForStatement)parent).getExpression()));
      }
    }
    return false;
  }
  
  private void resolveAndProcessReferencesInNode(final EObject astNode, final ASTMetaInfoCache cache) {
    EList<EReference> _eAllReferences = astNode.eClass().getEAllReferences();
    for (final EReference eRef : _eAllReferences) {
      if (((!eRef.isContainment()) && (!eRef.isContainer()))) {
        final Object node = astNode.eGet(eRef, true);
        if ((node instanceof EObject)) {
          this.recordReferencesToLocalVariables(eRef, astNode, ((EObject)node), cache);
        }
      }
    }
  }
  
  /**
   * Registers the given {@link TMigration} with the corresponding
   * principal argument.
   */
  private void registerMigrationWithTypes(final TMigration migration) {
    if ((null == migration)) {
      return;
    }
    boolean _isEmpty = migration.getTypeVars().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      return;
    }
    TMigratable _principalArgumentType = migration.getPrincipalArgumentType();
    boolean _tripleNotEquals = (null != _principalArgumentType);
    if (_tripleNotEquals) {
      this.registerMigrationWithType(migration, migration.getPrincipalArgumentType());
    }
  }
  
  private void registerMigrationWithType(final TMigration migration, final TMigratable migratable) {
    final Procedure0 _function = () -> {
      EList<TMigration> _migrations = migratable.getMigrations();
      _migrations.add(migration);
    };
    EcoreUtilN4.doWithDeliver(false, _function, migratable);
  }
  
  private void recordReferencesToLocalVariables(final EReference reference, final EObject sourceNode, final EObject targetNode, final ASTMetaInfoCache cache) {
    boolean _eIsProxy = targetNode.eIsProxy();
    if (_eIsProxy) {
      return;
    }
    Resource _eResource = sourceNode.eResource();
    Resource _eResource_1 = targetNode.eResource();
    boolean _tripleNotEquals = (_eResource != _eResource_1);
    if (_tripleNotEquals) {
      return;
    }
    if ((targetNode instanceof VariableDeclaration)) {
      if ((targetNode instanceof ExportedVariableDeclaration)) {
        return;
      }
      cache.storeLocalVariableReference(((VariableDeclaration)targetNode), sourceNode);
    }
  }
  
  private <T extends Object> List<T> bringToFront(final List<T> l, final T... elements) {
    final ArrayList<T> result = new ArrayList<T>(l);
    final List<T> elemSanitized = IterableExtensions.<T>toList(IterableExtensions.<T>filterNull(((Iterable<T>)Conversions.doWrapArray(elements))));
    result.removeAll(elemSanitized);
    result.addAll(0, elemSanitized);
    return result;
  }
}
