/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.resource;

import com.google.common.base.Objects;
import com.google.common.base.Strings;
import com.google.inject.Singleton;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.N4JSGlobals;

/**
 * This class provides methods for calculating file extensions. The calculation takes into account Xpect file extension
 * {@code .xt}. Custom file Xpect file extensions are not supported e.g. when your Xpect tests are configured to run with {@code .xxt} file extension.
 * Deeply nested structures are also not supported e.g. using {@code file.n4js.xt.xt}.
 */
@Singleton
@SuppressWarnings("all")
public class XpectAwareFileExtensionCalculator {
  public String getXpectAwareFileExtension(final EObject eob) {
    if (((eob == null) || (eob.eResource() == null))) {
      return "";
    }
    return this.getXpectAwareFileExtension(eob.eResource().getURI());
  }
  
  /**
   * Return the file extension of an URI
   */
  public String getXpectAwareFileExtension(final URI uri) {
    if ((uri == null)) {
      return "";
    }
    return this.getXpectAwareFileExtensionOrEmpty(uri);
  }
  
  /**
   * Returns the name of the file that is referenced by {@code uri}
   * without the potential additional X!PECT file extension.
   */
  public String getFilenameWithoutXpectExtension(final URI uri) {
    String _fileExtension = uri.fileExtension();
    String _xpectAwareFileExtension = this.getXpectAwareFileExtension(uri);
    boolean _notEquals = (!Objects.equal(_fileExtension, _xpectAwareFileExtension));
    if (_notEquals) {
      return uri.trimFileExtension().lastSegment();
    } else {
      return uri.lastSegment();
    }
  }
  
  private String getXpectAwareFileExtensionOrEmpty(final URI uri) {
    String ext = uri.fileExtension();
    boolean _equals = N4JSGlobals.XT_FILE_EXTENSION.equals(ext);
    if (_equals) {
      ext = uri.trimFileExtension().fileExtension();
    }
    return Strings.nullToEmpty(ext);
  }
}
