/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesbuilder;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.PropertyGetterDeclaration;
import org.eclipse.n4js.n4JS.PropertyMethodDeclaration;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.PropertySetterDeclaration;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TStructField;
import org.eclipse.n4js.ts.types.TStructGetter;
import org.eclipse.n4js.ts.types.TStructMember;
import org.eclipse.n4js.ts.types.TStructMethod;
import org.eclipse.n4js.ts.types.TStructSetter;
import org.eclipse.n4js.ts.types.TStructuralType;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesbuilder.N4JSFormalParameterTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSTypesBuilderHelper;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class N4JSObjectLiteralTypesBuilder {
  @Inject
  @Extension
  private N4JSTypesBuilderHelper _n4JSTypesBuilderHelper;
  
  @Inject
  @Extension
  private N4JSFormalParameterTypesBuilder _n4JSFormalParameterTypesBuilder;
  
  void createObjectLiteral(final ObjectLiteral objectLiteral, final TModule target, final boolean preLinkingPhase) {
    final BuiltInTypeScope builtInTypeScope = BuiltInTypeScope.get(objectLiteral.eResource().getResourceSet());
    final TStructuralType structType = TypesFactory.eINSTANCE.createTStructuralType();
    final Function1<PropertyAssignment, Boolean> _function = (PropertyAssignment it) -> {
      return Boolean.valueOf(((it.getName() != null) || it.hasComputedPropertyName()));
    };
    final Consumer<PropertyAssignment> _function_1 = (PropertyAssignment it) -> {
      final TStructMember typeModelElement = this.createTypeModelElement(structType, objectLiteral, it, builtInTypeScope, preLinkingPhase);
      if ((typeModelElement != null)) {
        typeModelElement.setAstElement(it);
        EList<TStructMember> _ownedMembers = structType.getOwnedMembers();
        _ownedMembers.add(typeModelElement);
      }
    };
    IterableExtensions.<PropertyAssignment>filter(objectLiteral.getPropertyAssignments(), _function).forEach(_function_1);
    structType.setAstElement(objectLiteral);
    objectLiteral.setDefinedType(structType);
    EList<Type> _internalTypes = target.getInternalTypes();
    _internalTypes.add(structType);
  }
  
  private TStructMember _createTypeModelElement(final TStructuralType structType, final ObjectLiteral objectLiteral, final PropertyAssignment assignment, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    return null;
  }
  
  /**
   * Creates a TStructField.
   */
  private TStructField _createTypeModelElement(final TStructuralType structType, final ObjectLiteral objectLiteral, final PropertyNameValuePair nameValuePair, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    final TStructField field = TypesFactory.eINSTANCE.createTStructField();
    this._n4JSTypesBuilderHelper.setMemberName(field, nameValuePair);
    field.setOptional(nameValuePair.isDeclaredOptional());
    TypeRef _declaredTypeRef = nameValuePair.getDeclaredTypeRef();
    boolean _tripleNotEquals = (_declaredTypeRef != null);
    if (_tripleNotEquals) {
      if ((!preLinkingPhase)) {
        field.setTypeRef(TypeUtils.<TypeRef>copyWithProxies(nameValuePair.getDeclaredTypeRef()));
      }
    } else {
      Expression _expression = nameValuePair.getExpression();
      boolean _tripleNotEquals_1 = (_expression != null);
      if (_tripleNotEquals_1) {
        field.setTypeRef(TypeUtils.createDeferredTypeRef());
      } else {
        field.setTypeRef(builtInTypeScope.getAnyTypeRef());
      }
    }
    field.setAstElement(nameValuePair);
    nameValuePair.setDefinedField(field);
    return field;
  }
  
  /**
   * Creates a TStructGetter.
   */
  private TStructGetter _createTypeModelElement(final TStructuralType structType, final ObjectLiteral objectLiteral, final PropertyGetterDeclaration getterDecl, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    final TStructGetter getter = TypesFactory.eINSTANCE.createTStructGetter();
    this._n4JSTypesBuilderHelper.setMemberName(getter, getterDecl);
    getter.setOptional(getterDecl.isDeclaredOptional());
    TypeRef _declaredTypeRef = getterDecl.getDeclaredTypeRef();
    boolean _tripleNotEquals = (_declaredTypeRef != null);
    if (_tripleNotEquals) {
      if ((!preLinkingPhase)) {
        getter.setDeclaredTypeRef(TypeUtils.<TypeRef>copyWithProxies(getterDecl.getDeclaredTypeRef()));
      }
    } else {
      getter.setDeclaredTypeRef(TypeUtils.createDeferredTypeRef());
    }
    getter.setAstElement(getterDecl);
    getterDecl.setDefinedGetter(getter);
    return getter;
  }
  
  /**
   * Creates a TStructSetter.
   */
  private TStructSetter _createTypeModelElement(final TStructuralType structType, final ObjectLiteral objectLiteral, final PropertySetterDeclaration setterDecl, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    final TStructSetter setter = TypesFactory.eINSTANCE.createTStructSetter();
    this._n4JSTypesBuilderHelper.setMemberName(setter, setterDecl);
    setter.setOptional(setterDecl.isDeclaredOptional());
    final TFormalParameter param = TypesFactory.eINSTANCE.createTFormalParameter();
    FormalParameter _fpar = setterDecl.getFpar();
    boolean _tripleNotEquals = (_fpar != null);
    if (_tripleNotEquals) {
      param.setName(setterDecl.getFpar().getName());
      final TypeRef fparDeclTypeRef = setterDecl.getFpar().getDeclaredTypeRef();
      if ((fparDeclTypeRef != null)) {
        if ((!preLinkingPhase)) {
          param.setTypeRef(TypeUtils.<TypeRef>copyWithProxies(fparDeclTypeRef));
        }
      } else {
        param.setTypeRef(TypeUtils.createDeferredTypeRef());
      }
    } else {
      param.setTypeRef(TypeUtils.createDeferredTypeRef());
    }
    setter.setFpar(param);
    setter.setAstElement(setterDecl);
    setterDecl.setDefinedSetter(setter);
    return setter;
  }
  
  /**
   * Creates a TStructMethod.
   */
  private TStructMethod _createTypeModelElement(final TStructuralType structType, final ObjectLiteral objectLiteral, final PropertyMethodDeclaration methodDecl, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    final TStructMethod result = TypesFactory.eINSTANCE.createTStructMethod();
    this._n4JSTypesBuilderHelper.setMemberName(result, methodDecl);
    EList<TFormalParameter> _fpars = result.getFpars();
    final Function1<FormalParameter, TFormalParameter> _function = (FormalParameter it) -> {
      return this._n4JSFormalParameterTypesBuilder.createFormalParameter(it, builtInTypeScope, preLinkingPhase);
    };
    List<TFormalParameter> _map = ListExtensions.<FormalParameter, TFormalParameter>map(methodDecl.getFpars(), _function);
    Iterables.<TFormalParameter>addAll(_fpars, _map);
    TypeRef _returnTypeRef = methodDecl.getReturnTypeRef();
    boolean _tripleNotEquals = (_returnTypeRef != null);
    if (_tripleNotEquals) {
      if ((!preLinkingPhase)) {
        result.setReturnTypeRef(TypeUtils.<TypeRef>copyWithProxies(methodDecl.getReturnTypeRef()));
      }
    } else {
      result.setReturnTypeRef(builtInTypeScope.getVoidTypeRef());
    }
    result.setAstElement(methodDecl);
    methodDecl.setDefinedType(result);
    return result;
  }
  
  private TStructMember createTypeModelElement(final TStructuralType structType, final ObjectLiteral objectLiteral, final PropertyAssignment getterDecl, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    if (getterDecl instanceof PropertyGetterDeclaration) {
      return _createTypeModelElement(structType, objectLiteral, (PropertyGetterDeclaration)getterDecl, builtInTypeScope, preLinkingPhase);
    } else if (getterDecl instanceof PropertyMethodDeclaration) {
      return _createTypeModelElement(structType, objectLiteral, (PropertyMethodDeclaration)getterDecl, builtInTypeScope, preLinkingPhase);
    } else if (getterDecl instanceof PropertyNameValuePair) {
      return _createTypeModelElement(structType, objectLiteral, (PropertyNameValuePair)getterDecl, builtInTypeScope, preLinkingPhase);
    } else if (getterDecl instanceof PropertySetterDeclaration) {
      return _createTypeModelElement(structType, objectLiteral, (PropertySetterDeclaration)getterDecl, builtInTypeScope, preLinkingPhase);
    } else if (getterDecl != null) {
      return _createTypeModelElement(structType, objectLiteral, getterDecl, builtInTypeScope, preLinkingPhase);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(structType, objectLiteral, getterDecl, builtInTypeScope, preLinkingPhase).toString());
    }
  }
}
