/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesystem.utils;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.scoping.members.TypingStrategyFilter;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.IntersectionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRefStructural;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.typeRefs.UnionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.Wildcard;
import org.eclipse.n4js.ts.types.AnyType;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TStructField;
import org.eclipse.n4js.ts.types.TStructGetter;
import org.eclipse.n4js.ts.types.TStructMember;
import org.eclipse.n4js.ts.types.TStructMethod;
import org.eclipse.n4js.ts.types.TStructSetter;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.utils.SuperTypesList;
import org.eclipse.n4js.ts.utils.TypeCompareHelper;
import org.eclipse.n4js.ts.utils.TypeHelper;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.GenericsComputer;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelperStrategy;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Type System Helper Strategy computing the join of a given collection of types.
 * <p>
 * Definition from [Pierce02a, pp. 218]:
 * <pre>
 * Join J = S v T, if
 *     S <: J, T <:J,                         common super type
 *     forall U: S <:U and T <:U => J <: U    least
 * </pre>
 * <p>
 * It is also rigor with regard to auto-conversion. That is, {@code string} and {@String} are
 * not automatically converted into each other. This is because both types behave differently.
 * E.g., if used as a boolean expression, {@code ""} and {@code new String("")} will be evaluated differently.
 * <em>The object type will always return true.</em> This is even true for {@code new Boolean(false)}, which will also
 * be evaluated to true (note however that {@code Boolean(false)} will return a {@code boolean} value, which
 * is then evaluated to false).
 * <p>
 * Special pseudo sub types of string, e.g. pathSelector, are handled as well.
 */
@SuppressWarnings("all")
class JoinComputer extends TypeSystemHelperStrategy {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  @Extension
  private TypeCompareHelper _typeCompareHelper;
  
  @Inject
  @Extension
  private TypeHelper _typeHelper;
  
  @Inject
  private ContainerTypesHelper containerTypesHelper;
  
  @Inject
  private GenericsComputer genericsComputer;
  
  /**
   * Returns the join, sometimes called least common super type (LCST),
   * of the given types. This may be an intersection type but not a union type.
   * See class description for details.
   * 
   * @return the join, may be a contained reference. Thus clients may need to create a copy!
   */
  public TypeRef join(final RuleEnvironment G, final Iterable<? extends TypeRef> typeRefsToJoin) {
    if ((typeRefsToJoin == null)) {
      return null;
    }
    final Iterable<? extends TypeRef> typeRefs = IterableExtensions.filterNull(typeRefsToJoin);
    boolean _isEmpty = IterableExtensions.isEmpty(typeRefs);
    if (_isEmpty) {
      return null;
    }
    int _size = IterableExtensions.size(typeRefs);
    boolean _tripleEquals = (_size == 1);
    if (_tripleEquals) {
      return IterableExtensions.head(typeRefs);
    }
    final Iterable<UnionTypeExpression> unionTypeRefs = Iterables.<UnionTypeExpression>filter(typeRefs, UnionTypeExpression.class);
    final Function1<TypeRef, Boolean> _function = (TypeRef it) -> {
      return Boolean.valueOf((!(it instanceof UnionTypeExpression)));
    };
    final Iterable<? extends TypeRef> nonUnionTypeRefs = IterableExtensions.filter(typeRefs, _function);
    final TypeRef nonUnionJoin = this.joinNonUnionTypes(G, nonUnionTypeRefs);
    boolean _isEmpty_1 = IterableExtensions.isEmpty(unionTypeRefs);
    if (_isEmpty_1) {
      return nonUnionJoin;
    } else {
      return this.joinNonUnionTypeWithUniontypes(G, nonUnionJoin, unionTypeRefs);
    }
  }
  
  /**
   * join result of non-union types join with all union types
   *  @see [N4JS Spec], 4.12 Union Type
   * @param unionTypeRefs cannot be empty, called only from method above where it is checked
   */
  private TypeRef joinNonUnionTypeWithUniontypes(final RuleEnvironment G, final TypeRef nonUnionJoin, final Iterable<UnionTypeExpression> unionTypeRefs) {
    if (((nonUnionJoin == null) && (IterableExtensions.size(unionTypeRefs) == 1))) {
      return this._typeSystemHelper.<UnionTypeExpression>simplify(G, IterableExtensions.<UnionTypeExpression>head(unionTypeRefs));
    }
    final UnionTypeExpression union = TypeRefsFactory.eINSTANCE.createUnionTypeExpression();
    if ((nonUnionJoin != null)) {
      union.getTypeRefs().add(TypeUtils.<TypeRef>copyIfContained(nonUnionJoin));
    }
    final Function1<UnionTypeExpression, UnionTypeExpression> _function = (UnionTypeExpression it) -> {
      return TypeUtils.<UnionTypeExpression>copyIfContained(it);
    };
    Iterables.<TypeRef>addAll(union.getTypeRefs(), IterableExtensions.<UnionTypeExpression, UnionTypeExpression>map(unionTypeRefs, _function));
    return this._typeSystemHelper.<UnionTypeExpression>simplify(G, union);
  }
  
  /**
   * Called internally to compute the join of non-union types only.
   * @param nonUnionTypeRefs type references, must not contain UnionTypeExpressions
   */
  private TypeRef joinNonUnionTypes(final RuleEnvironment G, final Iterable<? extends TypeRef> nonUnionTypeRefs) {
    boolean _isEmpty = IterableExtensions.isEmpty(nonUnionTypeRefs);
    if (_isEmpty) {
      return null;
    }
    int _size = IterableExtensions.size(nonUnionTypeRefs);
    boolean _tripleEquals = (_size == 1);
    if (_tripleEquals) {
      return IterableExtensions.head(nonUnionTypeRefs);
    }
    List<TypeRef> commonSuperTypesIgnoreTypeArgs = null;
    final Type firstType = IterableExtensions.head(nonUnionTypeRefs).getDeclaredType();
    if (((firstType != null) && IterableExtensions.forall(nonUnionTypeRefs, ((Function1<TypeRef, Boolean>) (TypeRef it) -> {
      Type _declaredType = it.getDeclaredType();
      return Boolean.valueOf((_declaredType == firstType));
    })))) {
      final TypeRef head = IterableExtensions.head(nonUnionTypeRefs);
      if (((!head.isParameterized()) && IterableExtensions.forall(nonUnionTypeRefs, ((Function1<TypeRef, Boolean>) (TypeRef it) -> {
        boolean _isUseSiteStructuralTyping = it.isUseSiteStructuralTyping();
        return Boolean.valueOf((!_isUseSiteStructuralTyping));
      })))) {
        return head;
      }
      commonSuperTypesIgnoreTypeArgs = Collections.<TypeRef>singletonList(head);
    } else {
      commonSuperTypesIgnoreTypeArgs = this.commonSuperTypesTypeargsIgnored(G, nonUnionTypeRefs);
      boolean _isEmpty_1 = commonSuperTypesIgnoreTypeArgs.isEmpty();
      if (_isEmpty_1) {
        return RuleEnvironmentExtensions.anyTypeRef(G);
      }
      int _size_1 = commonSuperTypesIgnoreTypeArgs.size();
      boolean _greaterThan = (_size_1 > 1);
      if (_greaterThan) {
        int i = 0;
        while ((i < commonSuperTypesIgnoreTypeArgs.size())) {
          {
            this.removeAllSuperTypesOfType(commonSuperTypesIgnoreTypeArgs, commonSuperTypesIgnoreTypeArgs.get(i), G);
            i = (i + 1);
          }
        }
      }
    }
    List<TypeRef> commonSuperTypesParameterized = null;
    final Function1<TypeRef, Boolean> _function = (TypeRef it) -> {
      boolean _isParameterized = it.isParameterized();
      return Boolean.valueOf((!_isParameterized));
    };
    boolean _forall = IterableExtensions.<TypeRef>forall(commonSuperTypesIgnoreTypeArgs, _function);
    if (_forall) {
      commonSuperTypesParameterized = commonSuperTypesIgnoreTypeArgs;
    } else {
      commonSuperTypesParameterized = Lists.<TypeRef>newArrayListWithCapacity(commonSuperTypesIgnoreTypeArgs.size());
      for (final TypeRef superTypeIgnoreTypeArgs : commonSuperTypesIgnoreTypeArgs) {
        boolean _isParameterized = superTypeIgnoreTypeArgs.isParameterized();
        boolean _not = (!_isParameterized);
        if (_not) {
          commonSuperTypesParameterized.add(TypeUtils.<TypeRef>copyIfContained(superTypeIgnoreTypeArgs));
        } else {
          final Set<TypeRef> parameterizedSuperTypes = this.collectParameterizedSuperType(nonUnionTypeRefs, 
            superTypeIgnoreTypeArgs.getDeclaredType(), G);
          int _size_2 = parameterizedSuperTypes.size();
          boolean _equals = (_size_2 == 1);
          if (_equals) {
            commonSuperTypesParameterized.add(TypeUtils.<TypeRef>copyIfContained(IterableExtensions.<TypeRef>head(parameterizedSuperTypes)));
          } else {
            final TypeRef merged = TypeUtils.<TypeRef>copy(IterableExtensions.<TypeRef>head(parameterizedSuperTypes));
            merged.getTypeArgs().clear();
            int i_1 = 0;
            while ((i_1 < merged.getDeclaredType().getTypeVars().size())) {
              {
                final int currentIndex = i_1;
                final Function1<TypeRef, TypeRef> _function_1 = (TypeRef it) -> {
                  return this.ts.upperBound(G, it.getTypeArgs().get(currentIndex));
                };
                final TypeRef upperBound = this.join(G, 
                  IterableExtensions.<TypeRef, TypeRef>map(parameterizedSuperTypes, _function_1));
                final Function1<TypeRef, TypeRef> _function_2 = (TypeRef it) -> {
                  return this.ts.lowerBound(G, it.getTypeArgs().get(currentIndex));
                };
                final TypeRef lowerBound = this._typeSystemHelper.meet(G, 
                  IterableExtensions.<TypeRef, TypeRef>map(parameterizedSuperTypes, _function_2));
                int _compare = this._typeCompareHelper.compare(upperBound, lowerBound);
                boolean _equals_1 = (_compare == 0);
                if (_equals_1) {
                  merged.getTypeArgs().add(TypeUtils.<TypeRef>copyIfContained(upperBound));
                } else {
                  final Wildcard wildcard = TypeRefsFactory.eINSTANCE.createWildcard();
                  if ((upperBound.isTopType() && (!lowerBound.isBottomType()))) {
                    wildcard.setDeclaredLowerBound(TypeUtils.<TypeRef>copyIfContained(lowerBound));
                  } else {
                    wildcard.setDeclaredUpperBound(TypeUtils.<TypeRef>copyIfContained(upperBound));
                  }
                  merged.getTypeArgs().add(wildcard);
                }
                i_1 = (i_1 + 1);
              }
            }
            commonSuperTypesParameterized.add(merged);
          }
        }
      }
    }
    final Function1<TypeRef, Boolean> _function_1 = (TypeRef it) -> {
      Type _declaredType = it.getDeclaredType();
      return Boolean.valueOf((_declaredType instanceof ContainerType<?>));
    };
    boolean _forall_1 = IterableExtensions.forall(nonUnionTypeRefs, _function_1);
    if (_forall_1) {
      TypingStrategy _xifexpression = null;
      final Function1<TypeRef, Boolean> _function_2 = (TypeRef it) -> {
        TypingStrategy _typingStrategy = it.getTypingStrategy();
        return Boolean.valueOf((_typingStrategy == TypingStrategy.STRUCTURAL_FIELDS));
      };
      boolean _exists = IterableExtensions.exists(nonUnionTypeRefs, _function_2);
      if (_exists) {
        _xifexpression = TypingStrategy.STRUCTURAL_FIELDS;
      } else {
        TypingStrategy _xifexpression_1 = null;
        final Function1<TypeRef, Boolean> _function_3 = (TypeRef it) -> {
          TypingStrategy _typingStrategy = it.getTypingStrategy();
          return Boolean.valueOf((_typingStrategy == TypingStrategy.STRUCTURAL));
        };
        boolean _exists_1 = IterableExtensions.exists(nonUnionTypeRefs, _function_3);
        if (_exists_1) {
          _xifexpression_1 = TypingStrategy.STRUCTURAL;
        } else {
          _xifexpression_1 = TypingStrategy.DEFAULT;
        }
        _xifexpression = _xifexpression_1;
      }
      final TypingStrategy typingStrategy = _xifexpression;
      if ((typingStrategy != TypingStrategy.DEFAULT)) {
        final ParameterizedTypeRefStructural ptrs = TypeRefsFactory.eINSTANCE.createParameterizedTypeRefStructural();
        TypeRef _xifexpression_2 = null;
        TypeRef _head = IterableExtensions.<TypeRef>head(commonSuperTypesParameterized);
        if ((_head instanceof ParameterizedTypeRef)) {
          _xifexpression_2 = TypeUtils.<TypeRef>copyIfContained(IterableExtensions.<TypeRef>head(commonSuperTypesParameterized));
        } else {
          ParameterizedTypeRef _objectTypeRef = RuleEnvironmentExtensions.objectTypeRef(G);
          ParameterizedTypeRef _copyIfContained = null;
          if (_objectTypeRef!=null) {
            _copyIfContained=TypeUtils.<ParameterizedTypeRef>copyIfContained(_objectTypeRef);
          }
          _xifexpression_2 = _copyIfContained;
        }
        final TypeRef trTemplate = _xifexpression_2;
        ptrs.setDefinedTypingStrategy(typingStrategy);
        ptrs.setDeclaredType(trTemplate.getDeclaredType());
        ptrs.getTypeArgs().addAll(trTemplate.getTypeArgs());
        final TypingStrategyFilter filter = new TypingStrategyFilter(typingStrategy);
        final HashMap<String, TMember> structuralMembersByName = new HashMap<String, TMember>();
        final HashSet<String> structuralMembersWithDifferentTypeOrAlreadyContained = new HashSet<String>();
        Type _declaredType = ptrs.getDeclaredType();
        final Function1<TMember, Boolean> _function_4 = (TMember it) -> {
          return Boolean.valueOf(filter.apply(it));
        };
        final Function1<TMember, String> _function_5 = (TMember it) -> {
          return it.getName();
        };
        Iterables.<String>addAll(structuralMembersWithDifferentTypeOrAlreadyContained, 
          IterableExtensions.<TMember, String>map(IterableExtensions.<TMember>filter(this.containerTypesHelper.fromContext(RuleEnvironmentExtensions.getContextResource(G)).members(((ContainerType<?>) _declaredType)), _function_4), _function_5));
        final Function1<TypeRef, Boolean> _function_6 = (TypeRef it) -> {
          Type _declaredType_1 = it.getDeclaredType();
          Type _declaredType_2 = ptrs.getDeclaredType();
          return Boolean.valueOf((_declaredType_1 != _declaredType_2));
        };
        Iterable<? extends TypeRef> _filter = IterableExtensions.filter(nonUnionTypeRefs, _function_6);
        for (final TypeRef tr : _filter) {
          Type _declaredType_1 = tr.getDeclaredType();
          final Function1<TMember, Boolean> _function_7 = (TMember it) -> {
            return Boolean.valueOf(filter.apply(it));
          };
          Iterable<TMember> _concat = Iterables.<TMember>concat(tr.getStructuralMembers(), 
            IterableExtensions.<TMember>filter(this.containerTypesHelper.fromContext(RuleEnvironmentExtensions.getContextResource(G)).members(((ContainerType<?>) _declaredType_1)), _function_7));
          for (final TMember structMember : _concat) {
            boolean _contains = structuralMembersWithDifferentTypeOrAlreadyContained.contains(structMember.getName());
            boolean _not_1 = (!_contains);
            if (_not_1) {
              final TMember duplicate = structuralMembersByName.get(structMember.getName());
              if ((duplicate == null)) {
                structuralMembersByName.put(structMember.getName(), structMember);
              } else {
                boolean _similarMember = this.similarMember(G, duplicate, structMember);
                boolean _not_2 = (!_similarMember);
                if (_not_2) {
                  structuralMembersWithDifferentTypeOrAlreadyContained.add(structMember.getName());
                  structuralMembersByName.remove(structMember.getName());
                }
              }
            }
          }
        }
        final Function1<TMember, TMember> _function_8 = (TMember it) -> {
          return this.substituted(G, it);
        };
        Iterables.<TStructMember>addAll(ptrs.getGenStructuralMembers(), Iterables.<TStructMember>filter(IterableExtensions.<TMember, TMember>map(structuralMembersByName.values(), _function_8), TStructMember.class));
        return ptrs;
      }
    }
    TypeRef _switchResult = null;
    int _size_3 = commonSuperTypesParameterized.size();
    switch (_size_3) {
      case 0:
        throw new IllegalStateException(
          "Error processing least common super type, parameterization removed all types");
      case 1:
        _switchResult = IterableExtensions.<TypeRef>head(commonSuperTypesParameterized);
        break;
      default:
        IntersectionTypeExpression _xblockexpression = null;
        {
          final IntersectionTypeExpression intersectionTypeExpr = TypeRefsFactory.eINSTANCE.createIntersectionTypeExpression();
          final Consumer<TypeRef> _function_9 = (TypeRef it) -> {
            intersectionTypeExpr.getTypeRefs().add(TypeUtils.<TypeRef>copyIfContained(it));
          };
          commonSuperTypesParameterized.forEach(_function_9);
          _xblockexpression = intersectionTypeExpr;
        }
        _switchResult = _xblockexpression;
        break;
    }
    final TypeRef singleLCST = _switchResult;
    return singleLCST;
  }
  
  private TMember _substituted(final RuleEnvironment G, final TMember member) {
    return member;
  }
  
  private TMember _substituted(final RuleEnvironment G, final TField member) {
    boolean _isParameterized = member.getTypeRef().isParameterized();
    if (_isParameterized) {
      final TStructField subst = TypesFactory.eINSTANCE.createTStructField();
      subst.setName(member.getName());
      subst.setTypeRef(this.ts.substTypeVariables(G, member.getTypeRef()));
      TypeRef _typeRef = subst.getTypeRef();
      boolean _tripleEquals = (_typeRef == null);
      if (_tripleEquals) {
        subst.setTypeRef(RuleEnvironmentExtensions.anyTypeRef(G));
      }
      return subst;
    } else {
      return member;
    }
  }
  
  private TMember _substituted(final RuleEnvironment G, final TGetter member) {
    if (((member.getDeclaredTypeRef() != null) && member.getDeclaredTypeRef().isParameterized())) {
      final TStructGetter subst = TypesFactory.eINSTANCE.createTStructGetter();
      subst.setName(member.getName());
      subst.setDeclaredTypeRef(this.ts.substTypeVariables(G, member.getDeclaredTypeRef()));
      TypeRef _declaredTypeRef = subst.getDeclaredTypeRef();
      boolean _tripleEquals = (_declaredTypeRef == null);
      if (_tripleEquals) {
        subst.setDeclaredTypeRef(RuleEnvironmentExtensions.anyTypeRef(G));
      }
      return subst;
    } else {
      return member;
    }
  }
  
  private TMember _substituted(final RuleEnvironment G, final TSetter member) {
    if ((((member.getFpar() != null) && (member.getFpar().getTypeRef() != null)) && member.getFpar().getTypeRef().isParameterized())) {
      final TStructSetter subst = TypesFactory.eINSTANCE.createTStructSetter();
      subst.setName(member.getName());
      TypeRef tr = this.ts.substTypeVariables(G, member.getFpar().getTypeRef());
      if ((tr == null)) {
        tr = RuleEnvironmentExtensions.anyTypeRef(G);
      }
      subst.setFpar(TypesFactory.eINSTANCE.createTFormalParameter());
      TFormalParameter _fpar = subst.getFpar();
      _fpar.setName(member.getFpar().getName());
      TFormalParameter _fpar_1 = subst.getFpar();
      _fpar_1.setTypeRef(TypeUtils.<TypeRef>copyIfContained(tr));
      return subst;
    } else {
      return member;
    }
  }
  
  private TMember _substituted(final RuleEnvironment G, final TMethod member) {
    TypeRef _type = this.ts.type(G, member);
    final FunctionTypeExpression ftype = ((FunctionTypeExpression) _type);
    final TStructMethod subst = TypesFactory.eINSTANCE.createTStructMethod();
    subst.setName(member.getName());
    subst.getFpars().addAll(ftype.getFpars());
    subst.setReturnTypeRef(ftype.getReturnTypeRef());
    final Function1<TypeVariable, TypeVariable> _function = (TypeVariable it) -> {
      return TypeUtils.<TypeVariable>copyIfContained(it);
    };
    subst.getTypeVars().addAll(ListExtensions.<TypeVariable, TypeVariable>map(member.getTypeVars(), _function));
    return subst;
  }
  
  public boolean similarMember(final RuleEnvironment G, final TMember m1, final TMember m2) {
    final TypeRef t1 = this.ts.type(G, m1);
    final TypeRef t2 = this.ts.type(G, m2);
    return (this.ts.subtypeSucceeded(G, t1, t2) && this.ts.subtypeSucceeded(G, t2, t1));
  }
  
  /**
   * Removes all super types of ref from list of refs. This method is optimized for leastCommonSuperType and
   * assumes that all types in orderedRefs are ordered as returned by collecAllDeclaredSuperTypes().
   */
  private void removeAllSuperTypesOfType(final List<TypeRef> orderedRefs, final TypeRef ref, final RuleEnvironment G) {
    SuperTypesList<TypeRef> _collectAllDeclaredSuperTypesTypeargsIgnored = this._typeHelper.collectAllDeclaredSuperTypesTypeargsIgnored(ref, false);
    List<ParameterizedTypeRef> _collectAllImplicitSuperTypes = RuleEnvironmentExtensions.collectAllImplicitSuperTypes(G, ref);
    final Iterable<TypeRef> nonLeastSuperTypes = Iterables.<TypeRef>concat(_collectAllDeclaredSuperTypesTypeargsIgnored, _collectAllImplicitSuperTypes);
    for (final TypeRef nonLeastSuperType : nonLeastSuperTypes) {
      boolean _removeTypeRef = this._typeHelper.removeTypeRef(orderedRefs, nonLeastSuperType);
      boolean _not = (!_removeTypeRef);
      if (_not) {
        return;
      }
    }
  }
  
  /**
   * Search for reference in list of super types of each type ref matching given rawSuperType, using a
   * depth first search.
   * 
   * @param refs 		collection of type references, size > 1
   * @param rawSuperTypeRef a raw super type, which is a common super type of all types in refs
   */
  private Set<TypeRef> collectParameterizedSuperType(final Iterable<? extends TypeRef> refs, final Type rawSuperType, final RuleEnvironment G) {
    final TreeSet<TypeRef> result = CollectionLiterals.<TypeRef>newTreeSet(this._typeCompareHelper.getTypeRefComparator());
    for (final TypeRef typeRef : refs) {
      {
        final List<TypeRef> pathFromSuperType = this.computePathFromSuperTypeReflexive(typeRef, rawSuperType, CollectionLiterals.<Type>newHashSet());
        if ((pathFromSuperType == null)) {
          throw new IllegalStateException(((("Did not found " + rawSuperType) + " in super types of ") + typeRef));
        }
        final TypeRef concreteSuperTypeRef = IterableExtensions.<TypeRef>head(pathFromSuperType);
        TypeRef _xifexpression = null;
        boolean _containsUnboundTypeVariables = concreteSuperTypeRef.containsUnboundTypeVariables();
        boolean _not = (!_containsUnboundTypeVariables);
        if (_not) {
          _xifexpression = concreteSuperTypeRef;
        } else {
          TypeRef _xblockexpression = null;
          {
            final RuleEnvironment Gnext = RuleEnvironmentExtensions.wrap(G);
            _xblockexpression = this.genericsComputer.bindTypeVariables(Gnext, concreteSuperTypeRef);
          }
          _xifexpression = _xblockexpression;
        }
        result.add(
          TypeUtils.<TypeRef>copyIfContained(_xifexpression));
      }
    }
    return result;
  }
  
  /**
   * Returns transitive reflexive closure of <em>common</em> super types.
   * Type arguments are ignored here.
   * @return the returned list is sorted, that is, a type's super types are always AFTER the type in the list
   */
  private List<TypeRef> commonSuperTypesTypeargsIgnored(final RuleEnvironment G, final Iterable<? extends TypeRef> typeRefs) {
    final List<TypeRef> commonSuperTypes = CollectionLiterals.<TypeRef>newArrayList();
    for (final TypeRef t : typeRefs) {
      boolean _addSuperTypesToCommonList = this.addSuperTypesToCommonList(G, t, commonSuperTypes);
      boolean _not = (!_addSuperTypesToCommonList);
      if (_not) {
        return Collections.<TypeRef>emptyList();
      }
    }
    return commonSuperTypes;
  }
  
  /**
   * Returns transitive, non-reflexive closure of implicit super types. Relexive means, that in case of e.g., Object, Object is returned itself.
   */
  private void collectAllImplicitSuperTypes(final TypeRef ref, final RuleEnvironment G, final SuperTypesList<TypeRef> superTypesList) {
    superTypesList.addAll(RuleEnvironmentExtensions.collectAllImplicitSuperTypes(G, ref));
  }
  
  /**
   * Adds or intersects types in reflexive transitive closure of a given type <i>t</i> to/with given list of super types.
   * 
   * @return true, if super types have been added and if client should proceed; <br/>
   * 		  false, if no common super type can ever by found and client can stop looking for it
   */
  private boolean _addSuperTypesToCommonList(final RuleEnvironment G, final TypeRef t, final List<TypeRef> commonSuperTypes) {
    Type _declaredType = t.getDeclaredType();
    boolean _matched = false;
    if (_declaredType instanceof TClassifier) {
      _matched=true;
      final SuperTypesList<TypeRef> allDeclaredSuperTypes = SuperTypesList.<TypeRef>newSuperTypesList(this._typeCompareHelper.getTypeRefComparator());
      allDeclaredSuperTypes.add(t);
      this._typeHelper.collectAllDeclaredSuperTypesTypeargsIgnored(t, allDeclaredSuperTypes);
      this.collectAllImplicitSuperTypes(t, G, allDeclaredSuperTypes);
      this.addOrIntersectTypes(G, commonSuperTypes, allDeclaredSuperTypes);
    }
    if (!_matched) {
      if (_declaredType instanceof AnyType) {
        _matched=true;
        return false;
      }
    }
    if (!_matched) {
      if (_declaredType instanceof PrimitiveType) {
        _matched=true;
        this.addOrIntersectTypeWithAssignmentCompatibles(G, commonSuperTypes, t);
      }
    }
    if (!_matched) {
    }
    return true;
  }
  
  private boolean _addSuperTypesToCommonList(final RuleEnvironment G, final IntersectionTypeExpression t, final List<TypeRef> commonSuperTypes) {
    final SuperTypesList<TypeRef> allDeclaredSuperTypes = SuperTypesList.<TypeRef>newSuperTypesList(this._typeCompareHelper.getTypeRefComparator());
    EList<TypeRef> _typeRefs = t.getTypeRefs();
    for (final TypeRef containedRef : _typeRefs) {
      {
        allDeclaredSuperTypes.add(containedRef);
        this._typeHelper.collectAllDeclaredSuperTypesTypeargsIgnored(containedRef, allDeclaredSuperTypes);
        this.collectAllImplicitSuperTypes(containedRef, G, allDeclaredSuperTypes);
      }
    }
    this.addOrIntersectTypes(G, commonSuperTypes, allDeclaredSuperTypes);
    return true;
  }
  
  private boolean _addSuperTypesToCommonList(final RuleEnvironment G, final UnionTypeExpression t, final List<TypeRef> commonSuperTypes) {
    final SuperTypesList<TypeRef> allDeclaredSuperTypes = SuperTypesList.<TypeRef>newSuperTypesList(this._typeCompareHelper.getTypeRefComparator());
    allDeclaredSuperTypes.add(t);
    this.addOrIntersectTypes(G, commonSuperTypes, allDeclaredSuperTypes);
    return true;
  }
  
  private void addOrIntersectTypeWithAssignmentCompatibles(final RuleEnvironment G, final List<TypeRef> commonSuperTypes, final TypeRef typeRef) {
    boolean _isEmpty = commonSuperTypes.isEmpty();
    if (_isEmpty) {
      commonSuperTypes.add(typeRef);
      return;
    }
    boolean _containsByType = this._typeHelper.containsByType(commonSuperTypes, typeRef);
    if (_containsByType) {
      int _size = commonSuperTypes.size();
      boolean _equals = (_size == 1);
      if (_equals) {
        return;
      }
      commonSuperTypes.clear();
      commonSuperTypes.add(typeRef);
      return;
    }
    final Type type = typeRef.getDeclaredType();
    if ((type instanceof PrimitiveType)) {
      final int index = this._typeHelper.findTypeRefOrAssignmentCompatible(commonSuperTypes, typeRef);
      if ((index >= 0)) {
        PrimitiveType _assignmentCompatible = ((PrimitiveType)type).getAssignmentCompatible();
        boolean _tripleEquals = (_assignmentCompatible == null);
        if (_tripleEquals) {
          commonSuperTypes.clear();
          commonSuperTypes.add(typeRef);
        } else {
          int _size_1 = commonSuperTypes.size();
          boolean _notEquals = (_size_1 != 1);
          if (_notEquals) {
            final TypeRef tr = commonSuperTypes.get(index);
            commonSuperTypes.clear();
            commonSuperTypes.add(tr);
          }
        }
        return;
      }
    }
    commonSuperTypes.clear();
  }
  
  private boolean _addSuperTypesToCommonList(final RuleEnvironment G, final FunctionTypeExprOrRef f, final List<TypeRef> commonSuperTypes) {
    final SuperTypesList<TypeRef> allDeclaredSuperTypes = SuperTypesList.<TypeRef>newSuperTypesList(this._typeCompareHelper.getTypeRefComparator());
    allDeclaredSuperTypes.add(f);
    this.collectAllImplicitSuperTypes(f, G, allDeclaredSuperTypes);
    this.addOrIntersectTypes(G, commonSuperTypes, allDeclaredSuperTypes);
    return true;
  }
  
  /**
   * Returns path from super type to current ref, including the super type and the initial type.
   * @return path, or null if the raw super type has not been found (which probably is an illegal result, except in combination with intersection types)
   */
  private List<TypeRef> _computePathFromSuperTypeReflexive(final TypeRef ref, final Type rawSuperType, final Set<Type> processedTypes) {
    Type _declaredType = ref.getDeclaredType();
    boolean _equals = Objects.equal(_declaredType, rawSuperType);
    if (_equals) {
      return CollectionLiterals.<TypeRef>newArrayList(ref);
    } else {
      Iterable<? extends ParameterizedTypeRef> _declaredSuperTypes = TypeUtils.declaredSuperTypes(ref.getDeclaredType());
      for (final ParameterizedTypeRef superTypeRef : _declaredSuperTypes) {
        boolean _add = processedTypes.add(superTypeRef.getDeclaredType());
        if (_add) {
          final List<TypeRef> superPath = this.computePathFromSuperTypeReflexive(superTypeRef, rawSuperType, processedTypes);
          if ((superPath != null)) {
            superPath.add(ref);
            return superPath;
          }
        }
      }
    }
    return null;
  }
  
  /**
   * For intersection types, this returns the first path found.
   */
  private List<TypeRef> _computePathFromSuperTypeReflexive(final IntersectionTypeExpression ref, final Type rawSuperType, final Set<Type> processedTypes) {
    EList<TypeRef> _typeRefs = ref.getTypeRefs();
    for (final TypeRef typeRef : _typeRefs) {
      {
        final List<TypeRef> path = this.computePathFromSuperTypeReflexive(typeRef, rawSuperType, processedTypes);
        if ((path != null)) {
          return path;
        }
      }
    }
    return null;
  }
  
  private void addOrIntersectTypes(final RuleEnvironment G, final List<TypeRef> commonSuperTypes, final SuperTypesList<TypeRef> allDeclaredSuperTypes) {
    boolean _isEmpty = commonSuperTypes.isEmpty();
    if (_isEmpty) {
      commonSuperTypes.addAll(allDeclaredSuperTypes);
    } else {
      final FunctionTypeExprOrRef currentSuperFunction = IterableExtensions.<FunctionTypeExprOrRef>head(Iterables.<FunctionTypeExprOrRef>filter(allDeclaredSuperTypes, FunctionTypeExprOrRef.class));
      FunctionTypeExprOrRef _xifexpression = null;
      if ((currentSuperFunction != null)) {
        _xifexpression = IterableExtensions.<FunctionTypeExprOrRef>head(Iterables.<FunctionTypeExprOrRef>filter(commonSuperTypes, FunctionTypeExprOrRef.class));
      } else {
        _xifexpression = null;
      }
      final FunctionTypeExprOrRef prevCommonSuperFunction = _xifexpression;
      this._typeHelper.retainAllTypeRefs(commonSuperTypes, allDeclaredSuperTypes);
      int _compare = this._typeCompareHelper.getTypeRefComparator().compare(prevCommonSuperFunction, currentSuperFunction);
      boolean _notEquals = (_compare != 0);
      if (_notEquals) {
        final FunctionTypeExprOrRef commonSuperFunction = this.joinFunctionTypeRefs(G, currentSuperFunction, prevCommonSuperFunction);
        if ((commonSuperFunction != null)) {
          commonSuperTypes.add(commonSuperFunction);
        }
      }
    }
  }
  
  /**
   * May return null if no join is possible, e.g., in f(string) and f(number)
   */
  private FunctionTypeExprOrRef joinFunctionTypeRefs(final RuleEnvironment G, final FunctionTypeExprOrRef f1, final FunctionTypeExprOrRef f2) {
    final FunctionTypeExpression joinedFunctionTypeExpr = TypeRefsFactory.eINSTANCE.createFunctionTypeExpression();
    if (((f1.getReturnTypeRef() != null) && (f2.getReturnTypeRef() != null))) {
      joinedFunctionTypeExpr.setReturnTypeRef(
        TypeUtils.<TypeRef>copyIfContained(this._typeSystemHelper.join(G, f1.getReturnTypeRef(), f2.getReturnTypeRef())));
    }
    joinedFunctionTypeExpr.setReturnValueMarkedOptional((f1.isReturnValueOptional() || f2.isReturnValueOptional()));
    final int maxParSize = Math.max(f1.getFpars().size(), f2.getFpars().size());
    int i = 0;
    boolean varOrOpt1 = false;
    boolean varOrOpt2 = false;
    while ((i < maxParSize)) {
      {
        final TFormalParameter par1 = this.getFParSmartAndFailSafe(f1, i);
        final TFormalParameter par2 = this.getFParSmartAndFailSafe(f2, i);
        TFormalParameter fpar = null;
        if ((par1 == null)) {
          fpar = TypeUtils.<TFormalParameter>copy(par2);
        } else {
          if ((par2 == null)) {
            fpar = TypeUtils.<TFormalParameter>copy(par1);
          } else {
            boolean _isVariadicOrOptional = par1.isVariadicOrOptional();
            if (_isVariadicOrOptional) {
              varOrOpt1 = true;
            }
            boolean _isVariadicOrOptional_1 = par2.isVariadicOrOptional();
            if (_isVariadicOrOptional_1) {
              varOrOpt2 = true;
            }
            fpar = TypesFactory.eINSTANCE.createTFormalParameter();
            final TypeRef meet = this._typeSystemHelper.meet(G, par1.getTypeRef(), par2.getTypeRef());
            if ((meet == null)) {
              if ((varOrOpt1 && varOrOpt2)) {
                return joinedFunctionTypeExpr;
              } else {
                return null;
              }
            }
            final TypeRef parType = TypeUtils.<TypeRef>copyIfContained(meet);
            fpar.setTypeRef(parType);
            if ((par1.isVariadic() && par2.isVariadic())) {
              fpar.setVariadic(true);
            } else {
              if ((par1.isVariadicOrOptional() && par2.isVariadicOrOptional())) {
                fpar.setHasInitializerAssignment(true);
              }
            }
          }
        }
        joinedFunctionTypeExpr.getFpars().add(fpar);
        i = (i + 1);
      }
    }
    return joinedFunctionTypeExpr;
  }
  
  private TFormalParameter getFParSmartAndFailSafe(final FunctionTypeExprOrRef f, final int index) {
    int _size = f.getFpars().size();
    boolean _equals = (_size == 0);
    if (_equals) {
      return null;
    }
    int _size_1 = f.getFpars().size();
    boolean _lessThan = (index < _size_1);
    if (_lessThan) {
      return f.getFpars().get(index);
    }
    final TFormalParameter last = IterableExtensions.<TFormalParameter>last(f.getFpars());
    boolean _isVariadic = last.isVariadic();
    if (_isVariadic) {
      return last;
    }
    return null;
  }
  
  private TMember substituted(final RuleEnvironment G, final TMember member) {
    if (member instanceof TMethod) {
      return _substituted(G, (TMethod)member);
    } else if (member instanceof TGetter) {
      return _substituted(G, (TGetter)member);
    } else if (member instanceof TSetter) {
      return _substituted(G, (TSetter)member);
    } else if (member instanceof TField) {
      return _substituted(G, (TField)member);
    } else if (member != null) {
      return _substituted(G, member);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(G, member).toString());
    }
  }
  
  private boolean addSuperTypesToCommonList(final RuleEnvironment G, final TypeRef t, final List<TypeRef> commonSuperTypes) {
    if (t instanceof IntersectionTypeExpression) {
      return _addSuperTypesToCommonList(G, (IntersectionTypeExpression)t, commonSuperTypes);
    } else if (t instanceof UnionTypeExpression) {
      return _addSuperTypesToCommonList(G, (UnionTypeExpression)t, commonSuperTypes);
    } else if (t instanceof FunctionTypeExprOrRef) {
      return _addSuperTypesToCommonList(G, (FunctionTypeExprOrRef)t, commonSuperTypes);
    } else if (t != null) {
      return _addSuperTypesToCommonList(G, t, commonSuperTypes);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(G, t, commonSuperTypes).toString());
    }
  }
  
  private List<TypeRef> computePathFromSuperTypeReflexive(final TypeRef ref, final Type rawSuperType, final Set<Type> processedTypes) {
    if (ref instanceof IntersectionTypeExpression) {
      return _computePathFromSuperTypeReflexive((IntersectionTypeExpression)ref, rawSuperType, processedTypes);
    } else if (ref != null) {
      return _computePathFromSuperTypeReflexive(ref, rawSuperType, processedTypes);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(ref, rawSuperType, processedTypes).toString());
    }
  }
}
