/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.helper;

import com.google.common.collect.ImmutableList;
import java.util.Collection;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.utils.DependencyTraverser;
import org.eclipse.n4js.utils.DependencyTraverser.DependencyProvider;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * A {@link DependencyProvider} implementation for traversing the dependency
 * graph defined by {@link IN4JSProject} entities via {@link DependencyTraverser}.
 */
@SuppressWarnings("all")
public class SourceContainerAwareDependencyProvider implements DependencyTraverser.DependencyProvider<IN4JSProject> {
  private final boolean ignoreExternalPlainJsProjects;
  
  /**
   * Creates a new traverser instance with the given root node.
   * 
   * @param rootNode
   * 				The root node to start the traversal from.
   * @param ignoreExternalPlainJsProjects
   * 				Specifies whether dependency edges to external {@link ProjectType#PLAINJS} projects should
   * 				be excluded when traversing the dependency graph.
   * @param ignoreCycles
   * 				Specifies whether the traverser should terminate early when dependency cycles are
   * 				detected, or whether it should continue.
   */
  public SourceContainerAwareDependencyProvider(final boolean ignoreExternalPlainJsProjects) {
    this.ignoreExternalPlainJsProjects = ignoreExternalPlainJsProjects;
  }
  
  @Override
  public Collection<? extends IN4JSProject> getDependencies(final IN4JSProject p) {
    if (this.ignoreExternalPlainJsProjects) {
      final Function1<IN4JSProject, Boolean> _function = (IN4JSProject dep) -> {
        boolean _isIgnored = SourceContainerAwareDependencyProvider.isIgnored(dep);
        return Boolean.valueOf((!_isIgnored));
      };
      return ImmutableList.<IN4JSProject>copyOf(IterableExtensions.filter(p.getAllDirectDependencies(), _function));
    } else {
      return p.getAllDirectDependencies();
    }
  }
  
  private static boolean isIgnored(final IN4JSProject project) {
    return (project.isExternal() && ((project.getProjectType() == ProjectType.VALIDATION) || 
      (project.getProjectType() == ProjectType.PLAINJS)));
  }
}
