/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.common.base.Predicate;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.ExpressionStatement;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.IndexedAccessExpression;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.n4JS.NewExpression;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.ReturnStatement;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.SuperLiteral;
import org.eclipse.n4js.n4JS.ThisLiteral;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;

/**
 * Validate use of super keyword.
 * 
 * N4JS spec  6.2.1 and 4.18.2
 */
@SuppressWarnings("all")
public class N4JSSuperValidator extends AbstractN4JSDeclarativeValidator {
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  @Check
  public Boolean checkSuper(final SuperLiteral superLiteral) {
    boolean _xblockexpression = false;
    {
      final EObject container = superLiteral.eContainer();
      final N4MemberDeclaration containingMemberDecl = EcoreUtil2.<N4MemberDeclaration>getContainerOfType(container, N4MemberDeclaration.class);
      boolean _xifexpression = false;
      boolean _isSuperMemberAccess = superLiteral.isSuperMemberAccess();
      if (_isSuperMemberAccess) {
        _xifexpression = this.internalCheckSuperMemberAccess(superLiteral, containingMemberDecl);
      } else {
        boolean _xifexpression_1 = false;
        boolean _isSuperConstructorAccess = superLiteral.isSuperConstructorAccess();
        if (_isSuperConstructorAccess) {
          _xifexpression_1 = this.internalCheckSuperConstructorAccess(superLiteral, containingMemberDecl, 
            ((ParameterizedCallExpression) container));
        } else {
          if ((container instanceof NewExpression)) {
            final String message = IssueCodes.getMessageForKEY_SUP_NEW_NOT_SUPPORTED();
            this.addIssue(message, superLiteral.eContainer(), superLiteral.eContainmentFeature(), IssueCodes.KEY_SUP_NEW_NOT_SUPPORTED);
          } else {
            final String message_1 = IssueCodes.getMessageForKEY_SUP_INVALID_USAGE();
            this.addIssue(message_1, superLiteral.eContainer(), superLiteral.eContainmentFeature(), IssueCodes.KEY_SUP_INVALID_USAGE);
          }
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return Boolean.valueOf(_xblockexpression);
  }
  
  /**
   * N4JS spec 4.18.2 and 6.2.1
   * Constraints 97
   */
  private boolean internalCheckSuperConstructorAccess(final SuperLiteral superLiteral, final N4MemberDeclaration containingMemberDecl, final ParameterizedCallExpression cexpr) {
    boolean _xifexpression = false;
    boolean _holdsExpressionOfExpressionStmt = this.holdsExpressionOfExpressionStmt(superLiteral, cexpr);
    if (_holdsExpressionOfExpressionStmt) {
      boolean _xblockexpression = false;
      {
        EObject _eContainer = cexpr.eContainer();
        final ExpressionStatement exprStmt = ((ExpressionStatement) _eContainer);
        _xblockexpression = (this.holdsDirectlyContainedInConstructor(superLiteral, exprStmt, containingMemberDecl) && 
          this.holdsNoThisOrReturnBefore(superLiteral, exprStmt, containingMemberDecl));
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  /**
   * Constraint 97.2
   */
  private boolean holdsExpressionOfExpressionStmt(final SuperLiteral superLiteral, final ParameterizedCallExpression cexpr) {
    EObject _eContainer = cexpr.eContainer();
    if ((_eContainer instanceof ExpressionStatement)) {
      return true;
    }
    this.addIssue(IssueCodes.getMessageForKEY_SUP_CTOR_EXPRSTMT(), cexpr, superLiteral.eContainmentFeature(), IssueCodes.KEY_SUP_CTOR_EXPRSTMT);
    return false;
  }
  
  /**
   * N4JS constraints 97.3
   */
  private boolean holdsDirectlyContainedInConstructor(final SuperLiteral superLiteral, final ExpressionStatement exprStmt, final N4MemberDeclaration containingMemberDecl) {
    final EObject block = exprStmt.eContainer();
    boolean _not = (!((containingMemberDecl instanceof N4MethodDeclaration) && Objects.equal(containingMemberDecl.getName(), "constructor")));
    if (_not) {
      return false;
    }
    boolean _not_1 = (!((block instanceof Block) && (block.eContainer() == containingMemberDecl)));
    if (_not_1) {
      EObject _xifexpression = null;
      if ((block instanceof Block)) {
        _xifexpression = ((Block)block).eContainer();
      } else {
        _xifexpression = block;
      }
      final String msg = IssueCodes.getMessageForKEY_SUP_CTOR_NESTED(
        this.validatorMessageHelper.descriptionWithLine(_xifexpression));
      this.addIssue(msg, superLiteral.eContainer(), superLiteral.eContainmentFeature(), IssueCodes.KEY_SUP_CTOR_NESTED);
      return false;
    }
    return true;
  }
  
  /**
   * N4JS constraints 97.4
   * @see IDEBUG-147
   */
  private boolean holdsNoThisOrReturnBefore(final SuperLiteral superLiteral, final ExpressionStatement exprStmt, final N4MemberDeclaration containingMemberDecl) {
    EList<Statement> _statements = this.getBody(containingMemberDecl).getStatements();
    for (final Statement stmt : _statements) {
      {
        if ((stmt == exprStmt)) {
          return true;
        }
        if ((stmt instanceof ReturnStatement)) {
          final String msg = IssueCodes.getMessageForKEY_SUP_CTOR_INVALID_EXPR_BEFORE(this.validatorMessageHelper.descriptionWithLine(stmt));
          this.addIssue(msg, superLiteral.eContainer(), superLiteral.eContainmentFeature(), 
            IssueCodes.KEY_SUP_CTOR_INVALID_EXPR_BEFORE);
          return false;
        }
        final Predicate<EObject> _function = (EObject it) -> {
          return (!(it instanceof FunctionOrFieldAccessor));
        };
        final Function1<EObject, Boolean> _function_1 = (EObject it) -> {
          return Boolean.valueOf(((it instanceof ThisLiteral) || (it instanceof ReturnStatement)));
        };
        final EObject thisKeyword = IteratorExtensions.<EObject>findFirst(EcoreUtilN4.getAllContentsFiltered(stmt, _function), _function_1);
        if ((thisKeyword != null)) {
          final String msg_1 = IssueCodes.getMessageForKEY_SUP_CTOR_INVALID_EXPR_BEFORE(this.validatorMessageHelper.descriptionWithLine(thisKeyword));
          this.addIssue(msg_1, superLiteral.eContainer(), superLiteral.eContainmentFeature(), 
            IssueCodes.KEY_SUP_CTOR_INVALID_EXPR_BEFORE);
          return false;
        }
      }
    }
    final String msg = IssueCodes.getMessageForKEY_SUP_CTOR_NESTED(this.validatorMessageHelper.descriptionWithLine(exprStmt.eContainer()));
    this.addIssue(msg, superLiteral.eContainer(), superLiteral.eContainmentFeature(), IssueCodes.KEY_SUP_CTOR_NESTED);
    return false;
  }
  
  /**
   * N4JS spec 6.2.1
   * Constraints 98
   */
  private boolean internalCheckSuperMemberAccess(final SuperLiteral superLiteral, final N4MemberDeclaration containingMemberDecl) {
    return (((this.holdsSuperCallInMethodOrFieldAccessor(superLiteral, containingMemberDecl) && this.holdsSuperCallNotNested(superLiteral, containingMemberDecl)) && this.holdsSuperIsReceiverOfCall(superLiteral)) && this.holdsSuperMethodIsConcrete(superLiteral));
  }
  
  /**
   * Makes sure that super method calls are only allowed for non-abstract super methods.
   */
  private boolean holdsSuperMethodIsConcrete(final SuperLiteral superLiteral) {
    final EObject literalContainer = superLiteral.eContainer();
    if (((literalContainer instanceof ParameterizedPropertyAccessExpression) && 
      (((ParameterizedPropertyAccessExpression) literalContainer).getProperty() instanceof TMethod))) {
      IdentifiableElement _property = ((ParameterizedPropertyAccessExpression) literalContainer).getProperty();
      final TMethod method = ((TMethod) _property);
      boolean _isAbstract = method.isAbstract();
      if (_isAbstract) {
        this.addIssue(IssueCodes.getMessageForCLF_CANNOT_CALL_ABSTRACT_SUPER_METHOD(), literalContainer, 
          N4JSPackage.Literals.PARAMETERIZED_PROPERTY_ACCESS_EXPRESSION__PROPERTY, 
          IssueCodes.CLF_CANNOT_CALL_ABSTRACT_SUPER_METHOD);
        return false;
      }
    }
    return true;
  }
  
  /**
   * Constraint 98.1, special cases
   */
  private boolean holdsSuperIsReceiverOfCall(final SuperLiteral superLiteral) {
    if (((superLiteral.eContainer() instanceof ParameterizedPropertyAccessExpression) && 
      (((ParameterizedPropertyAccessExpression) superLiteral.eContainer()).getProperty() instanceof TField))) {
      EObject _eContainer = superLiteral.eContainer();
      this.addIssue(IssueCodes.getMessageForKEY_SUP_ACCESS_FIELD(), ((ParameterizedPropertyAccessExpression) _eContainer), 
        N4JSPackage.Literals.PARAMETERIZED_PROPERTY_ACCESS_EXPRESSION__PROPERTY, IssueCodes.KEY_SUP_ACCESS_FIELD);
      return false;
    }
    EObject _eContainer_1 = superLiteral.eContainer();
    if ((_eContainer_1 instanceof IndexedAccessExpression)) {
      EObject _eContainer_2 = superLiteral.eContainer();
      this.addIssue(IssueCodes.getMessageForKEY_SUP_CALL_NO_INDEXACCESS(), ((IndexedAccessExpression) _eContainer_2), 
        N4JSPackage.eINSTANCE.getIndexedAccessExpression_Target(), IssueCodes.KEY_SUP_CALL_NO_INDEXACCESS);
      return false;
    }
    return true;
  }
  
  /**
   * Constraint 98.3
   */
  private boolean holdsSuperCallNotNested(final SuperLiteral superLiteral, final N4MemberDeclaration containingMemberDecl) {
    FunctionOrFieldAccessor _containerOfType = EcoreUtil2.<FunctionOrFieldAccessor>getContainerOfType(superLiteral.eContainer(), FunctionOrFieldAccessor.class);
    boolean _tripleNotEquals = (_containerOfType != containingMemberDecl);
    if (_tripleNotEquals) {
      this.addIssue(IssueCodes.getMessageForKEY_SUP_NESTED(), superLiteral.eContainer(), superLiteral.eContainmentFeature(), IssueCodes.KEY_SUP_NESTED);
      return false;
    }
    return true;
  }
  
  /**
   * Constraint 98.2
   */
  public boolean holdsSuperCallInMethodOrFieldAccessor(final SuperLiteral superLiteral, final N4MemberDeclaration containingMemberDecl) {
    if ((!(((containingMemberDecl instanceof N4MethodDeclaration) || 
      (containingMemberDecl instanceof N4GetterDeclaration)) || (containingMemberDecl instanceof N4SetterDeclaration)))) {
      return false;
    }
    EObject _eContainer = null;
    if (containingMemberDecl!=null) {
      _eContainer=containingMemberDecl.eContainer();
    }
    if ((_eContainer instanceof N4InterfaceDeclaration)) {
      this.addIssue(IssueCodes.getMessageForKEY_SUP_ACCESS_INVALID_LOC_INTERFACE(), superLiteral.eContainer(), superLiteral.eContainmentFeature(), 
        IssueCodes.KEY_SUP_ACCESS_INVALID_LOC_INTERFACE);
      return false;
    }
    return true;
  }
}
