/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.N4ClassExpression;
import org.eclipse.n4js.n4JS.NewExpression;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.ParenExpression;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.postprocessing.ASTMetaInfoUtils;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.validators.N4JSExpressionValidator;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure4;

/**
 * Validations for variable declarations and variables.
 */
@SuppressWarnings("all")
public class N4JSVariableValidator extends AbstractN4JSDeclarativeValidator {
  /**
   * NEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  @Check
  public void checkVariableDeclaration(final VariableDeclaration varDecl) {
    Expression _expression = varDecl.getExpression();
    boolean _tripleNotEquals = (_expression != null);
    if (_tripleNotEquals) {
      final List<IdentifierRef> refs = N4JSVariableValidator.collectIdentifierRefsTo(varDecl.getExpression(), varDecl, CollectionLiterals.<IdentifierRef>newArrayList());
      for (final IdentifierRef currRef : refs) {
        {
          final String message = IssueCodes.getMessageForAST_VAR_DECL_RECURSIVE(varDecl.getName());
          this.addIssue(message, currRef, null, IssueCodes.AST_VAR_DECL_RECURSIVE);
        }
      }
      final Expression expression = varDecl.getExpression();
      if ((expression instanceof IdentifierRef)) {
        final Procedure4<String, EObject, EStructuralFeature, String> _function = (String message, EObject source, EStructuralFeature feature, String issueCode) -> {
          this.addIssue(message, source, feature, issueCode);
        };
        N4JSExpressionValidator.internalCheckNameRestrictionInMethodBodies(((IdentifierRef)expression), _function);
      }
    }
  }
  
  @Check
  public void checkUnusedVariables(final VariableDeclaration varDecl) {
    if ((varDecl instanceof ExportedVariableDeclaration)) {
      return;
    }
    boolean _isEmpty = ASTMetaInfoUtils.getLocalVariableReferences(varDecl).isEmpty();
    if (_isEmpty) {
      final String message = IssueCodes.getMessageForCFG_LOCAL_VAR_UNUSED(varDecl.getName());
      this.addIssue(message, varDecl, this.findNameFeature(varDecl).getValue(), IssueCodes.CFG_LOCAL_VAR_UNUSED);
    }
  }
  
  private static List<IdentifierRef> collectIdentifierRefsTo(final EObject astNode, final VariableDeclaration varDecl, final List<IdentifierRef> result) {
    if (((astNode instanceof N4ClassExpression) && (!N4JSVariableValidator.isInstantiatedOrCalled(astNode)))) {
      return result;
    }
    if (((astNode instanceof FunctionExpression) && (!N4JSVariableValidator.isInstantiatedOrCalled(astNode)))) {
      return result;
    }
    IdentifiableElement _xifexpression = null;
    if ((varDecl instanceof ExportedVariableDeclaration)) {
      _xifexpression = ((ExportedVariableDeclaration)varDecl).getDefinedVariable();
    } else {
      _xifexpression = varDecl;
    }
    final IdentifiableElement targetForReferencesToVarDecl = _xifexpression;
    if ((astNode instanceof IdentifierRef)) {
      IdentifiableElement _id = ((IdentifierRef)astNode).getId();
      boolean _tripleEquals = (_id == targetForReferencesToVarDecl);
      if (_tripleEquals) {
        result.add(((IdentifierRef)astNode));
      }
    }
    EList<EObject> _eContents = astNode.eContents();
    for (final EObject child : _eContents) {
      N4JSVariableValidator.collectIdentifierRefsTo(child, varDecl, result);
    }
    return result;
  }
  
  public static boolean containsIdentifierRefsTo(final EObject astNode, final VariableDeclaration varDecl) {
    boolean _isEmpty = N4JSVariableValidator.collectIdentifierRefsTo(astNode, varDecl, CollectionLiterals.<IdentifierRef>newArrayList()).isEmpty();
    return (!_isEmpty);
  }
  
  /**
   * Tells if given astNode is an expression serving as target to a new or call expression. This provides <b>only a
   * heuristic</b>; might produce false negatives (but no false positives).
   */
  private static boolean isInstantiatedOrCalled(final EObject astNode) {
    EObject curr = astNode;
    while ((curr.eContainer() instanceof ParenExpression)) {
      curr = curr.eContainer();
    }
    final EObject parent = curr.eContainer();
    boolean _switchResult = false;
    boolean _matched = false;
    if (parent instanceof NewExpression) {
      _matched=true;
      Expression _callee = ((NewExpression)parent).getCallee();
      _switchResult = (_callee == curr);
    }
    if (!_matched) {
      if (parent instanceof ParameterizedCallExpression) {
        _matched=true;
        Expression _target = ((ParameterizedCallExpression)parent).getTarget();
        _switchResult = (_target == curr);
      }
    }
    if (!_matched) {
      _switchResult = false;
    }
    return _switchResult;
  }
}
