/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.tests.codegen;

import java.util.List;
import java.util.Objects;
import org.eclipse.n4js.tests.codegen.Fragment;
import org.eclipse.n4js.tests.codegen.Member;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;

/**
 * Abstract base class for classifiers.
 */
@SuppressWarnings("all")
public abstract class Classifier<T extends Classifier<T>> extends Fragment<T> {
  /**
   * Possible visibility modifiers for classifiers.
   */
  public enum Visibility {
    PRIVATE,
    
    PROJECT,
    
    PUBLIC_INTERNAL,
    
    PUBLIC;
  }
  
  /**
   * Extension methods for the {@link Visibility} enumeration.
   */
  public static class VisibilityExtensions {
    /**
     * Builds a classifier name from the given name and visibility by appending an appropriate string
     * to the given name.
     * 
     * @param visibility the visibility value
     * @param the classifier name prefix
     * 
     * @return the newly created classifier name
     */
    public static String makeName(final Classifier.Visibility visibility, final String classifierName) {
      String _nameExtension = Classifier.VisibilityExtensions.getNameExtension(visibility);
      return (classifierName + _nameExtension);
    }
    
    /**
     * Returns an appropriate classifier name extension depending on the given visibility.
     * 
     * @param visibility the visibility value
     * 
     * @return the name extension
     */
    public static String getNameExtension(final Classifier.Visibility visibility) {
      String _switchResult = null;
      if (visibility != null) {
        switch (visibility) {
          case PRIVATE:
            _switchResult = "_private";
            break;
          case PROJECT:
            _switchResult = "_project";
            break;
          case PUBLIC_INTERNAL:
            _switchResult = "_public_internal";
            break;
          case PUBLIC:
            _switchResult = "_public";
            break;
          default:
            break;
        }
      }
      return _switchResult;
    }
    
    /**
     * Builds an appropriate code fragment for the given classifier visibility.
     * 
     * @param visibility the visibility value
     * 
     * @return the code fragment
     */
    public static String generate(final Classifier.Visibility visibility) {
      String _switchResult = null;
      if (visibility != null) {
        switch (visibility) {
          case PRIVATE:
            _switchResult = "/* private */";
            break;
          case PROJECT:
            _switchResult = "export project";
            break;
          case PUBLIC_INTERNAL:
            _switchResult = "export @Internal public";
            break;
          case PUBLIC:
            _switchResult = "export public";
            break;
          default:
            break;
        }
      }
      return _switchResult;
    }
  }
  
  private Classifier.Visibility visibility = Classifier.Visibility.PRIVATE;
  
  private String name;
  
  private List<Member<?>> members;
  
  /**
   * Creates a new classifier instance with the given name.
   * 
   * @param name the name of the new classifier
   */
  protected Classifier(final String name) {
    this.name = Objects.<String>requireNonNull(name);
  }
  
  /**
   * Returns the name of this classifier.
   * 
   * @return the name of this classifier
   */
  public String getName() {
    return this.name;
  }
  
  /**
   * Set the visibility to <code>project</code>.
   */
  public T makeProjectVisible() {
    return this.setVisibility(Classifier.Visibility.PROJECT);
  }
  
  /**
   * Set the visibility to <code>public @Internal</code>.
   */
  public T makePublicInternal() {
    return this.setVisibility(Classifier.Visibility.PUBLIC_INTERNAL);
  }
  
  /**
   * Set the visibility to <code>public</code>.
   */
  public T makePublic() {
    return this.setVisibility(Classifier.Visibility.PUBLIC);
  }
  
  /**
   * Set the visibility.
   * 
   * @param visibility the visibility to set
   */
  public T setVisibility(final Classifier.Visibility visibility) {
    this.visibility = visibility;
    return ((T) this);
  }
  
  /**
   * Add the given member to this builder.
   * 
   * @param member the member to add
   */
  public T addMember(final Member<?> member) {
    if ((this.members == null)) {
      this.members = CollectionLiterals.<Member<?>>newLinkedList();
    }
    this.members.add(Objects.<Member<?>>requireNonNull(member));
    return ((T) this);
  }
  
  @Override
  public CharSequence generate() {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _generateVisibility = this.generateVisibility();
    _builder.append(_generateVisibility);
    String _generateAbstract = this.generateAbstract();
    _builder.append(_generateAbstract);
    CharSequence _generateType = this.generateType();
    _builder.append(_generateType);
    _builder.append(this.name);
    CharSequence _generateTypeRelations = this.generateTypeRelations();
    _builder.append(_generateTypeRelations);
    _builder.append(" ");
    {
      boolean _hasMembers = this.hasMembers();
      boolean _not = (!_hasMembers);
      if (_not) {
        _builder.append("{}");
      } else {
        _builder.append("{");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        CharSequence _generateMembers = this.generateMembers();
        _builder.append(_generateMembers, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("}");
        _builder.newLine();
      }
    }
    return _builder;
  }
  
  /**
   * Generate an appropriate code fragment for this classifier's visibility.
   * 
   * @return the generated visibility code fragment
   */
  protected CharSequence generateVisibility() {
    StringConcatenation _builder = new StringConcatenation();
    String _generate = Classifier.VisibilityExtensions.generate(this.visibility);
    _builder.append(_generate);
    _builder.append(" ");
    return _builder;
  }
  
  /**
   * Generate the code fragments for each of this classifier's members.
   * 
   * @return the generated member code fragment
   */
  protected CharSequence generateMembers() {
    StringConcatenation _builder = new StringConcatenation();
    {
      for(final Member<?> m : this.members) {
        CharSequence _generate = m.generate();
        _builder.append(_generate);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }
  
  /**
   * Generates a code fragment for the actual type of this classifier.
   * 
   * @return the generated code fragment
   */
  protected abstract CharSequence generateType();
  
  /**
   * Generates a code fragment for the type relations of this classifier, e.g.
   * its base types or implemented interfaces.
   */
  protected abstract CharSequence generateTypeRelations();
  
  private boolean hasMembers() {
    return ((this.members != null) && (!this.members.isEmpty()));
  }
  
  @Override
  public String toString() {
    return this.generate().toString();
  }
}
