/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.assistants;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.AnnotationList;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.TypeDefiningElement;
import org.eclipse.n4js.transpiler.TransformationAssistant;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.im.ParameterizedPropertyAccessExpression_IM;
import org.eclipse.n4js.transpiler.im.ParameterizedTypeRef_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.transpiler.utils.ConcreteMembersOrderedForTranspiler;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class TypeAssistant extends TransformationAssistant {
  @Inject
  private ContainerTypesHelper containerTypesHelper;
  
  /**
   * Some assertions related to {@link N4ClassifierDeclaration}s that apply to several transformations and are
   * therefore factored out into this helper method.
   */
  public void assertClassifierPreConditions() {
    final List<N4ClassifierDeclaration> allClassifierDecls = this.<N4ClassifierDeclaration>collectNodes(this.getState().im, N4ClassifierDeclaration.class, false);
    final Function1<N4ClassifierDeclaration, Boolean> _function = (N4ClassifierDeclaration it) -> {
      TClassifier _originalDefinedType = this.getState().info.getOriginalDefinedType(it);
      return Boolean.valueOf((_originalDefinedType != null));
    };
    this.assertTrue("all classifier declarations must have an original defined type", 
      IterableExtensions.<N4ClassifierDeclaration>forall(allClassifierDecls, _function));
    final Function1<N4ClassDeclaration, ParameterizedTypeRef> _function_1 = (N4ClassDeclaration it) -> {
      return it.getSuperClassRef();
    };
    final Function1<ParameterizedTypeRef_IM, Boolean> _function_2 = (ParameterizedTypeRef_IM it) -> {
      final IdentifiableElement originalDeclType = it.getOriginalTargetOfRewiredTarget();
      return Boolean.valueOf(((originalDeclType instanceof TClass) || (originalDeclType instanceof TObjectPrototype)));
    };
    this.assertTrue("all class declarations must have a superClassRef pointing to a STE with an original target (if non-null)", 
      IterableExtensions.<ParameterizedTypeRef_IM>forall(Iterables.<ParameterizedTypeRef_IM>filter(IterableExtensions.<ParameterizedTypeRef>filterNull(IterableExtensions.<N4ClassDeclaration, ParameterizedTypeRef>map(Iterables.<N4ClassDeclaration>filter(allClassifierDecls, N4ClassDeclaration.class), _function_1)), ParameterizedTypeRef_IM.class), _function_2));
    final Function1<N4ClassifierDeclaration, Iterable<ParameterizedTypeRef>> _function_3 = (N4ClassifierDeclaration it) -> {
      return it.getImplementedOrExtendedInterfaceRefs();
    };
    final Function1<ParameterizedTypeRef_IM, Boolean> _function_4 = (ParameterizedTypeRef_IM it) -> {
      final IdentifiableElement originalDeclType = it.getOriginalTargetOfRewiredTarget();
      return Boolean.valueOf((originalDeclType instanceof TInterface));
    };
    this.assertTrue("all classifier declarations must have all implementedOrExtendedInterfaceRefs pointing to a STE with an original target", 
      IterableExtensions.<ParameterizedTypeRef_IM>forall(Iterables.<ParameterizedTypeRef_IM>filter(Iterables.<ParameterizedTypeRef>concat(ListExtensions.<N4ClassifierDeclaration, Iterable<ParameterizedTypeRef>>map(allClassifierDecls, _function_3)), ParameterizedTypeRef_IM.class), _function_4));
  }
  
  /**
   * Returns symbol table entry for super class of given class declaration.
   */
  public SymbolTableEntryOriginal getSuperClassSTE(final N4ClassDeclaration classDecl) {
    final ParameterizedTypeRef superClassRef = classDecl.getSuperClassRef();
    if ((superClassRef instanceof ParameterizedTypeRef_IM)) {
      final SymbolTableEntry superClassSTE = ((ParameterizedTypeRef_IM)superClassRef).getDeclaredType_IM();
      if ((superClassSTE instanceof SymbolTableEntryOriginal)) {
        return ((SymbolTableEntryOriginal)superClassSTE);
      }
    }
    return this.getSymbolTableEntryOriginal(RuleEnvironmentExtensions.n4ObjectType(this.getState().G), true);
  }
  
  /**
   * Returns super interfaces (i.e. implemented or extended interfaces) of given classifier.
   */
  public List<SymbolTableEntryOriginal> getSuperInterfacesSTEs(final N4ClassifierDeclaration classifierDecl) {
    EList<ParameterizedTypeRef> _switchResult = null;
    boolean _matched = false;
    if (classifierDecl instanceof N4ClassDeclaration) {
      _matched=true;
      _switchResult = ((N4ClassDeclaration)classifierDecl).getImplementedInterfaceRefs();
    }
    if (!_matched) {
      if (classifierDecl instanceof N4InterfaceDeclaration) {
        _matched=true;
        _switchResult = ((N4InterfaceDeclaration)classifierDecl).getSuperInterfaceRefs();
      }
    }
    if (!_matched) {
      String _name = null;
      if (classifierDecl!=null) {
        _name=classifierDecl.getName();
      }
      String _plus = ("unsupported subclass of N4ClassifierDeclaration: " + _name);
      throw new IllegalStateException(_plus);
    }
    final EList<ParameterizedTypeRef> superIfcRefs = _switchResult;
    final Function1<ParameterizedTypeRef, SymbolTableEntry> _function = (ParameterizedTypeRef superIfcRef) -> {
      SymbolTableEntry _xifexpression = null;
      if ((superIfcRef instanceof ParameterizedTypeRef_IM)) {
        _xifexpression = ((ParameterizedTypeRef_IM)superIfcRef).getDeclaredType_IM();
      }
      return _xifexpression;
    };
    return IterableExtensions.<SymbolTableEntryOriginal>toList(Iterables.<SymbolTableEntryOriginal>filter(ListExtensions.<ParameterizedTypeRef, SymbolTableEntry>map(superIfcRefs, _function), SymbolTableEntryOriginal.class));
  }
  
  /**
   * Tells if the given classifier is declared on top level.
   */
  public boolean isTopLevel(final TypeDefiningElement typeDef) {
    EObject parent = typeDef.eContainer();
    while (((parent instanceof ExportDeclaration) || (parent instanceof AnnotationList))) {
      parent = parent.eContainer();
    }
    return (parent instanceof Script);
  }
  
  /**
   * For a member name that represents a symbol, such as <code>#iterator</code>, this method will return a property
   * access expression that will evaluate to the corresponding symbol, e.g. <code>Symbol.iterator</code>.
   */
  public ParameterizedPropertyAccessExpression_IM getMemberNameAsSymbol(final String memberName) {
    boolean _startsWith = memberName.startsWith(N4JSLanguageUtils.SYMBOL_IDENTIFIER_PREFIX);
    boolean _not = (!_startsWith);
    if (_not) {
      throw new IllegalArgumentException("given member name does not denote a symbol");
    }
    return TranspilerBuilderBlocks._PropertyAccessExpr(
      this.getSymbolTableEntryOriginal(RuleEnvironmentExtensions.symbolObjectType(this.getState().G), true), 
      this.getSymbolTableEntryInternal(memberName.substring(1), true));
  }
  
  /**
   * Returns an instance of {@link ConcreteMembersOrderedForTranspiler} for the given classifier, using a cached
   * instance if available.
   */
  public ConcreteMembersOrderedForTranspiler getOrCreateCMOFT(final TClassifier classifier) {
    final ConcreteMembersOrderedForTranspiler cachedCMOFT = this.getState().info.getCachedCMOFT(classifier);
    if ((cachedCMOFT != null)) {
      return cachedCMOFT;
    } else {
      final ConcreteMembersOrderedForTranspiler newCMOFT = ConcreteMembersOrderedForTranspiler.create(
        this.containerTypesHelper, classifier, this.getState().resource.getScript());
      this.getState().info.cacheCMOFT(classifier, newCMOFT);
      return newCMOFT;
    }
  }
}
