/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.organize.imports;

import com.google.common.base.Objects;
import org.eclipse.n4js.organize.imports.ImportProvidedElement;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * Enhanced information, not deducible from the IEObjectDescription:
 * <ul>
 *  <li>how to import {@link #exportedAsDefault}</li>
 *  <li>name used in script
 * </ul>
 * Mainly it provides information used in cases of default exports,
 * where the IEObjectdescription must be handled differently.
 * 
 * Also overrides {@code equals()} and {@code hashCode()} to enable set-based operations.
 * 
 * It differs from {@link ImportProvidedElement} as this light-weight structure only describes potential imports,
 * while instances of {@link ImportProvidedElement} are objects for tracking usage of already imported elements.
 */
@Data
@SuppressWarnings("all")
public final class ImportableObject {
  private final String name;
  
  private final TExportableElement te;
  
  private final boolean exportedAsDefault;
  
  private final boolean asNamespace;
  
  public ImportableObject(final String name, final TExportableElement te, final boolean asDefault) {
    this(name, te, asDefault, false);
  }
  
  public ImportableObject(final String name, final TExportableElement te, final boolean asDefault, final boolean asNamespace) {
    this.name = name;
    this.te = te;
    this.exportedAsDefault = asDefault;
    this.asNamespace = asNamespace;
  }
  
  @Override
  public boolean equals(final Object o) {
    if ((o instanceof ImportableObject)) {
      return ((Objects.equal(this.name, ((ImportableObject)o).name) && (this.exportedAsDefault == ((ImportableObject)o).exportedAsDefault)) && 
        this.te.equals(((ImportableObject)o).te));
    } else {
      return false;
    }
  }
  
  @Override
  public int hashCode() {
    int _hashCode = this.name.hashCode();
    int _xifexpression = (int) 0;
    if (this.exportedAsDefault) {
      _xifexpression = 13;
    } else {
      _xifexpression = 7;
    }
    int _plus = (_hashCode + _xifexpression);
    int _hashCode_1 = this.te.hashCode();
    return (_plus + _hashCode_1);
  }
  
  @Override
  @Pure
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("name", this.name);
    b.add("te", this.te);
    b.add("exportedAsDefault", this.exportedAsDefault);
    b.add("asNamespace", this.asNamespace);
    return b.toString();
  }
  
  @Pure
  public String getName() {
    return this.name;
  }
  
  @Pure
  public TExportableElement getTe() {
    return this.te;
  }
  
  @Pure
  public boolean isExportedAsDefault() {
    return this.exportedAsDefault;
  }
  
  @Pure
  public boolean isAsNamespace() {
    return this.asNamespace;
  }
}
