/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.quickfix;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.binaries.IllegalBinaryStateException;
import org.eclipse.n4js.external.LibraryManager;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.ModifiableElement;
import org.eclipse.n4js.n4JS.ModifierUtils;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4FieldAccessor;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.N4Modifier;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.PropertyNameOwner;
import org.eclipse.n4js.projectDescription.ProjectDependency;
import org.eclipse.n4js.projectDescription.ProjectReference;
import org.eclipse.n4js.semver.Semver.NPMVersionRequirement;
import org.eclipse.n4js.semver.SemverUtils;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.SyntaxRelatedTElement;
import org.eclipse.n4js.ts.types.TAnnotableElement;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ui.binaries.IllegalBinaryStateDialog;
import org.eclipse.n4js.ui.changes.ChangeProvider;
import org.eclipse.n4js.ui.changes.IChange;
import org.eclipse.n4js.ui.changes.SemanticChangeProvider;
import org.eclipse.n4js.ui.internal.N4JSActivator;
import org.eclipse.n4js.ui.labeling.helper.ImageNames;
import org.eclipse.n4js.ui.quickfix.AbstractN4JSQuickfixProvider;
import org.eclipse.n4js.ui.quickfix.N4Modification;
import org.eclipse.n4js.ui.quickfix.QuickfixUtil;
import org.eclipse.n4js.ui.quickfix.TopLevelVisibilityFixProvider;
import org.eclipse.n4js.ui.utils.ImportUtil;
import org.eclipse.n4js.ui.utils.UIUtils;
import org.eclipse.n4js.utils.StatusHelper;
import org.eclipse.n4js.utils.StatusUtils;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.IssueUserDataKeys;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.xtext.diagnostics.Diagnostic;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.ui.editor.model.edit.IModificationContext;
import org.eclipse.xtext.ui.editor.quickfix.Fix;
import org.eclipse.xtext.ui.editor.quickfix.IssueResolutionAcceptor;
import org.eclipse.xtext.validation.Issue;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * N4JS quick fixes.
 * 
 * see http://www.eclipse.org/Xtext/documentation.html#quickfixes
 */
@SuppressWarnings("all")
public class N4JSQuickfixProvider extends AbstractN4JSQuickfixProvider {
  @Inject
  @Extension
  private ImportUtil _importUtil;
  
  @Inject
  @Extension
  private StatusHelper _statusHelper;
  
  @Inject
  @Extension
  private QuickfixUtil.IssueUserDataKeysExtension _issueUserDataKeysExtension;
  
  @Inject
  private TopLevelVisibilityFixProvider topLevelVisibilityFixProvider;
  
  @Inject
  @Extension
  private SemanticChangeProvider _semanticChangeProvider;
  
  @Inject
  protected JavaScriptVariantHelper jsVariantHelper;
  
  @Inject
  private LibraryManager libraryManager;
  
  /**
   * Retrieve annotation constants from AnnotationDefinition
   */
  private final static String INTERNAL_ANNOTATION = AnnotationDefinition.INTERNAL.name;
  
  private final static String OVERRIDE_ANNOTATION = AnnotationDefinition.OVERRIDE.name;
  
  private final static String FINAL_ANNOTATION = AnnotationDefinition.FINAL.name;
  
  public void someSimpleQuickFixExample(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final N4Modification _function = new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        IChange _insertLineAbove = ChangeProvider.insertLineAbove(context.getXtextDocument(), offset, "@SomeAnnotationToBeAdded", true);
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_insertLineAbove));
      }
    };
    this.accept(acceptor, issue, "Some Label", "Some enlightening description.", "SomeImage.gif", _function);
  }
  
  public void someComplexQuickFixExample(final Issue issue, final IssueResolutionAcceptor acceptor) {
    this.accept(acceptor, issue, "Some Label", "Some enlightening description.", "SomeImage.gif", new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        return null;
      }
      
      @Override
      public boolean supportsMultiApply() {
        return true;
      }
      
      @Override
      public boolean isApplicableTo(final IMarker marker) {
        return true;
      }
    });
  }
  
  @Fix(IssueCodes.CLF_FIELD_OPTIONAL_OLD_SYNTAX)
  public void fixOldSyntaxForOptionalFields(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final N4Modification _function = new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        final int offsetNameEnd = N4JSQuickfixProvider.this.getOffsetOfNameEnd(element.eContainer());
        IChange _replace = ChangeProvider.replace(context.getXtextDocument(), ((offset + length) - 1), 1, "");
        IChange _replace_1 = ChangeProvider.replace(context.getXtextDocument(), offsetNameEnd, 0, "?");
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_replace, _replace_1));
      }
    };
    this.accept(acceptor, issue, "Change to new syntax", "The syntax for optional fields has changed. This quick fix will change the code to the new syntax.", ImageNames.REORDER, _function);
  }
  
  private int getOffsetOfNameEnd(final EObject parent) {
    INode _switchResult = null;
    boolean _matched = false;
    if (parent instanceof N4FieldDeclaration) {
      _matched=true;
      _switchResult = NodeModelUtils.findActualNodeFor(((PropertyNameOwner) parent).getDeclaredName());
    }
    if (!_matched) {
      if (parent instanceof TField) {
        _matched=true;
        _switchResult = IterableExtensions.<INode>head(NodeModelUtils.findNodesForFeature(parent, TypesPackage.eINSTANCE.getIdentifiableElement_Name()));
      }
    }
    final INode nodeOfName = _switchResult;
    int _xifexpression = (int) 0;
    if ((nodeOfName != null)) {
      int _offset = nodeOfName.getOffset();
      int _length = nodeOfName.getLength();
      _xifexpression = (_offset + _length);
    } else {
      _xifexpression = (-1);
    }
    return _xifexpression;
  }
  
  @Fix(IssueCodes.FUN_PARAM_OPTIONAL_WRONG_SYNTAX)
  public void fixOldSyntaxForOptionalFpars(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final N4Modification _function = new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        IChange _replace = ChangeProvider.replace(context.getXtextDocument(), ((offset + length) - 1), 1, " = undefined");
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_replace));
      }
    };
    this.accept(acceptor, issue, "Change to Default Parameter", "Some enlightening description.", ImageNames.REORDER, _function);
  }
  
  @Fix(IssueCodes.CLF_OVERRIDE_ANNOTATION)
  public void addOverrideAnnotation(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final N4Modification _function = new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        IChange _insertLineAbove = ChangeProvider.insertLineAbove(context.getXtextDocument(), offset, ("@" + N4JSQuickfixProvider.OVERRIDE_ANNOTATION), true);
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_insertLineAbove));
      }
    };
    this.accept(acceptor, issue, "Add @Override", "Add missing @Override annotation.", ImageNames.ANNOTATION_ADD, _function);
  }
  
  @Fix(IssueCodes.CLF_OVERRIDE_NON_EXISTENT)
  public void removeOverrideAnnotation(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final N4Modification _function = new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        if ((element instanceof N4MethodDeclaration)) {
          final Function1<Annotation, Boolean> _function = (Annotation it) -> {
            String _name = it.getName();
            return Boolean.valueOf(Objects.equal(_name, N4JSQuickfixProvider.OVERRIDE_ANNOTATION));
          };
          IChange _removeSemanticObject = ChangeProvider.removeSemanticObject(context.getXtextDocument(), IterableExtensions.<Annotation>findFirst(((N4MethodDeclaration)element).getAnnotations(), _function), true);
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_removeSemanticObject));
        } else {
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
        }
      }
    };
    this.accept(acceptor, issue, "Remove @Override", "Remove unnecessary @Override annotation.", 
      ImageNames.ANNOTATION_REMOVE, _function);
  }
  
  @Fix(IssueCodes.SYN_MODIFIER_BAD_ORDER)
  public void orderModifiers(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final N4Modification _function = new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        if ((element instanceof ModifiableElement)) {
          final String modifierStr = IterableExtensions.join(ModifierUtils.getSortedModifiers(((ModifiableElement)element).getDeclaredModifiers()), " ");
          IChange _replace = ChangeProvider.replace(context.getXtextDocument(), offset, length, modifierStr);
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_replace));
        } else {
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
        }
      }
    };
    this.accept(acceptor, issue, "Order modifiers", "Rearrange the modifiers to make them appear in correct order.", 
      ImageNames.REORDER, _function);
  }
  
  @Fix(Diagnostic.LINKING_DIAGNOSTIC)
  public void addImportForUnresolvedReference(final Issue issue, final IssueResolutionAcceptor acceptor) {
    boolean _startsWith = issue.getMessage().startsWith("Couldn\'t resolve reference to ");
    if (_startsWith) {
      final List<ICompletionProposal> proposals = this._importUtil.findImportCandidates(issue, false);
      for (final ICompletionProposal currProp : proposals) {
        String _displayString = currProp.getDisplayString();
        String _plus = ("Import " + _displayString);
        this.accept(acceptor, issue, _plus, "Add import declaration for the element.", 
          ImageNames.IMPORT, currProp);
      }
    }
  }
  
  @Fix(IssueCodes.CLF_ABSTRACT_BODY)
  public void removeAbstractAnnotationFromMethodWithBody(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final N4Modification _function = new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        if ((element instanceof N4MethodDeclaration)) {
          IChange _removeModifier = N4JSQuickfixProvider.this._semanticChangeProvider.removeModifier(context.getXtextDocument(), ((ModifiableElement)element), N4Modifier.ABSTRACT);
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_removeModifier));
        }
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
      }
    };
    this.accept(acceptor, issue, "Remove abstract annotation", "", ImageNames.ANNOTATION_REMOVE, _function);
  }
  
  @Fix(IssueCodes.CLF_ABSTRACT_MISSING)
  public void declareClassWithAbstractMethodAsAbstract(final Issue issue, final IssueResolutionAcceptor acceptor) {
    this.accept(acceptor, issue, "Declare class as abstract", "", ImageNames.ANNOTATION_ADD, new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        if ((element instanceof N4MemberDeclaration)) {
          final EObject containingClass = ((N4MemberDeclaration)element).eContainer();
          if ((containingClass instanceof N4ClassDeclaration)) {
            IChange _addModifier = N4JSQuickfixProvider.this._semanticChangeProvider.addModifier(context.getXtextDocument(), ((ModifiableElement)containingClass), N4Modifier.ABSTRACT);
            return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_addModifier));
          }
        }
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
      }
      
      @Override
      public boolean supportsMultiApply() {
        return false;
      }
    });
  }
  
  @Fix(IssueCodes.CLF_MISSING_IMPLEMENTATION)
  public void declareClassNotImplementingAbstractMethodAsAbstract(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final N4Modification _function = new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        if ((element instanceof N4ClassDeclaration)) {
          IChange _addModifier = N4JSQuickfixProvider.this._semanticChangeProvider.addModifier(context.getXtextDocument(), ((ModifiableElement)element), N4Modifier.ABSTRACT);
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_addModifier));
        }
        return null;
      }
    };
    this.accept(acceptor, issue, "Declare class as abstract", "", null, _function);
  }
  
  @Fix(IssueCodes.CLF_MISSING_BODY)
  public void declareMemberWithoutBodyAsAbstract(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final N4Modification _function = new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        if ((element instanceof N4MethodDeclaration)) {
          IChange _addModifier = N4JSQuickfixProvider.this._semanticChangeProvider.addModifier(context.getXtextDocument(), ((ModifiableElement)element), N4Modifier.ABSTRACT);
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_addModifier));
        } else {
          if ((element instanceof N4FieldAccessor)) {
            IChange _addModifier_1 = N4JSQuickfixProvider.this._semanticChangeProvider.addModifier(context.getXtextDocument(), ((ModifiableElement)element), N4Modifier.ABSTRACT);
            return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_addModifier_1));
          }
        }
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
      }
    };
    this.accept(acceptor, issue, "Declare member as abstract", "", null, _function);
  }
  
  @Fix(IssueCodes.VIS_ILLEGAL_MEMBER_ACCESS)
  public void changeAccessModifierOfMemberDeclaration(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final String declarationURI = this._issueUserDataKeysExtension.getUserData(issue, IssueUserDataKeys.VIS_ILLEGAL_MEMBER_ACCESS.DECLARATION_OBJECT_URI);
    final String accessorSuggestion = this._issueUserDataKeysExtension.getUserData(issue, IssueUserDataKeys.VIS_ILLEGAL_MEMBER_ACCESS.ACCESS_SUGGESTION);
    if (((declarationURI == null) || (accessorSuggestion == null))) {
      return;
    }
    boolean _isContainingResourceModifiable = QuickfixUtil.isContainingResourceModifiable(declarationURI);
    boolean _not = (!_isContainingResourceModifiable);
    if (_not) {
      return;
    }
    final N4Modifier fixModifier = QuickfixUtil.modifierForSuggestion(accessorSuggestion);
    final boolean fixAddInternal = QuickfixUtil.modifierSuggestionIsInternal(accessorSuggestion);
    String _readableStringForSuggestion = QuickfixUtil.readableStringForSuggestion(accessorSuggestion);
    final String quickFixLabel = ("Declare member as " + _readableStringForSuggestion);
    this.accept(acceptor, issue, quickFixLabel, "", null, 
      new N4Modification() {
        @Override
        public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
          if ((element instanceof ParameterizedPropertyAccessExpression)) {
            final IdentifiableElement propertyDeclaration = ((ParameterizedPropertyAccessExpression)element).getProperty();
            if (((((propertyDeclaration instanceof Type) || (propertyDeclaration instanceof TMember)) && 
              (propertyDeclaration instanceof SyntaxRelatedTElement)) && 
              (propertyDeclaration instanceof TAnnotableElement))) {
              final IXtextDocument doc = context.getXtextDocument(propertyDeclaration.eResource().getURI());
              final EObject astElement = ((SyntaxRelatedTElement) propertyDeclaration).getAstElement();
              if ((astElement instanceof N4MemberDeclaration)) {
                ArrayList<IChange> changes = new ArrayList<IChange>();
                changes.add(N4JSQuickfixProvider.this._semanticChangeProvider.setAccessModifiers(doc, ((ModifiableElement)astElement), fixModifier));
                if (fixAddInternal) {
                  changes.add(N4JSQuickfixProvider.this._semanticChangeProvider.addAnnotation(doc, ((AnnotableElement)astElement), N4JSQuickfixProvider.INTERNAL_ANNOTATION));
                } else {
                  changes.add(
                    N4JSQuickfixProvider.this._semanticChangeProvider.removeAnnotation(doc, ((AnnotableElement)astElement), N4JSQuickfixProvider.INTERNAL_ANNOTATION));
                }
                return changes;
              }
            }
          }
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
        }
        
        @Override
        public boolean supportsMultiApply() {
          return false;
        }
      });
  }
  
  @Fix(IssueCodes.VIS_ILLEGAL_TYPE_ACCESS)
  public void changeAccessModifierOfTypeDeclaration(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final String accessModifierSuggestion = this._issueUserDataKeysExtension.getUserData(issue, 
      IssueUserDataKeys.VIS_ILLEGAL_TYPE_ACCESS.ACCESS_SUGGESTION);
    final String declarationObjectUri = this._issueUserDataKeysExtension.getUserData(issue, 
      IssueUserDataKeys.VIS_ILLEGAL_TYPE_ACCESS.DECLARATION_OBJECT_URI);
    if (((accessModifierSuggestion == null) || (declarationObjectUri == null))) {
      return;
    }
    boolean _isContainingResourceModifiable = QuickfixUtil.isContainingResourceModifiable(declarationObjectUri);
    boolean _not = (!_isContainingResourceModifiable);
    if (_not) {
      return;
    }
    TopLevelVisibilityFixProvider.TopLevelVisibilityFix fix = null;
    try {
      fix = this.topLevelVisibilityFixProvider.provideFixFor("type", accessModifierSuggestion);
    } catch (final Throwable _t) {
      if (_t instanceof IllegalArgumentException) {
        return;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    final TopLevelVisibilityFixProvider.TopLevelVisibilityFix finalFix = fix;
    String _description = fix.getDescription();
    this.accept(acceptor, issue, _description, "", null, new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        EObject typeDeclaration = null;
        if ((element instanceof ParameterizedPropertyAccessExpression)) {
          typeDeclaration = ((ParameterizedPropertyAccessExpression)element).getProperty();
        } else {
          if ((element instanceof TypeRef)) {
            typeDeclaration = ((TypeRef)element).getDeclaredType();
          } else {
            if ((element instanceof NamedImportSpecifier)) {
              typeDeclaration = ((NamedImportSpecifier)element).getImportedElement();
            } else {
              if ((element instanceof IdentifierRef)) {
                typeDeclaration = ((IdentifierRef)element).getId();
              }
            }
          }
        }
        if ((typeDeclaration == null)) {
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
        } else {
          if ((((typeDeclaration instanceof Type) && 
            (typeDeclaration instanceof SyntaxRelatedTElement)) && 
            (typeDeclaration instanceof TAnnotableElement))) {
            EObject declarationAstElement = ((SyntaxRelatedTElement) typeDeclaration).getAstElement();
            final IXtextDocument doc = context.getXtextDocument(URI.createURI(declarationObjectUri));
            return (Collection<? extends IChange>)Conversions.doWrapArray(finalFix.changes(declarationAstElement, doc));
          }
        }
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
      }
      
      @Override
      public boolean supportsMultiApply() {
        return false;
      }
    });
  }
  
  @Fix(IssueCodes.VIS_ILLEGAL_VARIABLE_ACCESS)
  public void changeAccessModifierOfVariableDeclaration(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final String accessModifierSuggestion = this._issueUserDataKeysExtension.getUserData(issue, 
      IssueUserDataKeys.VIS_ILLEGAL_VARIABLE_ACCESS.ACCESS_SUGGESTION);
    final String declarationObjectURI = this._issueUserDataKeysExtension.getUserData(issue, 
      IssueUserDataKeys.VIS_ILLEGAL_VARIABLE_ACCESS.DECLARATION_OBJECT_URI);
    if (((accessModifierSuggestion == null) || (declarationObjectURI == null))) {
      return;
    }
    boolean _isContainingResourceModifiable = QuickfixUtil.isContainingResourceModifiable(declarationObjectURI);
    boolean _not = (!_isContainingResourceModifiable);
    if (_not) {
      return;
    }
    TopLevelVisibilityFixProvider.TopLevelVisibilityFix fix = null;
    try {
      fix = this.topLevelVisibilityFixProvider.provideFixFor("variable", accessModifierSuggestion);
    } catch (final Throwable _t) {
      if (_t instanceof IllegalArgumentException) {
        return;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    final TopLevelVisibilityFixProvider.TopLevelVisibilityFix finalFix = fix;
    String _description = fix.getDescription();
    this.accept(acceptor, issue, _description, "", null, new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        EObject variableDeclaration = null;
        if ((element instanceof IdentifierRef)) {
          variableDeclaration = ((IdentifierRef)element).getId();
        } else {
          if ((element instanceof ParameterizedPropertyAccessExpression)) {
            variableDeclaration = ((ParameterizedPropertyAccessExpression)element).getProperty();
          } else {
            if ((element instanceof NamedImportSpecifier)) {
              TExportableElement _importedElement = ((NamedImportSpecifier)element).getImportedElement();
              if ((_importedElement instanceof TVariable)) {
                variableDeclaration = ((NamedImportSpecifier)element).getImportedElement();
              }
            }
          }
        }
        if ((variableDeclaration == null)) {
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
        }
        if ((variableDeclaration instanceof TVariable)) {
          EObject variableNode = ((TVariable)variableDeclaration).getAstElement();
          final IXtextDocument doc = context.getXtextDocument(URI.createURI(declarationObjectURI));
          if ((variableNode instanceof ExportedVariableDeclaration)) {
            final EObject statement = ((ExportedVariableDeclaration)variableNode).eContainer();
            return (Collection<? extends IChange>)Conversions.doWrapArray(finalFix.changes(statement, doc));
          }
        }
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
      }
      
      @Override
      public boolean supportsMultiApply() {
        return false;
      }
    });
  }
  
  @Fix(IssueCodes.VIS_ILLEGAL_FUN_ACCESS)
  public void changeAccessModifierOfFunctionDeclaration(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final String accessModifierSuggestion = this._issueUserDataKeysExtension.getUserData(issue, 
      IssueUserDataKeys.VIS_ILLEGAL_FUN_ACCESS.ACCESS_SUGGESTION);
    final String declarationObjectUri = this._issueUserDataKeysExtension.getUserData(issue, 
      IssueUserDataKeys.VIS_ILLEGAL_FUN_ACCESS.DECLARATION_OBJECT_URI);
    if (((accessModifierSuggestion == null) || (declarationObjectUri == null))) {
      return;
    }
    boolean _isContainingResourceModifiable = QuickfixUtil.isContainingResourceModifiable(declarationObjectUri);
    boolean _not = (!_isContainingResourceModifiable);
    if (_not) {
      return;
    }
    TopLevelVisibilityFixProvider.TopLevelVisibilityFix fix = null;
    try {
      fix = this.topLevelVisibilityFixProvider.provideFixFor("function", accessModifierSuggestion);
    } catch (final Throwable _t) {
      if (_t instanceof IllegalArgumentException) {
        return;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    final TopLevelVisibilityFixProvider.TopLevelVisibilityFix finalFix = fix;
    String _description = fix.getDescription();
    this.accept(acceptor, issue, _description, "", null, new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        EObject functionImpl = null;
        if ((element instanceof IdentifierRef)) {
          functionImpl = ((IdentifierRef)element).getId();
        } else {
          if ((element instanceof ParameterizedPropertyAccessExpression)) {
            functionImpl = ((ParameterizedPropertyAccessExpression)element).getProperty();
          } else {
            if ((element instanceof NamedImportSpecifier)) {
              functionImpl = ((NamedImportSpecifier)element).getImportedElement();
            }
          }
        }
        if ((functionImpl == null)) {
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
        }
        if ((functionImpl instanceof TFunction)) {
          final EObject functionAstElement = ((TFunction)functionImpl).getAstElement();
          final IXtextDocument doc = context.getXtextDocument(URI.createURI(declarationObjectUri));
          return (Collection<? extends IChange>)Conversions.doWrapArray(finalFix.changes(functionAstElement, doc));
        }
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
      }
      
      @Override
      public boolean supportsMultiApply() {
        return false;
      }
    });
  }
  
  @Fix(IssueCodes.CLF_OVERRIDE_VISIBILITY)
  public void changeOverriddenMemberAccessModifier(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final String accessSuggestion = this._issueUserDataKeysExtension.getUserData(issue, 
      IssueUserDataKeys.CLF_OVERRIDE_VISIBILITY.OVERRIDDEN_MEMBER_ACCESS_MODIFIER);
    final String memberName = this._issueUserDataKeysExtension.getUserData(issue, IssueUserDataKeys.CLF_OVERRIDE_VISIBILITY.OVERRIDDEN_MEMBER_NAME);
    final String superClassName = this._issueUserDataKeysExtension.getUserData(issue, IssueUserDataKeys.CLF_OVERRIDE_VISIBILITY.SUPER_CLASS_NAME);
    if ((((accessSuggestion == null) || (memberName == null)) || (superClassName == null))) {
      return;
    }
    String _readableStringForSuggestion = QuickfixUtil.readableStringForSuggestion(accessSuggestion);
    String _plus = ("Set access modifier to \"" + _readableStringForSuggestion);
    String _plus_1 = (_plus + "\" (align with ");
    String _plus_2 = (_plus_1 + superClassName);
    String _plus_3 = (_plus_2 + ".");
    String _plus_4 = (_plus_3 + memberName);
    final String msg = (_plus_4 + ")");
    final N4Modification _function = new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        if ((element instanceof N4MemberDeclaration)) {
          ArrayList<IChange> changes = new ArrayList<IChange>();
          changes.add(N4JSQuickfixProvider.this._semanticChangeProvider.setAccessModifiers(context.getXtextDocument(), ((ModifiableElement)element), QuickfixUtil.modifierForSuggestion(accessSuggestion)));
          boolean _modifierSuggestionIsInternal = QuickfixUtil.modifierSuggestionIsInternal(accessSuggestion);
          if (_modifierSuggestionIsInternal) {
            changes.add(N4JSQuickfixProvider.this._semanticChangeProvider.addAnnotation(context.getXtextDocument(), ((AnnotableElement)element), N4JSQuickfixProvider.INTERNAL_ANNOTATION));
          } else {
            changes.add(N4JSQuickfixProvider.this._semanticChangeProvider.removeAnnotation(context.getXtextDocument(), ((AnnotableElement)element), N4JSQuickfixProvider.INTERNAL_ANNOTATION));
          }
          return changes;
        }
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
      }
    };
    this.accept(acceptor, issue, msg, "", null, _function);
  }
  
  @Fix(IssueCodes.CLF_NOT_EXPORTED_NOT_PRIVATE)
  public void markElementAsExported(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final N4Modification _function = new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        if ((N4JSQuickfixProvider.this.jsVariantHelper.isExternalMode(element) && (element instanceof ModifiableElement))) {
          IChange _addCustomModifier = N4JSQuickfixProvider.this._semanticChangeProvider.addCustomModifier(context.getXtextDocument(), ((ModifiableElement) element), ((N4JSLanguageConstants.EXPORT_KEYWORD + " ") + N4JSLanguageConstants.EXTERNAL_KEYWORD));
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_addCustomModifier));
        } else {
          if ((element instanceof ModifiableElement)) {
            IChange _addCustomModifier_1 = N4JSQuickfixProvider.this._semanticChangeProvider.addCustomModifier(context.getXtextDocument(), ((ModifiableElement)element), N4JSLanguageConstants.EXPORT_KEYWORD);
            return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_addCustomModifier_1));
          }
        }
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
      }
    };
    this.accept(acceptor, issue, "Declare element as exported", "", null, _function);
  }
  
  @Fix(IssueCodes.CLF_EXTEND_FINAL)
  public void unmarkSuperTypeAsFinal(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final String superTypeDeclarationUri = this._issueUserDataKeysExtension.getUserData(issue, 
      IssueUserDataKeys.CLF_EXTEND_FINAL.SUPER_TYPE_DECLARATION_URI);
    if ((superTypeDeclarationUri == null)) {
      return;
    }
    boolean _isContainingResourceModifiable = QuickfixUtil.isContainingResourceModifiable(superTypeDeclarationUri);
    boolean _not = (!_isContainingResourceModifiable);
    if (_not) {
      return;
    }
    this.accept(acceptor, issue, "Remove @Final annotation from super type", "", null, new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        if ((element instanceof ParameterizedTypeRef)) {
          final Type superClassDeclaration = ((ParameterizedTypeRef)element).getDeclaredType();
          if ((superClassDeclaration instanceof TClassifier)) {
            final EObject astDeclaration = ((TClassifier)superClassDeclaration).getAstElement();
            if ((astDeclaration instanceof N4ClassifierDeclaration)) {
              final IXtextDocument doc = context.getXtextDocument(URI.createURI(superTypeDeclarationUri));
              IChange _removeAnnotation = N4JSQuickfixProvider.this._semanticChangeProvider.removeAnnotation(doc, ((AnnotableElement)astDeclaration), N4JSQuickfixProvider.FINAL_ANNOTATION);
              return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_removeAnnotation));
            }
          }
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
        }
        return null;
      }
      
      @Override
      public boolean supportsMultiApply() {
        return false;
      }
    });
  }
  
  @Fix(IssueCodes.CLF_OVERRIDE_FINAL)
  public void removeFinalAnnotationFromMember(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final String overriddenMemberDeclarationUri = this._issueUserDataKeysExtension.getUserData(issue, 
      IssueUserDataKeys.CLF_OVERRIDE_FINAL.OVERRIDDEN_MEMBER_URI);
    if ((overriddenMemberDeclarationUri == null)) {
      return;
    }
    boolean _isContainingResourceModifiable = QuickfixUtil.isContainingResourceModifiable(overriddenMemberDeclarationUri);
    boolean _not = (!_isContainingResourceModifiable);
    if (_not) {
      return;
    }
    this.accept(acceptor, issue, "Remove @Final annotation from overridden member", "", null, new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        final Resource resource = element.eResource();
        final EObject overriddenMemberDeclaration = QuickfixUtil.getEObjectForUri(resource.getResourceSet(), overriddenMemberDeclarationUri);
        if ((overriddenMemberDeclaration == null)) {
          return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
        }
        if ((overriddenMemberDeclaration instanceof SyntaxRelatedTElement)) {
          final EObject astMemberDeclaration = ((SyntaxRelatedTElement)overriddenMemberDeclaration).getAstElement();
          if ((astMemberDeclaration instanceof N4MemberDeclaration)) {
            final IXtextDocument doc = context.getXtextDocument(URI.createURI(overriddenMemberDeclarationUri));
            IChange _removeAnnotation = N4JSQuickfixProvider.this._semanticChangeProvider.removeAnnotation(doc, ((AnnotableElement)astMemberDeclaration), N4JSQuickfixProvider.FINAL_ANNOTATION);
            return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_removeAnnotation));
          }
        }
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
      }
      
      @Override
      public boolean supportsMultiApply() {
        return false;
      }
    });
  }
  
  @Fix(IssueCodes.NON_EXISTING_PROJECT)
  public void tryInstallMissingDependencyFromNpm(final Issue issue, final IssueResolutionAcceptor acceptor) {
    abstract class __N4JSQuickfixProvider_9 extends N4Modification {
      boolean multipleInvocations;
      
      public abstract Collection<? extends IChange> invokeNpmManager(final EObject element) throws Exception;
    }
    
    final __N4JSQuickfixProvider_9 modification = new __N4JSQuickfixProvider_9() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        return this.invokeNpmManager(element);
      }
      
      @Override
      public Collection<? extends IChange> computeOneOfMultipleChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        return this.invokeNpmManager(element);
      }
      
      @Override
      public void computeFinalChanges() throws Exception {
        if (this.multipleInvocations) {
          Shell _shell = UIUtils.getShell();
          final IRunnableWithProgress _function = (IProgressMonitor monitor) -> {
            try {
              ResourcesPlugin.getWorkspace().build(IncrementalProjectBuilder.CLEAN_BUILD, monitor);
            } catch (final Throwable _t) {
              if (_t instanceof IllegalBinaryStateException) {
              } else if (_t instanceof CoreException) {
              } else {
                throw Exceptions.sneakyThrow(_t);
              }
            }
          };
          new ProgressMonitorDialog(_shell).run(true, true, _function);
        }
      }
      
      public Collection<? extends IChange> invokeNpmManager(final EObject element) throws Exception {
        final ProjectReference dependency = ((ProjectReference) element);
        final String packageName = dependency.getProjectName();
        NPMVersionRequirement _xifexpression = null;
        if ((dependency instanceof ProjectDependency)) {
          _xifexpression = ((ProjectDependency)dependency).getVersionRequirement();
        } else {
          _xifexpression = SemverUtils.createEmptyVersionRequirement();
        }
        final NPMVersionRequirement packageVersion = _xifexpression;
        final AtomicReference<IllegalBinaryStateException> illegalBinaryExcRef = new AtomicReference<IllegalBinaryStateException>();
        final MultiStatus multiStatus = N4JSQuickfixProvider.this._statusHelper.createMultiStatus((("Installing npm \'" + packageName) + "\'."));
        Shell _shell = UIUtils.getShell();
        final IRunnableWithProgress _function = (IProgressMonitor monitor) -> {
          try {
            final Map<String, NPMVersionRequirement> package_ = Collections.<String, NPMVersionRequirement>singletonMap(packageName, packageVersion);
            multiStatus.merge(N4JSQuickfixProvider.this.libraryManager.installNPMs(package_, false, monitor));
          } catch (final Throwable _t) {
            if (_t instanceof IllegalBinaryStateException) {
              final IllegalBinaryStateException e = (IllegalBinaryStateException)_t;
              illegalBinaryExcRef.set(e);
            } else if (_t instanceof Exception) {
              final Exception e_1 = (Exception)_t;
              final String msg = (("Error while uninstalling npm dependency: \'" + packageName) + "\'.");
              multiStatus.merge(N4JSQuickfixProvider.this._statusHelper.createError(msg, e_1));
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
        };
        new ProgressMonitorDialog(_shell).run(true, false, _function);
        IllegalBinaryStateException _get = illegalBinaryExcRef.get();
        boolean _tripleNotEquals = (null != _get);
        if (_tripleNotEquals) {
          IllegalBinaryStateException _get_1 = illegalBinaryExcRef.get();
          new IllegalBinaryStateDialog(_get_1).open();
        } else {
          boolean _isOK = multiStatus.isOK();
          boolean _not = (!_isOK);
          if (_not) {
            N4JSActivator.getInstance().getLog().log(multiStatus);
            final Runnable _function_1 = () -> {
              final String title = "NPM Install Failed";
              final String descr = StatusUtils.getErrorMessage(multiStatus, true);
              ErrorDialog.openError(UIUtils.getShell(), title, descr, multiStatus);
            };
            UIUtils.getDisplay().asyncExec(_function_1);
          }
        }
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList());
      }
    };
    this.accept(acceptor, issue, "Install npm package to workspace", "Download and install missing dependency from npm.", null, modification);
  }
  
  /**
   * N4IDL-related quick-fix which adds a "@VersionAware" annotation to
   * classes which do not declare an explicit type version.
   */
  @Fix(IssueCodes.IDL_VERSIONED_ELEMENT_MISSING_VERSION)
  public void addVersionAwareAnnotation(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final N4Modification _function = new N4Modification() {
      @Override
      public Collection<? extends IChange> computeChanges(final IModificationContext context, final IMarker marker, final int offset, final int length, final EObject element) throws Exception {
        IChange _insertLineAbove = ChangeProvider.insertLineAbove(context.getXtextDocument(), offset, ("@" + AnnotationDefinition.VERSION_AWARE.name), true);
        return Collections.<IChange>unmodifiableList(CollectionLiterals.<IChange>newArrayList(_insertLineAbove));
      }
    };
    this.accept(acceptor, issue, "Declare this type as @VersionAware", "Add @VersionAware annotation.", ImageNames.ANNOTATION_ADD, _function);
  }
}
