/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.wizard.interfaces;

import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.eclipse.emf.common.util.URI;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.ui.wizard.classifiers.N4JSNewClassifierWizardGenerator;
import org.eclipse.n4js.ui.wizard.generator.ImportRequirement;
import org.eclipse.n4js.ui.wizard.generator.N4JSImportRequirementResolver;
import org.eclipse.n4js.ui.wizard.generator.WizardGeneratorHelper;
import org.eclipse.n4js.ui.wizard.interfaces.N4JSInterfaceWizardModel;
import org.eclipse.n4js.ui.wizard.model.ClassifierReference;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * A file generator for {@link N4JSInterfaceWizardModel}.
 */
@SuppressWarnings("all")
public class N4JSNewInterfaceWizardGenerator extends N4JSNewClassifierWizardGenerator<N4JSInterfaceWizardModel> {
  @Inject
  @Extension
  private WizardGeneratorHelper generatorUtils;
  
  /**
   * Generates the interface code.
   */
  @Override
  protected String generateClassifierCode(final N4JSInterfaceWizardModel model, final Map<URI, String> aliasBindings) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isN4jsAnnotated = model.isN4jsAnnotated();
      if (_isN4jsAnnotated) {
        _builder.append("@N4JS");
      }
    }
    _builder.newLineIfNotEmpty();
    {
      boolean _isInternal = model.isInternal();
      if (_isInternal) {
        _builder.append("@Internal ");
      }
    }
    String _addSpace = this.generatorUtils.addSpace(this.generatorUtils.exportStatement(model.getAccessModifier()));
    _builder.append(_addSpace);
    {
      boolean _isDefinitionFile = model.isDefinitionFile();
      if (_isDefinitionFile) {
        _builder.append("external ");
      }
    }
    String _addSpace_1 = this.generatorUtils.addSpace(model.getAccessModifier().toCodeRepresentation());
    _builder.append(_addSpace_1);
    _builder.append("interface ");
    String _name = model.getName();
    _builder.append(_name);
    {
      int _length = ((Object[])Conversions.unwrapArray(model.getInterfaces(), Object.class)).length;
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        _builder.append(" extends ");
      }
    }
    {
      List<ClassifierReference> _interfaces = model.getInterfaces();
      boolean _hasElements = false;
      for(final ClassifierReference iface : _interfaces) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(", ", "");
        }
        String _realOrAliasName = this.generatorUtils.realOrAliasName(iface, aliasBindings);
        _builder.append(_realOrAliasName);
      }
    }
    _builder.append(" {");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    return _builder.toString();
  }
  
  @Override
  protected List<IN4JSProject> getReferencedProjects(final N4JSInterfaceWizardModel model) {
    final Function1<ClassifierReference, IN4JSProject> _function = (ClassifierReference it) -> {
      return this.generatorUtils.projectOfUri(it.uri);
    };
    return ListExtensions.<ClassifierReference, IN4JSProject>map(model.getInterfaces(), _function);
  }
  
  /**
   * Return the import requirement of a N4JSInterfaceWizardModel
   */
  @Override
  protected List<ImportRequirement> getImportRequirements(final N4JSInterfaceWizardModel model) {
    ArrayList<ImportRequirement> _xblockexpression = null;
    {
      ArrayList<ImportRequirement> demandedImports = new ArrayList<ImportRequirement>();
      boolean _isEmpty = model.getInterfaces().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        demandedImports.addAll(N4JSImportRequirementResolver.classifierReferencesToImportRequirements(model.getInterfaces()));
      }
      _xblockexpression = demandedImports;
    }
    return _xblockexpression;
  }
}
