/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.generator;

import com.google.common.base.Objects;
import com.google.common.base.Optional;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.EMFPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.N4JSGlobals;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.generator.CompilerDescriptor;
import org.eclipse.n4js.generator.CompilerProperties;
import org.eclipse.n4js.generator.GeneratorException;
import org.eclipse.n4js.generator.GeneratorExceptionHandler;
import org.eclipse.n4js.generator.GeneratorOption;
import org.eclipse.n4js.generator.IGeneratorMarkerSupport;
import org.eclipse.n4js.generator.ISubGenerator;
import org.eclipse.n4js.generator.N4JSPreferenceAccess;
import org.eclipse.n4js.internal.RaceDetectionHelper;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.projectModel.IN4JSSourceContainer;
import org.eclipse.n4js.resource.N4JSCache;
import org.eclipse.n4js.resource.N4JSResource;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.utils.Log;
import org.eclipse.n4js.utils.ResourceNameComputer;
import org.eclipse.n4js.utils.StaticPolyfillHelper;
import org.eclipse.n4js.validation.helper.FolderContainmentHelper;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.generator.AbstractFileSystemAccess;
import org.eclipse.xtext.generator.IFileSystemAccess;
import org.eclipse.xtext.generator.OutputConfiguration;
import org.eclipse.xtext.service.OperationCanceledManager;
import org.eclipse.xtext.util.CancelIndicator;
import org.eclipse.xtext.validation.IResourceValidator;
import org.eclipse.xtext.validation.Issue;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * All sub generators should extend this class. It provides basic blocks of the logic, and
 * shared implementations.
 */
@Log
@SuppressWarnings("all")
public abstract class AbstractSubGenerator implements ISubGenerator {
  @Accessors
  private CompilerDescriptor compilerDescriptor = null;
  
  @Inject
  protected StaticPolyfillHelper staticPolyfillHelper;
  
  @Inject
  protected IN4JSCore n4jsCore;
  
  @Inject
  protected ResourceNameComputer resourceNameComputer;
  
  @Inject
  protected IResourceValidator resVal;
  
  @Inject
  protected N4JSCache cache;
  
  @Inject
  protected IGeneratorMarkerSupport genMarkerSupport;
  
  @Inject
  protected OperationCanceledManager operationCanceledManager;
  
  @Inject
  protected GeneratorExceptionHandler exceptionHandler;
  
  @Inject
  protected N4JSPreferenceAccess preferenceAccess;
  
  @Inject
  private FolderContainmentHelper containmentHelper;
  
  @Override
  public CompilerDescriptor getCompilerDescriptor() {
    if ((this.compilerDescriptor == null)) {
      this.compilerDescriptor = this.getDefaultDescriptor();
    }
    return this.compilerDescriptor;
  }
  
  /**
   * This override declares an {@link IFileSystemAccess} parameter. At runtime, its actual type depends on whether IDE or headless,
   * which in turn determines whether the actual argument has a progress monitor or not:
   * <ul>
   * <li>
   * IDE scenario: Actual type is {@code EclipseResourceFileSystemAccess2}. A progress monitor can be obtained via {@code getMonitor()}.
   * It is checked automatically behind the scenes, for example in {@code generateFile()}.
   * Upon detecting a pending cancellation request, an {@code OperationCanceledException} is raised.
   * </li>
   * <li>
   * Headless scenario: Actual type is {@code JavaIoFileSystemAccess}. No progress monitor is available.
   * </li>
   * </ul>
   */
  @Override
  public void doGenerate(final Resource input, final IFileSystemAccess fsa) {
    try {
      this.genMarkerSupport.deleteMarker(input);
      this.updateOutputPath(fsa, this.getCompilerID(), input);
      this.internalDoGenerate(input, GeneratorOption.DEFAULT_OPTIONS, fsa);
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        this.operationCanceledManager.propagateIfCancelException(e);
        String _elvis = null;
        String _xifexpression = null;
        if ((input instanceof N4JSResource)) {
          TModule _module = ((N4JSResource)input).getModule();
          String _moduleSpecifier = null;
          if (_module!=null) {
            _moduleSpecifier=_module.getModuleSpecifier();
          }
          _xifexpression = _moduleSpecifier;
        }
        if (_xifexpression != null) {
          _elvis = _xifexpression;
        } else {
          URI _uRI = input.getURI();
          String _string = null;
          if (_uRI!=null) {
            _string=_uRI.toString();
          }
          _elvis = _string;
        }
        final String target = _elvis;
        final String msgMarker = (("Severe error occurred while transpiling module " + target) + ". Check error log for details about the failure.");
        this.genMarkerSupport.createMarker(input, msgMarker, IGeneratorMarkerSupport.Severity.ERROR);
        if ((e instanceof GeneratorException)) {
          throw ((GeneratorException)e);
        }
        String _xifexpression_1 = null;
        String _message = e.getMessage();
        boolean _tripleEquals = (_message == null);
        if (_tripleEquals) {
          String _name = e.getClass().getName();
          _xifexpression_1 = ("type=" + _name);
        } else {
          String _message_1 = e.getMessage();
          _xifexpression_1 = ("message=" + _message_1);
        }
        String msg = _xifexpression_1;
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Severe error occurred in transpiler=");
        String _compilerID = this.getCompilerID();
        _builder.append(_compilerID);
        _builder.append(" ");
        _builder.append(msg);
        _builder.append(".");
        this.exceptionHandler.handleError(_builder.toString(), e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  @Override
  public boolean shouldBeCompiled(final Resource input, final CancelIndicator monitor) {
    final boolean autobuildEnabled = this.isActive(input);
    String _lowerCase = input.getURI().fileExtension().toLowerCase();
    final boolean isXPECTMode = Objects.equal(N4JSGlobals.XT_FILE_EXTENSION, _lowerCase);
    final URI inputUri = input.getURI();
    final boolean result = ((((((((autobuildEnabled && this.isGenerateProjectType(inputUri)) && this.hasOutput(inputUri)) && this.isOutputNotInSourceContainer(inputUri)) && this.isOutsideOfOutputFolder(inputUri)) && this.isSource(inputUri)) && 
      ((this.isNoValidate(inputUri) || this.isExternal(inputUri)) || ((EMFPlugin.IS_ECLIPSE_RUNNING && (!isXPECTMode)) || this.hasNoErrors(input, monitor)))) && (!this.isStaticPolyfillingModule(input))) && this.hasNoPolyfillErrors(input, monitor));
    if ((!result)) {
      RaceDetectionHelper.log("Skip generation of artifacts from %s", input.getURI());
    }
    return result;
  }
  
  private boolean hasOutput(final URI n4jsSourceURI) {
    String _outputPath = this.n4jsCore.getOutputPath(n4jsSourceURI);
    return (_outputPath != null);
  }
  
  private boolean isSource(final URI n4jsSourceURI) {
    return this.n4jsCore.findN4JSSourceContainer(n4jsSourceURI).isPresent();
  }
  
  private boolean isNoValidate(final URI n4jsSourceURI) {
    return this.n4jsCore.isNoValidate(n4jsSourceURI);
  }
  
  private boolean isExternal(final URI n4jsSourceURI) {
    final Optional<? extends IN4JSSourceContainer> sourceContainerOpt = this.n4jsCore.findN4JSSourceContainer(n4jsSourceURI);
    boolean _isPresent = sourceContainerOpt.isPresent();
    if (_isPresent) {
      final IN4JSSourceContainer sourceContainer = sourceContainerOpt.get();
      return sourceContainer.isExternal();
    }
    return false;
  }
  
  /**
   * If the resource has a static polyfill, then ensure it is error-free.
   * Calls {@link #hasNoErrors()} on the static polyfill resource.
   */
  private boolean hasNoPolyfillErrors(final Resource input, final CancelIndicator monitor) {
    final N4JSResource resSPoly = this.staticPolyfillHelper.getStaticPolyfillResource(input);
    if ((resSPoly == null)) {
      return true;
    }
    return this.hasNoErrors(resSPoly, monitor);
  }
  
  /**
   * Does validation report no errors for the given resource?
   * If errors exists, log them as a side-effect.
   * If validation was canceled before finishing, don't assume absence of errors.
   */
  private boolean hasNoErrors(final Resource input, final CancelIndicator monitor) {
    final List<Issue> issues = this.cache.getOrElseUpdateIssues(this.resVal, input, monitor);
    if ((null == issues)) {
      this.warnDueToCancelation(input, null);
      return false;
    }
    final Function1<Issue, Boolean> _function = (Issue it) -> {
      Severity _severity = it.getSeverity();
      return Boolean.valueOf(Objects.equal(_severity, Severity.ERROR));
    };
    final Iterable<Issue> errors = IterableExtensions.<Issue>filter(issues, _function);
    boolean _isEmpty = IterableExtensions.isEmpty(errors);
    if (_isEmpty) {
      return true;
    }
    boolean _isDebugEnabled = AbstractSubGenerator.logger.isDebugEnabled();
    if (_isDebugEnabled) {
      final Consumer<Issue> _function_1 = (Issue it) -> {
        URI _uRI = input.getURI();
        String _plus = (_uRI + "  ");
        String _message = it.getMessage();
        String _plus_1 = (_plus + _message);
        String _plus_2 = (_plus_1 + "  ");
        Severity _severity = it.getSeverity();
        String _plus_3 = (_plus_2 + _severity);
        String _plus_4 = (_plus_3 + " @L_");
        Integer _lineNumber = it.getLineNumber();
        String _plus_5 = (_plus_4 + _lineNumber);
        String _plus_6 = (_plus_5 + " ");
        AbstractSubGenerator.logger.debug(_plus_6);
      };
      errors.forEach(_function_1);
    }
    return false;
  }
  
  private void warnDueToCancelation(final Resource input, final Throwable exc) {
    URI _uRI = input.getURI();
    String _plus = ("User canceled the validation of " + _uRI);
    final String msg = (_plus + ". Will not compile.");
    if ((null == exc)) {
      AbstractSubGenerator.logger.warn(msg);
    } else {
      AbstractSubGenerator.logger.warn(msg, exc);
    }
  }
  
  /**
   * @return true iff the current project has a project type that is supposed to generate code.
   */
  public boolean isGenerateProjectType(final URI n4jsSourceURI) {
    final IN4JSProject project = this.n4jsCore.findProject(n4jsSourceURI).orNull();
    if ((project != null)) {
      final ProjectType projectType = project.getProjectType();
      if (((Objects.equal(projectType, ProjectType.VALIDATION) || 
        Objects.equal(projectType, ProjectType.DEFINITION)) || 
        Objects.equal(projectType, ProjectType.PLAINJS))) {
        return false;
      }
    }
    return true;
  }
  
  /**
   * @return true iff the given resource does not lie within the output folder.
   */
  public boolean isOutsideOfOutputFolder(final URI n4jsSourceURI) {
    boolean _isContainedInOutputFolder = this.containmentHelper.isContainedInOutputFolder(n4jsSourceURI);
    return (!_isContainedInOutputFolder);
  }
  
  /**
   * @return true iff the output folder of the given n4js resource is not contained by a source container.
   */
  public boolean isOutputNotInSourceContainer(final URI n4jsSourceURI) {
    final Optional<? extends IN4JSProject> project = this.n4jsCore.findProject(n4jsSourceURI);
    boolean _isPresent = project.isPresent();
    if (_isPresent) {
      boolean _isOutputContainedInSourceContainer = this.containmentHelper.isOutputContainedInSourceContainer(project.get());
      return (!_isOutputContainedInSourceContainer);
    } else {
      return false;
    }
  }
  
  /**
   * Actual generation to be overridden by subclasses.
   */
  protected abstract void internalDoGenerate(final Resource resource, final GeneratorOption[] options, final IFileSystemAccess access);
  
  /**
   * Returns the name of the target file (without path) to which the source is to be compiled to.
   * Default implementation returns a configured project Name with version + file name + extension.
   * E.g., "proj-0.0.1/p/A.js" for a file A in proj.
   * 
   * Convenience method, to provide all necessary API for the sub-classes.
   * Delegates to {@link ResourceNameComputer#getTargetFileName}.
   */
  public String getTargetFileName(final Resource n4jsSourceFile, final String compiledFileExtension) {
    return this.resourceNameComputer.generateFileDescriptor(n4jsSourceFile, compiledFileExtension);
  }
  
  /**
   * Convenient access to the Script-Element
   */
  public Script rootElement(final Resource resource) {
    return IterableExtensions.<Script>head(Iterables.<Script>filter(resource.getContents(), Script.class));
  }
  
  /**
   * The file-extension of the compiled result
   */
  public String getCompiledFileExtension(final Resource input) {
    return this.preferenceAccess.getPreference(input, this.getCompilerID(), CompilerProperties.COMPILED_FILE_EXTENSION, 
      this.getDefaultDescriptor());
  }
  
  /**
   * The file-extension of the source-map to the compiled result
   */
  public String getCompiledFileSourceMapExtension(final Resource input) {
    return this.preferenceAccess.getPreference(input, this.getCompilerID(), CompilerProperties.COMPILED_FILE_SOURCEMAP_EXTENSION, 
      this.getDefaultDescriptor());
  }
  
  /**
   * Adjust output-path of the generator to match the N4JS projects-settings.
   */
  public void updateOutputPath(final IFileSystemAccess fsa, final String compilerID, final Resource input) {
    String _elvis = null;
    String _outputPath = this.n4jsCore.getOutputPath(input.getURI());
    if (_outputPath != null) {
      _elvis = _outputPath;
    } else {
      _elvis = N4JSLanguageConstants.DEFAULT_PROJECT_OUTPUT;
    }
    final String outputPath = _elvis;
    if ((fsa instanceof AbstractFileSystemAccess)) {
      final OutputConfiguration conf = ((AbstractFileSystemAccess)fsa).getOutputConfigurations().get(compilerID);
      if ((conf != null)) {
        conf.setOutputDirectory(outputPath);
      }
    }
  }
  
  /**
   * Navigation from the generated output-location to the location of the input-resource
   */
  public Path calculateNavigationFromOutputToSourcePath(final IFileSystemAccess fsa, final String compilerID, final N4JSResource input) {
    final Optional<? extends IN4JSProject> projectctContainer = this.n4jsCore.findProject(input.getURI());
    final IN4JSProject project = projectctContainer.get();
    final Path projectPath = project.getLocationPath();
    final URI projectLocURI = project.getLocation().appendSegment("");
    final String outputPath = project.getOutputPath();
    final Path outputRelativeLocation = this.getOutputRelativeLocation(input);
    final URI completetSourceURI = input.getURI().trimSegments(1).deresolve(projectLocURI);
    String completetSource = completetSourceURI.toFileString();
    if (((null == completetSource) && (project.getLocation() == input.getURI().trimSegments(1)))) {
      completetSource = projectPath.toFile().getAbsolutePath();
    }
    final Path fullOutpath = projectPath.resolve(outputPath).normalize().resolve(outputRelativeLocation).normalize();
    final Path fullSourcePath = projectPath.resolve(completetSource).normalize();
    final Path rel = fullOutpath.relativize(fullSourcePath);
    return rel;
  }
  
  /**
   * Calculates local output path for a given resource.
   * Depending on the configuration this path can be in various forms, {@code Project-1.0.0/a/b/c/},
   * {@code Project/a/b/c/} or just {@code a/b/c/}
   */
  private Path getOutputRelativeLocation(final N4JSResource input) {
    final Path localOutputFilePath = Paths.get(this.resourceNameComputer.generateFileDescriptor(input.getURI(), ".XX"));
    int _nameCount = localOutputFilePath.getNameCount();
    boolean _lessThan = (_nameCount < 2);
    if (_lessThan) {
      return Paths.get("");
    }
    int _nameCount_1 = localOutputFilePath.getNameCount();
    int _minus = (_nameCount_1 - 1);
    return localOutputFilePath.subpath(0, _minus);
  }
  
  /**
   * Convenience for {@link AbstractSubGenerator#calculateOutputDirectory(String, String)},
   * uses default compiler ID.
   * 
   * TODO IDE-1487 currently there is no notion of default compiler. We fake call to the ES5 sub generator.
   */
  public static final String calculateProjectBasedOutputDirectory(final IN4JSProject project) {
    String _projectName = project.getProjectName();
    String _plus = (_projectName + "/");
    String _outputPath = project.getOutputPath();
    return (_plus + _outputPath);
  }
  
  /**
   * Access to compiler ID
   */
  public abstract String getCompilerID();
  
  /**
   * Access to compiler descriptor
   */
  protected abstract CompilerDescriptor getDefaultDescriptor();
  
  /**
   * Answers: Is this compiler activated for the input at hand?
   */
  public boolean isActive(final Resource input) {
    return (Boolean.valueOf(this.preferenceAccess.getPreference(input, this.getCompilerID(), CompilerProperties.IS_ACTIVE, this.getDefaultDescriptor()))).booleanValue();
  }
  
  /**
   * Checking the availability of a static polyfill, which will override the compilation of this module.
   */
  public boolean isNotStaticallyPolyfilled(final Resource resource) {
    if ((resource instanceof N4JSResource)) {
      final TModule tmodule = N4JSResource.getModule(resource);
      boolean _isStaticPolyfillAware = tmodule.isStaticPolyfillAware();
      if (_isStaticPolyfillAware) {
        boolean _hasStaticPolyfill = this.staticPolyfillHelper.hasStaticPolyfill(resource);
        if (_hasStaticPolyfill) {
          return false;
        }
      }
    }
    return true;
  }
  
  /**
   * Checking if this resource represents a static polyfill, which will contribute to a filled resource.
   */
  public boolean isStaticPolyfillingModule(final Resource resource) {
    final TModule tmodule = N4JSResource.getModule(resource);
    if ((null != tmodule)) {
      return tmodule.isStaticPolyfillModule();
    }
    return false;
  }
  
  /**
   * @return true if the composite generator is applicable to the given resource and false otherwise.
   */
  @Override
  public boolean isApplicableTo(final Resource input) {
    return this.shouldBeCompiled(input, null);
  }
  
  private final static Logger logger = Logger.getLogger(AbstractSubGenerator.class);
  
  public void setCompilerDescriptor(final CompilerDescriptor compilerDescriptor) {
    this.compilerDescriptor = compilerDescriptor;
  }
}
