/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesystem.utils;

import com.google.inject.Inject;
import java.util.List;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.ArrowFunction;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDefinition;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.ThisTarget;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TypableElement;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.utils.TypeExtensions;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelper;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelperStrategy;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * @see ThisTypeComputer#getThisTypeAtLocation(RuleEnvironment,EObject)
 */
@SuppressWarnings("all")
public class ThisTypeComputer extends TypeSystemHelperStrategy {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * Computes the this type at the given location. Since ECMAScript does not support lexical this typing, this
   * basically is a heuristic.
   * <p>
   * This code was moved here from judgment thisTypeRef in file 'n4js.xsemantics'.
   */
  public TypeRef getThisTypeAtLocation(final RuleEnvironment G, final EObject location) {
    if ((location instanceof ParameterizedTypeRef)) {
      return TypeUtils.createBoundThisTypeRef(((ParameterizedTypeRef)location));
    }
    final FunctionOrFieldAccessor containingFunctionOrAccessor = N4JSASTUtils.getContainingFunctionOrAccessor(location);
    if ((containingFunctionOrAccessor instanceof ArrowFunction)) {
      return this.getThisTypeAtLocation(G, containingFunctionOrAccessor);
    }
    IdentifiableElement _xifexpression = null;
    if ((containingFunctionOrAccessor != null)) {
      _xifexpression = containingFunctionOrAccessor.getDefinedFunctionOrAccessor();
    }
    final IdentifiableElement containingTFunctionOrAccessor = _xifexpression;
    final TypeRef declaredThisTypeRef = TypeSystemHelper.getDeclaredThisType(containingTFunctionOrAccessor);
    if ((declaredThisTypeRef != null)) {
      if ((declaredThisTypeRef instanceof ParameterizedTypeRef)) {
        return this.getThisTypeAtLocation(G, declaredThisTypeRef);
      }
      return declaredThisTypeRef;
    }
    final ThisTarget thisTarget = N4JSASTUtils.getProbableThisTarget(location);
    if ((thisTarget instanceof ObjectLiteral)) {
      return this.ts.type(G, ((TypableElement)thisTarget));
    } else {
      if ((thisTarget instanceof N4ClassifierDefinition)) {
        Type thisTargetDefType = ((N4ClassifierDefinition)thisTarget).getDefinedType();
        if ((thisTarget instanceof N4ClassDeclaration)) {
          final TClass clazz = ((N4ClassDeclaration)thisTarget).getDefinedTypeAsClass();
          if (((clazz != null) && clazz.isStaticPolyfill())) {
            final Type actualClazz = clazz.getSuperClassRef().getDeclaredType();
            if ((actualClazz != null)) {
              thisTargetDefType = actualClazz;
            }
          }
        }
        if ((thisTargetDefType != null)) {
          final FunctionDefinition containingFunction = N4JSASTUtils.getContainingFunction(location);
          if (((containingFunction instanceof N4MethodDeclaration) && 
            ((N4MemberDeclaration) containingFunction).isStatic())) {
            boolean _isInReturnDeclaration_Of_StaticMethod = RuleEnvironmentExtensions.isInReturnDeclaration_Of_StaticMethod(location, ((N4MethodDeclaration) containingFunction));
            if (_isInReturnDeclaration_Of_StaticMethod) {
              return this.getThisTypeAtLocation(G, this.createTypeRefWithParamsAsArgs(thisTargetDefType));
            } else {
              boolean _isInBody_Of_StaticMethod = RuleEnvironmentExtensions.isInBody_Of_StaticMethod(location, ((N4MethodDeclaration) containingFunction));
              if (_isInBody_Of_StaticMethod) {
                return TypeUtils.createClassifierBoundThisTypeRef(
                  TypeUtils.createTypeTypeRef(this.createTypeRefWithParamsAsArgs(thisTargetDefType), false));
              } else {
                return TypeUtils.createConstructorTypeRef(thisTargetDefType);
              }
            }
          } else {
            final N4FieldDeclaration n4Field = EcoreUtil2.<N4FieldDeclaration>getContainerOfType(location, N4FieldDeclaration.class);
            if (((n4Field != null) && n4Field.isStatic())) {
              return TypeRefsFactory.eINSTANCE.createUnknownTypeRef();
            } else {
              final N4GetterDeclaration n4Getter = EcoreUtil2.<N4GetterDeclaration>getContainerOfType(location, N4GetterDeclaration.class);
              if (((n4Getter != null) && n4Getter.isStatic())) {
                return TypeUtils.createConstructorTypeRef(thisTargetDefType);
              } else {
                final N4SetterDeclaration n4Setter = EcoreUtil2.<N4SetterDeclaration>getContainerOfType(location, N4SetterDeclaration.class);
                if (((n4Setter != null) && n4Setter.isStatic())) {
                  return TypeUtils.createConstructorTypeRef(thisTargetDefType);
                } else {
                  return this.getThisTypeAtLocation(G, this.createTypeRefWithParamsAsArgs(thisTargetDefType));
                }
              }
            }
          }
        } else {
          return RuleEnvironmentExtensions.anyTypeRefDynamic(G);
        }
      } else {
        boolean _hasGlobalObject = this.jsVariantHelper.hasGlobalObject(location);
        if (_hasGlobalObject) {
          return RuleEnvironmentExtensions.globalObjectTypeRef(G);
        }
        return RuleEnvironmentExtensions.undefinedTypeRef(G);
      }
    }
  }
  
  /**
   * Similar to utility methods [1] and [2], but if the given type is generic, then the generic type's
   * type parameters / type variables will be used as type arguments for the newly created ParameterizedTypeRef.
   * The utility methods [1] and [2] would instead either create a raw type reference or use wildcards as type
   * arguments.
   * <p>
   * [1] {@link TypeExtensions#ref(Type, TypeArgument...)}<br>
   * [2] {@link TypeUtils#createTypeRef(Type, TypingStrategy, boolean, TypeArgument...)}
   */
  private TypeRef createTypeRefWithParamsAsArgs(final Type type) {
    boolean _isGeneric = type.isGeneric();
    if (_isGeneric) {
      final Function1<TypeVariable, TypeRef> _function = (TypeVariable it) -> {
        return TypeExtensions.ref(it);
      };
      final List<TypeRef> typeArgs = IterableExtensions.<TypeRef>toList(ListExtensions.<TypeVariable, TypeRef>map(type.getTypeVars(), _function));
      return TypeExtensions.ref(type, ((TypeArgument[])Conversions.unwrapArray(typeArgs, TypeArgument.class)));
    }
    return TypeExtensions.ref(type);
  }
}
