/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import com.google.common.base.Strings;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.List;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.naming.N4JSQualifiedNameConverter;
import org.eclipse.n4js.naming.N4JSQualifiedNameProvider;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.semver.Semver.VersionNumber;
import org.eclipse.n4js.semver.Semver.VersionPart;
import org.eclipse.n4js.ts.scoping.N4TSQualifiedNameProvider;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeDefs;
import org.eclipse.n4js.utils.ProjectResolveHelper;
import org.eclipse.xtext.naming.IQualifiedNameConverter;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Helper class for computing descriptors for compiled files. Descriptors are used for file names and paths of generated files,
 * but also for meta data about them, e.g. for calculating import statements between compiled files or assembling code
 * execution runtime paths.
 * 
 * Note that paths computed here are relative to the output folder of the specific compiler, for a specific project.
 * Callers interested in full project relative path need to call concrete compiler for more information.
 */
@Singleton
@SuppressWarnings("all")
public final class ResourceNameComputer {
  /**
   * https://github.com/eclipse/n4js/issues/394
   * 
   * for simplifying node js compilation target we want to avoid project name and version in the compiled code location segments.
   * Instead we use node style specifiers that resolve to the project root.
   * Hide this behind the flag, as we anticipate that this needs to be configurable for other (than node.js) generators,
   * or we might make this configurable in the manifest.
   */
  private final static boolean MAKE_SIMPLE_DESCRIPTOR = true;
  
  /**
   * (pre-eclipse ticket)
   * IDE-2069
   * 
   * as part of adding node.js support, version information was suppressed in the compiled code segments.
   * Hidden behind feature flag, as we anticipate that this needs to be configurable for other (than node.js) generators,
   * or we might make this configurable in the manifest.
   */
  private final static boolean USE_PROJECT_VERSION = true;
  
  /**
   * Default value for descriptors to be generated in a way that is safe to be used as identifiers in JS.
   */
  private final static boolean AS_JS_IDENTIFIER = true;
  
  @Inject
  private N4JSQualifiedNameProvider qualifiedNameProvider;
  
  @Inject
  private IQualifiedNameConverter converter;
  
  @Inject
  private ProjectResolveHelper projectResolver;
  
  /**
   * The simple name of a declared type is the name which is used in its declaration. If type is anonymous, dummy name
   * will be generated.
   * <p>
   * Example: <code>C</code> for class C in file/module C in package p of project with version 1.0.0.
   * </p>
   */
  public String getSimpleTypeName(final Type type) {
    String name = type.getName();
    if (((name == null) || name.isEmpty())) {
      int _hashCode = type.hashCode();
      String _plus = ("__Anonymous_" + Integer.valueOf(_hashCode));
      name = _plus;
    }
    return name;
  }
  
  /**
   * The fully qualified name (FQN) of a declared type is its simple name, prefixed by the fully qualified module name
   * it is defined in.
   * <p>
   * Example: <code>p/C/C</code> for class C in file/module C in package p of project with version 1.0.0.
   * </p>
   */
  public String getFullyQualifiedTypeName(final Type type) {
    final EObject rootContainer = EcoreUtil.getRootContainer(type);
    if ((rootContainer instanceof TypeDefs)) {
      return this.getSimpleTypeName(type);
    }
    QualifiedName moduleFQN = this.qualifiedNameProvider.getFullyQualifiedName(rootContainer);
    boolean _isModulePolyfill = N4TSQualifiedNameProvider.isModulePolyfill(moduleFQN);
    if (_isModulePolyfill) {
      moduleFQN = moduleFQN.skipFirst(1);
    }
    return this.converter.toString(moduleFQN.append(this.getSimpleTypeName(type)));
  }
  
  /**
   * Like {@link #getFullyQualifiedTypeName(Type)}, but uses "." instead of the correct delimiter.
   * <p>
   * <b>THIS IS ONLY INTENDED FOR LEGACY PURPOSES WHEN CREATING THE QUALIFIED NAMES FOR THE META-DATA (e.g. N4Class in
   * transpiler, test catalog)!</b>
   * <p>
   * TODO IDE-2227 remove legacy support for old FQNs
   */
  public String getFullyQualifiedTypeName_WITH_LEGACY_SUPPORT(final Type type) {
    return this.getFullyQualifiedTypeName(type).replace(N4JSQualifiedNameConverter.DELIMITER, ".");
  }
  
  /**
   * The versioned module specifier is used only internally. It is derived from the module specifier with the version
   * (separated with a dash '-’) appended.
   * <p>
   * Based on provided file resource URI and extension will generate descriptor in form of
   * Project-0.0.1/module/path/Module Convenience method. Delegates to {@link ResourceNameComputer#formatDescriptor} For
   * delegation both project and unitPath are calculated from provided {@link TModule}.
   * <p>
   * Example: <code>project-1.0.0/p/C</code> for class C in file/module C in package p of project with version 1.0.0.
   * </p>
   * 
   * @module {@link TModule} for which we generate descriptor
   */
  public String getCompleteModuleSpecifier(final TModule module) {
    final IN4JSProject project = this.resolveProject(module);
    final String unitPath = module.getModuleSpecifier();
    return ResourceNameComputer.formatDescriptor(project, unitPath, "-", ".", "/", (!ResourceNameComputer.USE_PROJECT_VERSION), (!ResourceNameComputer.AS_JS_IDENTIFIER), ResourceNameComputer.MAKE_SIMPLE_DESCRIPTOR);
  }
  
  /**
   * Based on provided URI generates project in form of Project-0.0.1
   * Convenience method, delegates to {@link ResourceNameComputer#generateProjectDescriptor(IN4JSProject project)} with project
   * derived from the URI.
   * 
   * @n4jsSourceURI URI from file resource
   */
  public String generateProjectDescriptor(final URI n4jsSourceURI) {
    String _xblockexpression = null;
    {
      final IN4JSProject project = this.projectResolver.resolveProject(n4jsSourceURI);
      final String unitPath = "";
      _xblockexpression = ResourceNameComputer.formatDescriptor(project, unitPath, "-", ".", "", (!ResourceNameComputer.USE_PROJECT_VERSION), (!ResourceNameComputer.AS_JS_IDENTIFIER), (!ResourceNameComputer.MAKE_SIMPLE_DESCRIPTOR));
    }
    return _xblockexpression;
  }
  
  /**
   * Based on provided file resource URI and extension (must include dot!) will generate descriptor in form of Project/file/path/File.js
   * Convenience method. Delegates to {@link ResourceNameComputer#formatDescriptor}
   * For delegation both project and unitPath are calculated from provided URI.
   * 
   * @n4jsSourceURI URI from file resource
   * @fileExtension String containing desired extensions, should include dot
   */
  public String generateFileDescriptor(final Resource resource, final String fileExtension) {
    String _xblockexpression = null;
    {
      final IN4JSProject project = this.projectResolver.resolveProject(resource);
      final String unitPath = this.projectResolver.resolvePackageAndFileName(resource);
      String _formatDescriptor = ResourceNameComputer.formatDescriptor(project, unitPath, "-", ".", "/", (!ResourceNameComputer.USE_PROJECT_VERSION), (!ResourceNameComputer.AS_JS_IDENTIFIER), ResourceNameComputer.MAKE_SIMPLE_DESCRIPTOR);
      String _normalizeFileExtension = ResourceNameComputer.normalizeFileExtension(fileExtension);
      _xblockexpression = (_formatDescriptor + _normalizeFileExtension);
    }
    return _xblockexpression;
  }
  
  /**
   * Based on provided file resource URI and extension (must include dot!) will generate descriptor in form of Project/file/path/File.js
   * Convenience method. Delegates to {@link ResourceNameComputer#formatDescriptor}
   * For delegation both project and unitPath are calculated from provided URI.
   * 
   * @n4jsSourceURI URI from file resource
   * @fileExtension String containing desired extensions, should include dot
   */
  public String generateFileDescriptor(final URI n4jsSourceURI, final String fileExtension) {
    String _xblockexpression = null;
    {
      final IN4JSProject project = this.projectResolver.resolveProject(n4jsSourceURI);
      final String unitPath = this.projectResolver.resolvePackageAndFileName(n4jsSourceURI);
      String _formatDescriptor = ResourceNameComputer.formatDescriptor(project, unitPath, "-", ".", "/", (!ResourceNameComputer.USE_PROJECT_VERSION), (!ResourceNameComputer.AS_JS_IDENTIFIER), ResourceNameComputer.MAKE_SIMPLE_DESCRIPTOR);
      String _normalizeFileExtension = ResourceNameComputer.normalizeFileExtension(fileExtension);
      _xblockexpression = (_formatDescriptor + _normalizeFileExtension);
    }
    return _xblockexpression;
  }
  
  /**
   * Based on provided file resource URI and extension (must include dot!) will generate descriptor in form of Project/file/path/File.js
   * Convenience method. Delegates to {@link ResourceNameComputer#formatDescriptor}
   * For delegation both project and unitPath are calculated from provided URI.
   * 
   * @n4jsSourceURI URI from file resource
   * @fileExtension String containing desired extensions, should include dot
   */
  public String generateFileDescriptor(final IN4JSProject project, final URI n4jsSourceURI, final String fileExtension) {
    String _xblockexpression = null;
    {
      final String unitPath = this.projectResolver.resolvePackageAndFileName(n4jsSourceURI, project);
      String _formatDescriptor = ResourceNameComputer.formatDescriptor(project, unitPath, "-", ".", "/", (!ResourceNameComputer.USE_PROJECT_VERSION), (!ResourceNameComputer.AS_JS_IDENTIFIER), ResourceNameComputer.MAKE_SIMPLE_DESCRIPTOR);
      String _normalizeFileExtension = ResourceNameComputer.normalizeFileExtension(fileExtension);
      _xblockexpression = (_formatDescriptor + _normalizeFileExtension);
    }
    return _xblockexpression;
  }
  
  private IN4JSProject resolveProject(final TModule module) {
    return this.projectResolver.resolveProject(module.eResource().getURI());
  }
  
  /**
   * Simple normalization of provided file extension to form {@code .abc} or empty string.
   */
  private static String normalizeFileExtension(final String fileExtension) {
    boolean _isNullOrEmpty = Strings.isNullOrEmpty(fileExtension);
    if (_isNullOrEmpty) {
      return "";
    }
    boolean _startsWith = fileExtension.startsWith(".");
    if (_startsWith) {
      return fileExtension;
    }
    return ("." + fileExtension);
  }
  
  /**
   * Formats descriptor in form of
   * <pre>
   * projectName
   *  + sep1 + Project.declaredVersion.getMajor
   *  + sep2 + Project.declaredVersion.getMinor
   *  + sep2 + Project.declaredVersion.getMinor
   *  + sep3 + unitPath
   * </pre>
   * 
   * @param project  used to resolve declared version and project name.
   * @param unitPath  a path like string, can be derived from file Path, module specifier, or constructed manually.
   * @param useProjectVersion  tells if project version should be included or not. If false, then sep1 and sep2
   *                               will be ignored.
   * @param asJsIdentifier  tells if segments must be in form of a valid JS identifier.
   * @param makeSimpleDescriptor  tells if simple form of descriptor is to be used.
   */
  private static String formatDescriptor(final IN4JSProject project, final String unitPath, final String sep1, final String sep2, final String sep3, final boolean useProjectVersion, final boolean asJsIdentifier, final boolean makeSimpleDescriptor) {
    String projectName = project.getProjectName();
    String path = unitPath;
    if (asJsIdentifier) {
      projectName = ResourceNameComputer.getValidJavascriptIdentifierName(project.getProjectName());
      path = ResourceNameComputer.getValidUnitPath(unitPath);
    }
    if (makeSimpleDescriptor) {
      return path;
    }
    if (useProjectVersion) {
      String _projectVersionToStringWithoutQualifier = ResourceNameComputer.projectVersionToStringWithoutQualifier(project.getVersion(), sep2);
      String _plus = ((projectName + sep1) + _projectVersionToStringWithoutQualifier);
      String _plus_1 = (_plus + sep3);
      return (_plus_1 + path);
    }
    return ((projectName + sep3) + path);
  }
  
  /**
   * Ensures that all parts of the unit path are valid JS identifiers
   */
  private static String getValidUnitPath(final String unitPath) {
    final Function1<String, String> _function = (String it) -> {
      return ResourceNameComputer.getValidJavascriptIdentifierName(it);
    };
    return IterableExtensions.join(ListExtensions.<String, String>map(((List<String>)Conversions.doWrapArray(unitPath.split("/"))), _function), "/");
  }
  
  /**
   * Transforms given input string to form that can be used as JS identifier. Assumes that Java identifiers are valid JavaScript
   * identifiers, hence it actually just checks Java identifier validity and does not dive into JS specifics. Uses {@link Character#isJavaIdentifierStart}
   * and {@link Character#isJavaIdentifierPart} for heavy validity checks. Invalid characters are transformed to Unicode equivalents, see {@link #toUnicode}.
   * Valid inputs are returned unchanged.
   */
  private static String getValidJavascriptIdentifierName(final String input) {
    if (((input == null) || (input.length() == 0))) {
      return input;
    }
    final StringBuilder sb = new StringBuilder();
    for (int i = 0; (i < input.length()); i++) {
      {
        final char ch = input.charAt(i);
        boolean _xifexpression = false;
        if ((i == 0)) {
          _xifexpression = Character.isJavaIdentifierStart(ch);
        } else {
          _xifexpression = Character.isJavaIdentifierPart(ch);
        }
        final boolean isValid = _xifexpression;
        if (isValid) {
          sb.append(ch);
        } else {
          sb.append(ResourceNameComputer.toUnicode(ch));
        }
      }
    }
    return sb.toString();
  }
  
  private static String toUnicode(final char character) {
    String _substring = Integer.toHexString((character | 0x10000)).substring(1);
    return ("_u" + _substring);
  }
  
  /**
   * Transforms the version into a string used for variable, parameter, and file names.
   */
  private static String projectVersionToStringWithoutQualifier(final VersionNumber declaredVersion, final String separatorChar) {
    VersionPart _major = declaredVersion.getMajor();
    String _plus = (_major + separatorChar);
    VersionPart _minor = declaredVersion.getMinor();
    String _plus_1 = (_plus + _minor);
    String _plus_2 = (_plus_1 + separatorChar);
    VersionPart _patch = declaredVersion.getPatch();
    return (_plus_2 + _patch);
  }
}
