/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.n4js.n4JS.ArrayBindingPattern;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.BindingPattern;
import org.eclipse.n4js.n4JS.BindingProperty;
import org.eclipse.n4js.n4JS.DestructNode;
import org.eclipse.n4js.n4JS.DestructureUtils;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ForStatement;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.ObjectBindingPattern;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.PropertyNameValuePairSingleName;
import org.eclipse.n4js.n4JS.VariableBinding;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.Result;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.DestructureHelper;
import org.eclipse.n4js.utils.UtilN4;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Validations within destructuring patterns.
 */
@SuppressWarnings("all")
public class N4JSDestructureValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private DestructureHelper destructureHelper;
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  @Check
  public void checkNoEmptyPattern_Binding(final BindingPattern pattern) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (pattern instanceof ArrayBindingPattern) {
      _matched=true;
      _switchResult = ((ArrayBindingPattern)pattern).getElements().isEmpty();
    }
    if (!_matched) {
      if (pattern instanceof ObjectBindingPattern) {
        _matched=true;
        _switchResult = ((ObjectBindingPattern)pattern).getProperties().isEmpty();
      }
    }
    final boolean isEmpty = _switchResult;
    if (isEmpty) {
      final String message = IssueCodes.getMessageForDESTRUCT_EMPTY_PATTERN();
      this.addIssue(message, pattern, IssueCodes.DESTRUCT_EMPTY_PATTERN);
    }
  }
  
  @Check
  public void checkNoEmptyPattern_Assignment(final AssignmentExpression expr) {
    boolean _isTopOfDestructuringAssignment = DestructureUtils.isTopOfDestructuringAssignment(expr);
    if (_isTopOfDestructuringAssignment) {
      final Expression lhs = expr.getLhs();
      boolean _switchResult = false;
      boolean _matched = false;
      if (lhs instanceof ArrayLiteral) {
        _matched=true;
        _switchResult = ((ArrayLiteral)lhs).getElements().isEmpty();
      }
      if (!_matched) {
        if (lhs instanceof ObjectLiteral) {
          _matched=true;
          _switchResult = IterableExtensions.isEmpty(Iterables.<PropertyNameValuePair>filter(((ObjectLiteral)lhs).getPropertyAssignments(), PropertyNameValuePair.class));
        }
      }
      final boolean empty = _switchResult;
      if (empty) {
        final String message = IssueCodes.getMessageForDESTRUCT_EMPTY_PATTERN();
        this.addIssue(message, lhs, IssueCodes.DESTRUCT_EMPTY_PATTERN);
      }
    }
  }
  
  @Check
  public void checkTypesInDestructPatternInVariableBinding(final VariableBinding binding) {
    this.internal_checkDestructPattern(DestructNode.unify(binding), binding);
  }
  
  @Check
  public void checkTypesInDestructPatternInAssignmentExpression(final AssignmentExpression expr) {
    this.internal_checkDestructPattern(DestructNode.unify(expr), expr);
  }
  
  @Check
  public void checkTypesInDestructPatternInForInOfStatement(final ForStatement stmnt) {
    this.internal_checkDestructPattern(DestructNode.unify(stmnt), stmnt);
  }
  
  private void internal_checkDestructPattern(final DestructNode rootNode, final EObject contextObject) {
    if ((rootNode == null)) {
      return;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(contextObject);
    final HashMap<DestructNode, TypeRef> valueTypePerNode = CollectionLiterals.<DestructNode, TypeRef>newHashMap();
    this.destructureHelper.buildValueTypesMap(G, rootNode, valueTypePerNode, contextObject);
    this.internal_checkDestructNode(G, null, rootNode, null, valueTypePerNode, contextObject);
  }
  
  /**
   * @param parentNode
   *             parent node of {@code node} or <code>null</code> if {@code node} is a root.
   * @param parentMemberScope
   *             member scope of parentNode or <code>null</code> if {@code node} is a root.
   */
  private void internal_checkDestructNode(final RuleEnvironment G, final DestructNode parentNode, final DestructNode node, final IScope parentMemberScope, final Map<DestructNode, TypeRef> valueTypePerNode, final EObject contextObject) {
    boolean isValid = (this.holdsValidPropertyAccessInDestructNode(G, parentNode, node, parentMemberScope, valueTypePerNode) && this.holdsCorrectTypeInDestructNode(G, parentNode, node, valueTypePerNode));
    if (((isValid && (node.getNestedNodes() != null)) && (!((List<DestructNode>)Conversions.doWrapArray(node.getNestedNodes())).isEmpty()))) {
      final TypeRef valueTypeRef = valueTypePerNode.get(node);
      if ((valueTypeRef != null)) {
        final IScope memberScope = this.createMemberScope(node, valueTypeRef, contextObject);
        DestructNode[] _nestedNodes = node.getNestedNodes();
        for (final DestructNode childNode : _nestedNodes) {
          if ((childNode != null)) {
            this.internal_checkDestructNode(G, node, childNode, memberScope, valueTypePerNode, contextObject);
          }
        }
      }
    }
  }
  
  private boolean holdsValidPropertyAccessInDestructNode(final RuleEnvironment G, final DestructNode parentNode, final DestructNode node, final IScope parentMemberScope, final Map<DestructNode, TypeRef> valueTypePerNode) {
    if (((node.getPropName() != null) && (parentMemberScope != null))) {
      final StringBuffer errMsg = new StringBuffer();
      final TypeRef propTypeRef = this.destructureHelper.getPropertyTypeForNode(G, valueTypePerNode.get(parentNode), parentMemberScope, node.getPropName(), errMsg);
      int _length = errMsg.length();
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        final String msg = IssueCodes.getMessageForDESTRUCT_PROP_WITH_ERROR(node.getPropName(), UtilN4.trimSuffix(errMsg.toString().trim(), "."));
        final Pair<EObject, EStructuralFeature> astNodeOfPropName = node.getEObjectAndFeatureForPropName();
        this.addIssue(msg, astNodeOfPropName.getKey(), astNodeOfPropName.getValue(), IssueCodes.DESTRUCT_PROP_WITH_ERROR);
        return false;
      } else {
        if ((propTypeRef == null)) {
          TypeRef _get = valueTypePerNode.get(parentNode);
          String _typeRefAsString = null;
          if (_get!=null) {
            _typeRefAsString=_get.getTypeRefAsString();
          }
          final String msg_1 = IssueCodes.getMessageForDESTRUCT_PROP_MISSING(node.getPropName(), _typeRefAsString);
          final Pair<EObject, EStructuralFeature> astNodeOfPropName_1 = node.getEObjectAndFeatureForPropName();
          this.addIssue(msg_1, astNodeOfPropName_1.getKey(), astNodeOfPropName_1.getValue(), IssueCodes.DESTRUCT_PROP_MISSING);
          return false;
        }
      }
    }
    return true;
  }
  
  /**
   * @param parentNode
   *             parent node of {@code node} or <code>null</code> if {@code node} is a root.
   */
  private boolean holdsCorrectTypeInDestructNode(final RuleEnvironment G, final DestructNode parentNode, final DestructNode node, final Map<DestructNode, TypeRef> valueTypePerNode) {
    final TypeRef valueTypeRef = valueTypePerNode.get(node);
    if ((valueTypeRef == null)) {
      return true;
    }
    if (((node.getVarDecl() != null) || (node.getVarRef() != null))) {
      if (((node.getVarDecl() != null) && (node.getVarDecl().getDeclaredTypeRef() == null))) {
      } else {
        TypeRef _xifexpression = null;
        VariableDeclaration _varDecl = node.getVarDecl();
        boolean _tripleNotEquals = (_varDecl != null);
        if (_tripleNotEquals) {
          _xifexpression = this.ts.type(G, node.getVarDecl());
        } else {
          TypeRef _xifexpression_1 = null;
          IdentifierRef _varRef = node.getVarRef();
          boolean _tripleNotEquals_1 = (_varRef != null);
          if (_tripleNotEquals_1) {
            _xifexpression_1 = this.ts.type(G, node.getVarRef());
          }
          _xifexpression = _xifexpression_1;
        }
        TypeRef variableTypeRef = _xifexpression;
        Expression _defaultExpr = node.getDefaultExpr();
        boolean _tripleNotEquals_2 = (_defaultExpr != null);
        if (_tripleNotEquals_2) {
          final TypeRef defaultExprTypeRef = this.ts.type(G, node.getDefaultExpr());
          boolean _xifexpression_2 = false;
          if ((defaultExprTypeRef != null)) {
            _xifexpression_2 = this.ts.subtypeSucceeded(G, defaultExprTypeRef, variableTypeRef);
          }
          final boolean isOfCorrectType = _xifexpression_2;
          if ((!isOfCorrectType)) {
            return false;
          }
        }
        final Result result = this.ts.subtype(G, valueTypeRef, variableTypeRef);
        boolean _isFailure = result.isFailure();
        if (_isFailure) {
          String _elvis = null;
          String _elvis_1 = null;
          VariableDeclaration _varDecl_1 = node.getVarDecl();
          String _name = null;
          if (_varDecl_1!=null) {
            _name=_varDecl_1.getName();
          }
          if (_name != null) {
            _elvis_1 = _name;
          } else {
            IdentifierRef _varRef_1 = node.getVarRef();
            IdentifiableElement _id = null;
            if (_varRef_1!=null) {
              _id=_varRef_1.getId();
            }
            String _name_1 = null;
            if (_id!=null) {
              _name_1=_id.getName();
            }
            _elvis_1 = _name_1;
          }
          if (_elvis_1 != null) {
            _elvis = _elvis_1;
          } else {
            _elvis = "<unnamed>";
          }
          final String varName = _elvis;
          String _xifexpression_3 = null;
          boolean _isPositional = node.isPositional();
          if (_isPositional) {
            int _indexOf = ((List<DestructNode>)Conversions.doWrapArray(parentNode.getNestedNodes())).indexOf(node);
            _xifexpression_3 = ("at index " + Integer.valueOf(_indexOf));
          } else {
            String _propName = node.getPropName();
            String _plus = ("of property \'" + _propName);
            _xifexpression_3 = (_plus + "\'");
          }
          String elemDesc = _xifexpression_3;
          String tsMsg = UtilN4.trimSuffix(UtilN4.trimPrefix(result.getFailureMessage(), "failed: "), ".");
          final String msg = IssueCodes.getMessageForDESTRUCT_TYPE_ERROR_VAR(varName, elemDesc, tsMsg);
          VariableDeclaration _varDecl_2 = node.getVarDecl();
          boolean _tripleNotEquals_3 = (_varDecl_2 != null);
          if (_tripleNotEquals_3) {
            this.addIssue(msg, node.getVarDecl(), TypesPackage.eINSTANCE.getIdentifiableElement_Name(), IssueCodes.DESTRUCT_TYPE_ERROR_VAR);
          } else {
            this.addIssue(msg, node.getVarRef(), IssueCodes.DESTRUCT_TYPE_ERROR_VAR);
          }
          return false;
        }
      }
    } else {
      DestructNode[] _nestedNodes = node.getNestedNodes();
      boolean _tripleNotEquals_4 = (_nestedNodes != null);
      if (_tripleNotEquals_4) {
        final boolean isPositional = DestructNode.arePositional(node.getNestedNodes());
        ParameterizedTypeRef _xifexpression_4 = null;
        if (isPositional) {
          _xifexpression_4 = RuleEnvironmentExtensions.iterableTypeRef(G, TypeRefsFactory.eINSTANCE.createWildcard());
        } else {
          _xifexpression_4 = RuleEnvironmentExtensions.objectTypeRef(G);
        }
        final ParameterizedTypeRef expectedTypeRef = _xifexpression_4;
        final Result result_1 = this.ts.subtype(G, this.autoboxIfPrimitive(valueTypeRef), expectedTypeRef);
        boolean _isFailure_1 = result_1.isFailure();
        if (_isFailure_1) {
          String _xifexpression_5 = null;
          if (isPositional) {
            String _xifexpression_6 = null;
            if ((parentNode != null)) {
              _xifexpression_6 = "Nested array";
            } else {
              _xifexpression_6 = "Array";
            }
            _xifexpression_5 = _xifexpression_6;
          } else {
            String _xifexpression_7 = null;
            if ((parentNode != null)) {
              _xifexpression_7 = "Nested object";
            } else {
              _xifexpression_7 = "Object";
            }
            _xifexpression_5 = _xifexpression_7;
          }
          final String patternKind = _xifexpression_5;
          String _xifexpression_8 = null;
          if ((parentNode != null)) {
            String _xifexpression_9 = null;
            boolean _isPositional_1 = node.isPositional();
            if (_isPositional_1) {
              int _indexOf_1 = ((List<DestructNode>)Conversions.doWrapArray(parentNode.getNestedNodes())).indexOf(node);
              _xifexpression_9 = ("destructured value at index " + Integer.valueOf(_indexOf_1));
            } else {
              String _propName_1 = node.getPropName();
              String _plus_1 = ("destructured value of property \'" + _propName_1);
              _xifexpression_9 = (_plus_1 + "\'");
            }
            _xifexpression_8 = _xifexpression_9;
          } else {
            String _typeRefAsString = valueTypeRef.getTypeRefAsString();
            String _plus_2 = ("a value of type \'" + _typeRefAsString);
            _xifexpression_8 = (_plus_2 + "\'");
          }
          String elemDesc_1 = _xifexpression_8;
          String tsMsg_1 = UtilN4.trimSuffix(UtilN4.trimPrefix(result_1.getFailureMessage(), "failed: "), ".");
          final String msg_1 = IssueCodes.getMessageForDESTRUCT_TYPE_ERROR_PATTERN(patternKind, elemDesc_1, tsMsg_1);
          final EObject astElem = node.getAstElement();
          boolean _matched = false;
          if (astElem instanceof PropertyNameValuePair) {
            if ((!(astElem instanceof PropertyNameValuePairSingleName))) {
              _matched=true;
              this.addIssue(msg_1, astElem, N4JSPackage.eINSTANCE.getPropertyNameValuePair_Expression(), IssueCodes.DESTRUCT_TYPE_ERROR_PATTERN);
            }
          }
          if (!_matched) {
            if (astElem instanceof BindingProperty) {
              _matched=true;
              this.addIssue(msg_1, astElem, N4JSPackage.eINSTANCE.getBindingProperty_Value(), IssueCodes.DESTRUCT_TYPE_ERROR_PATTERN);
            }
          }
          if (!_matched) {
            this.addIssue(msg_1, astElem, IssueCodes.DESTRUCT_TYPE_ERROR_PATTERN);
          }
          return false;
        }
      }
    }
    return true;
  }
  
  /**
   * Create a member scope for looking up property names of the <em>children</em> of {@code node}, i.e. the {@code propName}
   * attribute of the {@code nestedNodes} of {@code node}. Will return <code>null</code> if node does not have nested nodes
   * or if they are positional (because then property look-up does not make sense).
   */
  private IScope createMemberScope(final DestructNode node, final TypeRef valueTypeRef, final EObject contextObject) {
    IScope _xifexpression = null;
    if (((!((List<DestructNode>)Conversions.doWrapArray(node.getNestedNodes())).isEmpty()) && (!DestructNode.arePositional(node.getNestedNodes())))) {
      _xifexpression = this.destructureHelper.createMemberScopeForPropertyAccess(valueTypeRef, contextObject, true);
    } else {
      _xifexpression = null;
    }
    return _xifexpression;
  }
  
  private TypeRef autoboxIfPrimitive(final TypeRef typeRef) {
    final Type declType = typeRef.getDeclaredType();
    if ((declType instanceof PrimitiveType)) {
      TClassifier _autoboxedType = ((PrimitiveType)declType).getAutoboxedType();
      boolean _tripleNotEquals = (_autoboxedType != null);
      if (_tripleNotEquals) {
        return TypeUtils.createTypeRef(((PrimitiveType)declType).getAutoboxedType());
      }
    }
    return typeRef;
  }
}
