/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.assistants;

import org.eclipse.n4js.n4JS.ArrayBindingPattern;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.BindingElement;
import org.eclipse.n4js.n4JS.BindingPattern;
import org.eclipse.n4js.n4JS.BindingProperty;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ObjectBindingPattern;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.PrimaryExpression;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableDeclarationOrBinding;
import org.eclipse.n4js.transpiler.TransformationAssistant;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * A {@link TransformationAssistant} providing helper functionality for dealing with ES2015 destructuring.
 */
@SuppressWarnings("all")
public class DestructuringAssistant extends TransformationAssistant {
  /**
   * Converts the given array or object binding pattern into an array or object literal that, if used on the
   * right-hand side of an assignment expression, performs the equivalent destructuring operation.
   * <p>
   * Expression for default values are removed from the given binding, so the given binding is incomplete after this
   * method returns. It is only guaranteed that (1) the given binding is not removed from its contained and (2) it
   * will still include the same variable declarations as before, which can be retrieved via
   * {@link VariableDeclarationOrBinding#getVariableDeclarations()} on the containing variable binding.
   */
  public PrimaryExpression convertBindingPatternToArrayOrObjectLiteral(final BindingPattern binding) {
    PrimaryExpression _switchResult = null;
    boolean _matched = false;
    if (binding instanceof ArrayBindingPattern) {
      _matched=true;
      _switchResult = this.convertArrayBindingPatternToArrayLiteral(((ArrayBindingPattern)binding));
    }
    if (!_matched) {
      if (binding instanceof ObjectBindingPattern) {
        _matched=true;
        _switchResult = this.convertObjectBindingPatternToObjectLiteral(((ObjectBindingPattern)binding));
      }
    }
    return _switchResult;
  }
  
  /**
   * Same as {@link #convertBindingPatternToArrayOrObjectLiteral(BindingPattern)}, but only for array binding
   * patterns.
   */
  public ArrayLiteral convertArrayBindingPatternToArrayLiteral(final ArrayBindingPattern binding) {
    final Function1<BindingElement, ArrayElement> _function = (BindingElement it) -> {
      return this.convertBindingElementToArrayElement(it);
    };
    return TranspilerBuilderBlocks._ArrLit(
      ((ArrayElement[])Conversions.unwrapArray(ListExtensions.<BindingElement, ArrayElement>map(binding.getElements(), _function), ArrayElement.class)));
  }
  
  /**
   * Same as {@link #convertBindingPatternToArrayOrObjectLiteral(BindingPattern)}, but only for object binding
   * patterns.
   */
  public ObjectLiteral convertObjectBindingPatternToObjectLiteral(final ObjectBindingPattern binding) {
    final Function1<BindingProperty, PropertyNameValuePair> _function = (BindingProperty it) -> {
      return this.convertBindingPropertyToPropertyNameValuePair(it);
    };
    return TranspilerBuilderBlocks._ObjLit(
      ((PropertyAssignment[])Conversions.unwrapArray(ListExtensions.<BindingProperty, PropertyNameValuePair>map(binding.getProperties(), _function), PropertyAssignment.class)));
  }
  
  private ArrayElement convertBindingElementToArrayElement(final BindingElement element) {
    final BindingPattern nestedPattern = element.getNestedPattern();
    final VariableDeclaration varDecl = element.getVarDecl();
    Expression lhs = null;
    Expression rhs = null;
    if ((nestedPattern != null)) {
      lhs = this.convertBindingPatternToArrayOrObjectLiteral(nestedPattern);
      rhs = element.getExpression();
    } else {
      if ((varDecl != null)) {
        final SymbolTableEntry ste_varDecl = this.findSymbolTableEntryForElement(varDecl, true);
        lhs = TranspilerBuilderBlocks._IdentRef(ste_varDecl);
        rhs = varDecl.getExpression();
      } else {
        return TranspilerBuilderBlocks._ArrayPadding();
      }
    }
    boolean _isRest = element.isRest();
    Expression _xifexpression = null;
    if ((rhs != null)) {
      _xifexpression = TranspilerBuilderBlocks._AssignmentExpr(lhs, rhs);
    } else {
      _xifexpression = lhs;
    }
    return TranspilerBuilderBlocks._ArrayElement(_isRest, _xifexpression);
  }
  
  private PropertyNameValuePair convertBindingPropertyToPropertyNameValuePair(final BindingProperty property) {
    return TranspilerBuilderBlocks._PropertyNameValuePair(
      property.getName(), 
      this.convertBindingElementToArrayElement(property.getValue()).getExpression());
  }
}
