/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.n4idl;

import java.util.Collections;
import java.util.function.Consumer;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.MigrationContextVariable;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4idl.N4IDLGlobals;
import org.eclipse.n4js.n4idl.versioning.MigrationUtils;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.im.IdentifierRef_IM;
import org.eclipse.n4js.transpiler.im.ParameterizedPropertyAccessExpression_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryInternal;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TMigration;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * N4IDL Migration declaration transformation which replaces all MigrationContext
 * references with 'this' references and replaces 'migrate' calls with migrate-calls
 * on the context.
 */
@SuppressWarnings("all")
public class N4IDLMigrationTransformation extends Transformation {
  @Override
  public void transform() {
    final Function1<FunctionDeclaration, Boolean> _function = (FunctionDeclaration m) -> {
      return Boolean.valueOf(MigrationUtils.isMigrationDefinition(m));
    };
    final Consumer<FunctionDeclaration> _function_1 = (FunctionDeclaration m) -> {
      this.transformMigrationDeclaration(m);
    };
    IterableExtensions.<FunctionDeclaration>filter(EcoreUtil2.<FunctionDeclaration>getAllContentsOfType(this.getState().im, FunctionDeclaration.class), _function).forEach(_function_1);
  }
  
  private void transformMigrationDeclaration(final FunctionDeclaration migrationDeclaration) {
    final Consumer<IdentifierRef_IM> _function = (IdentifierRef_IM identifier) -> {
      this.transformIdentifierReference(identifier);
    };
    EcoreUtil2.<IdentifierRef_IM>getAllContentsOfType(migrationDeclaration, IdentifierRef_IM.class).forEach(_function);
    final Consumer<ParameterizedCallExpression> _function_1 = (ParameterizedCallExpression callExpr) -> {
      this.transformCallExpression(this.findTMigration(migrationDeclaration), callExpr);
    };
    EcoreUtil2.<ParameterizedCallExpression>getAllContentsOfType(migrationDeclaration, ParameterizedCallExpression.class).forEach(_function_1);
  }
  
  /**
   * Obtains the type-model {@link TMigration} instance for the given migration declaration.
   * 
   * Raises an {@link IllegalStateException} if no {@link TMigration} can be obtained from
   * the given {@link FunctionDeclaration}.
   */
  private TMigration findTMigration(final FunctionDeclaration migrationDeclaration) {
    final SymbolTableEntry declarationSTE = this.getState().steCache.mapNamedElement_2_STE.get(migrationDeclaration);
    if ((!(declarationSTE instanceof SymbolTableEntryOriginal))) {
      throw new IllegalStateException(("Failed to obtain SymbolTableEntryOriginal for migration declaration " + migrationDeclaration));
    }
    IdentifiableElement _originalTarget = ((SymbolTableEntryOriginal) declarationSTE).getOriginalTarget();
    boolean _not = (!(_originalTarget instanceof TMigration));
    if (_not) {
      throw new IllegalStateException(("Encountered malformed migration declaration in transpiler: " + migrationDeclaration));
    }
    IdentifiableElement _originalTarget_1 = ((SymbolTableEntryOriginal) declarationSTE).getOriginalTarget();
    return ((TMigration) _originalTarget_1);
  }
  
  /**
   * Replaces the given {@link IdentifierRef} with a {@code this.context} expression,
   * if it refers to a {@link MigrationContextVariable}.
   */
  private void transformIdentifierReference(final IdentifierRef_IM ref) {
    final SymbolTableEntry refSTE = ref.getId_IM();
    if (((refSTE instanceof SymbolTableEntryOriginal) && (((SymbolTableEntryOriginal) refSTE).getOriginalTarget() instanceof MigrationContextVariable))) {
      final SymbolTableEntryInternal contextSTE = this.getSymbolTableEntryInternal(N4IDLGlobals.MIGRATION_CONTROLLER_CONTEXT_PROPERTY_NAME, true);
      final ParameterizedPropertyAccessExpression_IM contextAccess = TranspilerBuilderBlocks._PropertyAccessExpr(TranspilerBuilderBlocks._ThisLiteral(), contextSTE);
      this.replace(ref, contextAccess);
    }
  }
  
  /**
   * Transforms the given call expression, in case the call-expression refers to a migration (implicitly via {@code migrate}-call
   * or explicitly via an explicit reference to a function annotated as {@code @Migration}).
   */
  private void transformCallExpression(final TMigration contextMigration, final ParameterizedCallExpression callExpression) {
    boolean _isMigrateCall = MigrationUtils.isMigrateCall(callExpression);
    if (_isMigrateCall) {
      this.transformMigrateCallExpression(contextMigration, callExpression);
    } else {
      final Expression callTarget = callExpression.getTarget();
      if ((callTarget instanceof ParameterizedPropertyAccessExpression_IM)) {
        boolean _refersToMigration = this.refersToMigration(((ParameterizedPropertyAccessExpression_IM)callTarget).getRewiredTarget());
        if (_refersToMigration) {
          this.transformExplicitMigrationCallExpression(contextMigration, callExpression);
        }
      }
      if ((callTarget instanceof IdentifierRef_IM)) {
        boolean _refersToMigration_1 = this.refersToMigration(((IdentifierRef_IM)callTarget).getRewiredTarget());
        if (_refersToMigration_1) {
          this.transformExplicitMigrationCallExpression(contextMigration, callExpression);
        }
      }
    }
  }
  
  /**
   * Returns {@code true} iff the given {@code ste} refers to a {@link TMigration}.
   */
  private boolean refersToMigration(final SymbolTableEntry ste) {
    if ((ste instanceof SymbolTableEntryOriginal)) {
      final IdentifiableElement originalTarget = ((SymbolTableEntryOriginal)ste).getOriginalTarget();
      return (originalTarget instanceof TMigration);
    }
    return false;
  }
  
  /**
   * Transforms the given {@code migrate} call expression.
   */
  private void transformMigrateCallExpression(final TMigration contextMigration, final ParameterizedCallExpression callExpression) {
    final ParameterizedCallExpression transpiledCall = TranspilerBuilderBlocks._CallExpr();
    final SymbolTableEntryInternal migrateSTE = this.getSymbolTableEntryInternal(N4IDLGlobals.MIGRATION_CALL_IDENTIFIER, true);
    transpiledCall.setTarget(TranspilerBuilderBlocks._PropertyAccessExpr(TranspilerBuilderBlocks._ThisLiteral(), migrateSTE));
    final Function1<Argument, ArrayElement> _function = (Argument a) -> {
      return TranspilerBuilderBlocks._ArrayElement(a.getExpression());
    };
    Argument __Argument = TranspilerBuilderBlocks._Argument(TranspilerBuilderBlocks._ArrLit(((ArrayElement[])Conversions.unwrapArray(ListExtensions.<Argument, ArrayElement>map(callExpression.getArguments(), _function), ArrayElement.class))));
    transpiledCall.getArguments().addAll(
      Collections.<Argument>unmodifiableList(CollectionLiterals.<Argument>newArrayList(__Argument)));
    this.replace(callExpression, transpiledCall);
  }
  
  /**
   * Transforms explicit calls of migration functions to migrate-calls via the
   * migration controller (e.g. {@code migrateA(a.b)} -> {@code this.migrateWith(migrateA, [a.b])}).
   */
  private void transformExplicitMigrationCallExpression(final TMigration contextMigration, final ParameterizedCallExpression callExpression) {
    final ParameterizedCallExpression transpiledCall = TranspilerBuilderBlocks._CallExpr();
    final SymbolTableEntryInternal migrateWithSTE = this.getSymbolTableEntryInternal(N4IDLGlobals.MIGRATION_CONTROLLER_MIGRATE_WITH_FUNCTION_NAME, true);
    transpiledCall.setTarget(TranspilerBuilderBlocks._PropertyAccessExpr(TranspilerBuilderBlocks._ThisLiteral(), migrateWithSTE));
    Argument __Argument = TranspilerBuilderBlocks._Argument(callExpression.getTarget());
    final Function1<Argument, ArrayElement> _function = (Argument a) -> {
      return TranspilerBuilderBlocks._ArrayElement(a.getExpression());
    };
    Argument __Argument_1 = TranspilerBuilderBlocks._Argument(TranspilerBuilderBlocks._ArrLit(((ArrayElement[])Conversions.unwrapArray(ListExtensions.<Argument, ArrayElement>map(callExpression.getArguments(), _function), ArrayElement.class))));
    transpiledCall.getArguments().addAll(
      Collections.<Argument>unmodifiableList(CollectionLiterals.<Argument>newArrayList(__Argument, __Argument_1)));
    this.replace(callExpression, transpiledCall);
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void assertPreConditions() {
  }
}
