/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collections;
import java.util.function.Consumer;
import java.util.stream.Stream;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.compare.ProjectComparisonEntry;
import org.eclipse.n4js.n4JS.AnnotableScriptElement;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4EnumLiteral;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.ScriptElement;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TransformationDependency;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.assistants.TypeAssistant;
import org.eclipse.n4js.transpiler.es.assistants.DelegationAssistant;
import org.eclipse.n4js.transpiler.es.transform.MemberPatchingTransformation;
import org.eclipse.n4js.transpiler.im.DelegatingMember;
import org.eclipse.n4js.transpiler.im.Script_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryIMOnly;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryInternal;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.transpiler.utils.ConcreteMembersOrderedForTranspiler;
import org.eclipse.n4js.transpiler.utils.MissingApiMembersForTranspiler;
import org.eclipse.n4js.transpiler.utils.ScriptApiTracker;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TAnnotation;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TEnumLiteral;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.util.AccessorTuple;
import org.eclipse.n4js.ts.types.util.MemberList;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelper;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Generation of code for missing implementations in projects implementing a specific API.
 */
@TransformationDependency.RequiresBefore(MemberPatchingTransformation.class)
@SuppressWarnings("all")
public class ApiImplStubGenerationTransformation extends Transformation {
  @Inject
  private DelegationAssistant delegationAssistant;
  
  @Inject
  private TypeAssistant typeAssistant;
  
  @Inject
  private ScriptApiTracker scriptApiTracker;
  
  @Inject
  private ContainerTypesHelper containerTypesHelper;
  
  @Inject
  private N4JSElementKeywordProvider n4jsElementKeywordProvider;
  
  @Override
  public void assertPreConditions() {
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void analyze() {
    this.scriptApiTracker.initApiCompare(this.getState().resource.getScript());
  }
  
  @Override
  public void transform() {
    final Consumer<N4ClassifierDeclaration> _function = (N4ClassifierDeclaration it) -> {
      this.addMissingMembers(it);
    };
    this.<N4ClassifierDeclaration>collectNodes(this.getState().im, N4ClassifierDeclaration.class, false).forEach(_function);
    this.addMissingTopLevelElements();
  }
  
  private void addMissingMembers(final N4ClassifierDeclaration classifierDecl) {
    final TClassifier type = this.getState().info.getOriginalDefinedType(classifierDecl);
    final MissingApiMembersForTranspiler mamft = this.createMAMFT(type);
    for (final TMethod m : mamft.missingApiMethods) {
      {
        final N4MemberDeclaration member = this.createApiImplStub(classifierDecl, m);
        EList<N4MemberDeclaration> _ownedMembersRaw = classifierDecl.getOwnedMembersRaw();
        _ownedMembersRaw.add(member);
      }
    }
    for (final AccessorTuple accTuple : mamft.missingApiAccessorTuples) {
      {
        TGetter _getter = accTuple.getGetter();
        boolean _tripleNotEquals = (_getter != null);
        if (_tripleNotEquals) {
          final TGetter g = accTuple.getGetter();
          final N4MemberDeclaration member = this.createApiImplStub(classifierDecl, g);
          EList<N4MemberDeclaration> _ownedMembersRaw = classifierDecl.getOwnedMembersRaw();
          _ownedMembersRaw.add(member);
        }
        TSetter _setter = accTuple.getSetter();
        boolean _tripleNotEquals_1 = (_setter != null);
        if (_tripleNotEquals_1) {
          final TSetter s = accTuple.getSetter();
          final N4MemberDeclaration member_1 = this.createApiImplStub(classifierDecl, s);
          EList<N4MemberDeclaration> _ownedMembersRaw_1 = classifierDecl.getOwnedMembersRaw();
          _ownedMembersRaw_1.add(member_1);
        }
      }
    }
    for (final AccessorTuple accTuple_1 : mamft.missingApiAccessorTuples) {
      {
        if ((((accTuple_1.getInheritedGetter() != null) && (accTuple_1.getGetter() == null)) && (accTuple_1.getSetter() != null))) {
          final DelegatingMember delegator = this.delegationAssistant.createDelegatingMember(type, accTuple_1.getInheritedGetter());
          EList<N4MemberDeclaration> _ownedMembersRaw = classifierDecl.getOwnedMembersRaw();
          _ownedMembersRaw.add(delegator);
        }
        if ((((accTuple_1.getInheritedSetter() != null) && (accTuple_1.getGetter() != null)) && (accTuple_1.getSetter() == null))) {
          final DelegatingMember delegator_1 = this.delegationAssistant.createDelegatingMember(type, accTuple_1.getInheritedSetter());
          EList<N4MemberDeclaration> _ownedMembersRaw_1 = classifierDecl.getOwnedMembersRaw();
          _ownedMembersRaw_1.add(delegator_1);
        }
      }
    }
  }
  
  private void addMissingTopLevelElements() {
    final Script script = this.getState().resource.getScript();
    this.scriptApiTracker.initApiCompare(script);
    final ScriptApiTracker.ProjectComparisonAdapter comparison = ScriptApiTracker.firstProjectComparisonAdapter(script.eResource()).orElse(null);
    if ((null == comparison)) {
      return;
    }
    final Function1<ProjectComparisonEntry, Boolean> _function = (ProjectComparisonEntry it) -> {
      EObject _elementImpl = it.getElementImpl(0);
      return Boolean.valueOf((null == _elementImpl));
    };
    final Consumer<ProjectComparisonEntry> _function_1 = (ProjectComparisonEntry it) -> {
      EObject _elementAPI = it.getElementAPI();
      final EObject x = _elementAPI;
      boolean _matched = false;
      if (x instanceof TMethod) {
        _matched=true;
      }
      if (!_matched) {
        if (x instanceof TFunction) {
          _matched=true;
          this.missing(x);
        }
      }
      if (!_matched) {
        if (x instanceof TClass) {
          _matched=true;
          this.missing(x);
        }
      }
      if (!_matched) {
        if (x instanceof TInterface) {
          _matched=true;
          this.missing(x);
        }
      }
      if (!_matched) {
        if (x instanceof TEnum) {
          _matched=true;
          this.missing(x);
        }
      }
      if (!_matched) {
        if (x instanceof TVariable) {
          _matched=true;
          this.missing(x);
        }
      }
    };
    IterableExtensions.<ProjectComparisonEntry>filter(this.<ProjectComparisonEntry>toIterable(comparison.getEntryFor(script.getModule()).allChildren()), _function).forEach(_function_1);
  }
  
  private SymbolTableEntryIMOnly _missing(final TInterface tinter) {
    SymbolTableEntryIMOnly _xblockexpression = null;
    {
      final N4InterfaceDeclaration stub0 = TranspilerBuilderBlocks._N4InterfaceDeclaration(tinter.getName());
      final Function1<TAnnotation, AnnotationDefinition> _function = (TAnnotation it) -> {
        return AnnotationDefinition.find(it.getName());
      };
      stub0.setAnnotationList(TranspilerBuilderBlocks._AnnotationList(ListExtensions.<TAnnotation, AnnotationDefinition>map(tinter.getAnnotations(), _function)));
      EObject _wrapExported = this.wrapExported(tinter.isExported(), stub0);
      final ScriptElement stub = ((ScriptElement) _wrapExported);
      final MemberList<TMember> members = this.getState().memberCollector.members(tinter, false, false);
      for (final TMember m : members) {
        if ((!(m instanceof TField))) {
          EList<N4MemberDeclaration> _ownedMembersRaw = stub0.getOwnedMembersRaw();
          N4MemberDeclaration _createApiImplStub = this.createApiImplStub(stub0, m);
          _ownedMembersRaw.add(_createApiImplStub);
        }
      }
      this.appendToScript(stub);
      this.getState().info.setOriginalDefinedType(stub0, tinter);
      _xblockexpression = this.createSymbolTableEntryIMOnly(stub0);
    }
    return _xblockexpression;
  }
  
  private SymbolTableEntryIMOnly _missing(final TClass tclass) {
    SymbolTableEntryIMOnly _xblockexpression = null;
    {
      N4ClassDeclaration __N4ClassDeclaration = TranspilerBuilderBlocks._N4ClassDeclaration(tclass.getName());
      final Procedure1<N4ClassDeclaration> _function = (N4ClassDeclaration it) -> {
        EList<N4MemberDeclaration> _ownedMembersRaw = it.getOwnedMembersRaw();
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Class ");
        String _name = tclass.getName();
        _builder.append(_name);
        _builder.append(" is not implemented yet.");
        N4MethodDeclaration __N4MethodDecl = TranspilerBuilderBlocks._N4MethodDecl("constructor", 
          TranspilerBuilderBlocks._ThrowStmnt(TranspilerBuilderBlocks._NewExpr(
            TranspilerBuilderBlocks._IdentRef(this.N4ApiNotImplementedErrorSTE()), 
            TranspilerBuilderBlocks._StringLiteral(_builder.toString()))));
        _ownedMembersRaw.add(__N4MethodDecl);
      };
      final N4ClassDeclaration stub0 = ObjectExtensions.<N4ClassDeclaration>operator_doubleArrow(__N4ClassDeclaration, _function);
      final Function1<TAnnotation, AnnotationDefinition> _function_1 = (TAnnotation it) -> {
        return AnnotationDefinition.find(it.getName());
      };
      stub0.setAnnotationList(TranspilerBuilderBlocks._AnnotationList(ListExtensions.<TAnnotation, AnnotationDefinition>map(tclass.getAnnotations(), _function_1)));
      EObject _wrapExported = this.wrapExported(tclass.isExported(), stub0);
      final ScriptElement stub = ((ScriptElement) _wrapExported);
      final MemberList<TMember> members = this.getState().memberCollector.members(tclass, false, false);
      for (final TMember m : members) {
        if (((!(m instanceof TField)) && m.isStatic())) {
          EList<N4MemberDeclaration> _ownedMembersRaw = stub0.getOwnedMembersRaw();
          N4MemberDeclaration _createApiImplStub = this.createApiImplStub(stub0, m);
          _ownedMembersRaw.add(_createApiImplStub);
        }
      }
      this.appendToScript(stub);
      this.getState().info.setOriginalDefinedType(stub0, tclass);
      _xblockexpression = this.createSymbolTableEntryIMOnly(stub0);
    }
    return _xblockexpression;
  }
  
  private SymbolTableEntryIMOnly _missing(final TEnum tenum) {
    SymbolTableEntryIMOnly _xblockexpression = null;
    {
      final boolean stringBased = TypeSystemHelper.isStringBasedEnumeration(tenum);
      final Function1<TEnumLiteral, N4EnumLiteral> _function = (TEnumLiteral it) -> {
        return TranspilerBuilderBlocks._EnumLiteral(it.getName(), it.getName());
      };
      final N4EnumDeclaration stub0 = TranspilerBuilderBlocks._EnumDeclaration(tenum.getName(), ListExtensions.<TEnumLiteral, N4EnumLiteral>map(tenum.getLiterals(), _function));
      EObject _wrapExported = this.wrapExported(tenum.isExported(), stub0);
      ScriptElement stub = ((ScriptElement) _wrapExported);
      if (stringBased) {
        ((AnnotableScriptElement) stub).setAnnotationList(TranspilerBuilderBlocks._AnnotationList(Collections.<AnnotationDefinition>unmodifiableList(CollectionLiterals.<AnnotationDefinition>newArrayList(AnnotationDefinition.STRING_BASED))));
      }
      this.appendToScript(stub);
      this.getState().info.setOriginalDefinedType(stub0, tenum);
      _xblockexpression = this.createSymbolTableEntryIMOnly(stub0);
    }
    return _xblockexpression;
  }
  
  private Boolean appendToScript(final ScriptElement stub) {
    boolean _xblockexpression = false;
    {
      final Script_IM script = this.getState().im;
      boolean _xifexpression = false;
      boolean _isEmpty = script.getScriptElements().isEmpty();
      if (_isEmpty) {
        EList<ScriptElement> _scriptElements = script.getScriptElements();
        _xifexpression = _scriptElements.add(stub);
      } else {
        this.insertAfter(IterableExtensions.<ScriptElement>last(script.getScriptElements()), stub);
      }
      _xblockexpression = _xifexpression;
    }
    return Boolean.valueOf(_xblockexpression);
  }
  
  private SymbolTableEntryIMOnly _missing(final TVariable tvar) {
    this.missingFuncOrVar(tvar, tvar.isExported(), "variable");
    return null;
  }
  
  private SymbolTableEntryIMOnly _missing(final TFunction func) {
    this.missingFuncOrVar(func, func.isExported(), "function");
    return null;
  }
  
  private SymbolTableEntryInternal N4ApiNotImplementedErrorSTE() {
    return this.steFor_N4ApiNotImplementedError();
  }
  
  private void missingFuncOrVar(final IdentifiableElement func, final boolean exported, final String description) {
    final SymbolTableEntryOriginal funcSTE = this.getSymbolTableEntryOriginal(func, true);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(description);
    _builder.append(" ");
    String _name = funcSTE.getName();
    _builder.append(_name);
    _builder.append(" is not implemented yet.");
    FunctionDeclaration __FunDecl = TranspilerBuilderBlocks._FunDecl(funcSTE.getName(), TranspilerBuilderBlocks._ThrowStmnt(TranspilerBuilderBlocks._NewExpr(
      TranspilerBuilderBlocks._IdentRef(this.N4ApiNotImplementedErrorSTE()), 
      TranspilerBuilderBlocks._StringLiteral(_builder.toString()))));
    final Procedure1<FunctionDeclaration> _function = (FunctionDeclaration it) -> {
    };
    final FunctionDeclaration funcDecl = ObjectExtensions.<FunctionDeclaration>operator_doubleArrow(__FunDecl, _function);
    final EObject stub = this.wrapExported(exported, funcDecl);
    this.insertAfter(IterableExtensions.<ScriptElement>last(this.getState().im.getScriptElements()), stub);
  }
  
  private EObject wrapExported(final boolean exported, final ExportableElement toExportOrNotToExport) {
    EObject _xifexpression = null;
    if (exported) {
      _xifexpression = TranspilerBuilderBlocks._ExportDeclaration(toExportOrNotToExport);
    } else {
      _xifexpression = toExportOrNotToExport;
    }
    return _xifexpression;
  }
  
  /**
   * Creates a member that servers as the stub for a missing member on implementation side, corresponding to the given
   * member <code>apiMember</code> on API side.
   */
  private N4MemberDeclaration createApiImplStub(final N4ClassifierDeclaration classifierDecl, final TMember apiMember) {
    final SymbolTableEntryInternal N4ApiNotImplementedErrorSTE = this.steFor_N4ApiNotImplementedError();
    final String typeName = classifierDecl.getName();
    final String memberKeyword = this.n4jsElementKeywordProvider.keyword(apiMember);
    final String memberName = apiMember.getName();
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("API for ");
    _builder.append(memberKeyword);
    _builder.append(" ");
    _builder.append(typeName);
    _builder.append(".");
    _builder.append(memberName);
    _builder.append(" not implemented yet.");
    return TranspilerBuilderBlocks._N4MemberDecl(apiMember, 
      TranspilerBuilderBlocks._ThrowStmnt(
        TranspilerBuilderBlocks._NewExpr(TranspilerBuilderBlocks._IdentRef(N4ApiNotImplementedErrorSTE), TranspilerBuilderBlocks._StringLiteral(_builder.toString()))));
  }
  
  /**
   * Converts the stream into an iterable.
   */
  private <T extends Object> Iterable<T> toIterable(final Stream<T> stream) {
    final Iterable<T> _function = () -> {
      return stream.iterator();
    };
    return _function;
  }
  
  private MissingApiMembersForTranspiler createMAMFT(final TClassifier classifier) {
    final ConcreteMembersOrderedForTranspiler cmoft = this.typeAssistant.getOrCreateCMOFT(classifier);
    return MissingApiMembersForTranspiler.create(this.containerTypesHelper, this.scriptApiTracker, classifier, cmoft, this.getState().resource.getScript());
  }
  
  private SymbolTableEntryIMOnly missing(final EObject tclass) {
    if (tclass instanceof TClass) {
      return _missing((TClass)tclass);
    } else if (tclass instanceof TInterface) {
      return _missing((TInterface)tclass);
    } else if (tclass instanceof TEnum) {
      return _missing((TEnum)tclass);
    } else if (tclass instanceof TFunction) {
      return _missing((TFunction)tclass);
    } else if (tclass instanceof TVariable) {
      return _missing((TVariable)tclass);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(tclass).toString());
    }
  }
}
