/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.ExpressionStatement;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.assistants.TypeAssistant;
import org.eclipse.n4js.transpiler.es.assistants.BootstrapCallAssistant;
import org.eclipse.n4js.transpiler.utils.TranspilerUtils;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Transforms {@link N4EnumDeclaration}s into a <code>$makeEnum</code> call (for ordinary enums) or removes them
 * entirely (for <code>@StringBased</code> enums).
 */
@SuppressWarnings("all")
public class EnumDeclarationTransformation extends Transformation {
  @Inject
  private BootstrapCallAssistant bootstrapCallAssistant;
  
  @Inject
  private TypeAssistant typeAssistant;
  
  @Override
  public void assertPreConditions() {
    final Function1<N4EnumDeclaration, Boolean> _function = (N4EnumDeclaration it) -> {
      boolean _isTopLevel = this.typeAssistant.isTopLevel(it);
      return Boolean.valueOf((!_isTopLevel));
    };
    this.assertFalse("only top-level enums are supported, for now", 
      IteratorExtensions.<N4EnumDeclaration>exists(Iterators.<N4EnumDeclaration>filter(this.getState().im.eAllContents(), N4EnumDeclaration.class), _function));
  }
  
  @Override
  public void assertPostConditions() {
    final Function1<EObject, Boolean> _function = (EObject it) -> {
      return Boolean.valueOf((it instanceof N4EnumDeclaration));
    };
    this.assertFalse("there should not be any N4EnumDeclarations in the intermediate model", 
      IteratorExtensions.<EObject>exists(this.getState().im.eAllContents(), _function));
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final Consumer<N4EnumDeclaration> _function = (N4EnumDeclaration it) -> {
      this.transformEnumDecl(it);
    };
    this.<N4EnumDeclaration>collectNodes(this.getState().im, N4EnumDeclaration.class, false).forEach(_function);
  }
  
  private void transformEnumDecl(final N4EnumDeclaration enumDecl) {
    boolean _isStringBased = this.isStringBased(enumDecl);
    if (_isStringBased) {
      final EObject root = TranspilerUtils.orContainingExportDeclaration(enumDecl);
      this.remove(root);
    } else {
      EObject _xifexpression = null;
      boolean _isUseES6Imports = this.getState().project.isUseES6Imports();
      if (_isUseES6Imports) {
        _xifexpression = this.createFunDecl(enumDecl);
      } else {
        _xifexpression = this.createVarDecl(enumDecl);
      }
      final EObject varOrFunDecl = _xifexpression;
      final ExpressionStatement makeEnumCall = this.bootstrapCallAssistant.createMakeEnumCall(enumDecl);
      this.getState().tracer.copyTrace(enumDecl, makeEnumCall);
      EObject root_1 = null;
      if ((varOrFunDecl instanceof VariableDeclaration)) {
        this.replace(enumDecl, ((VariableDeclaration)varOrFunDecl));
        this.getState().info.markAsToHoist(((VariableDeclaration)varOrFunDecl));
        root_1 = TranspilerUtils.orContainingExportDeclaration(((VariableDeclaration)varOrFunDecl).eContainer());
      } else {
        if ((varOrFunDecl instanceof FunctionDeclaration)) {
          this.replace(enumDecl, ((FunctionDeclaration)varOrFunDecl));
          root_1 = TranspilerUtils.orContainingExportDeclaration(varOrFunDecl);
        }
      }
      this.insertAfter(root_1, makeEnumCall);
    }
  }
  
  /**
   * Creates declaration of the variable that will represent the enumeration.
   */
  private VariableDeclaration createVarDecl(final N4EnumDeclaration enumDecl) {
    VariableDeclaration __VariableDeclaration = TranspilerBuilderBlocks._VariableDeclaration(enumDecl.getName());
    final Procedure1<VariableDeclaration> _function = (VariableDeclaration it) -> {
      FormalParameter __Fpar = TranspilerBuilderBlocks._Fpar("name");
      FormalParameter __Fpar_1 = TranspilerBuilderBlocks._Fpar("value");
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("this.name = name;");
      _builder.newLine();
      _builder.append("this.value = value;");
      _builder.newLine();
      it.setExpression(TranspilerBuilderBlocks._FunExpr(false, enumDecl.getName(), new FormalParameter[] { __Fpar, __Fpar_1 }, 
        TranspilerBuilderBlocks._SnippetAsStmnt(_builder.toString())));
    };
    return ObjectExtensions.<VariableDeclaration>operator_doubleArrow(__VariableDeclaration, _function);
  }
  
  /**
   * Same as {@link #createVarDecl(N4EnumDeclaration)}, but creates a function declaration
   * instead of variable declaration with function expression as initializer.
   */
  private FunctionDeclaration createFunDecl(final N4EnumDeclaration enumDecl) {
    FormalParameter __Fpar = TranspilerBuilderBlocks._Fpar("name");
    FormalParameter __Fpar_1 = TranspilerBuilderBlocks._Fpar("value");
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("this.name = name;");
    _builder.newLine();
    _builder.append("this.value = value;");
    _builder.newLine();
    return TranspilerBuilderBlocks._FunDecl(enumDecl.getName(), new FormalParameter[] { __Fpar, __Fpar_1 }, 
      TranspilerBuilderBlocks._SnippetAsStmnt(_builder.toString()));
  }
  
  private boolean isStringBased(final N4EnumDeclaration enumDecl) {
    return AnnotationDefinition.STRING_BASED.hasAnnotation(enumDecl);
  }
}
