/**
 * Copyright (c) 2019 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.base.Optional;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.ModuleSpecifierAdjustment;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ModifiableElement;
import org.eclipse.n4js.n4JS.VariableBinding;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableDeclarationOrBinding;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.projectDescription.ModuleLoader;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.utils.ResourceNameComputer;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * New version of module wrapping using ECMAScript 2015 imports/exports in the output code and no longer using System.js.
 * Currently this mode is experimental and only active if package.json property 'useES6Imports' is set to 'true'.
 * User story GH-1281 will make this the default, remove the old 'ModuleWrappingTransformation' and remove suffix 'NEW'
 * from the name of this class.
 */
@SuppressWarnings("all")
public class ModuleWrappingTransformationNEW extends Transformation {
  @Inject
  private IN4JSCore n4jsCore;
  
  @Inject
  private ResourceNameComputer resourceNameComputer;
  
  @Override
  public void assertPreConditions() {
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final Consumer<ImportDeclaration> _function = (ImportDeclaration it) -> {
      this.transformImportDecl(it);
    };
    this.<ImportDeclaration>collectNodes(this.getState().im, ImportDeclaration.class, false).forEach(_function);
    final Function1<ExportDeclaration, ExportableElement> _function_1 = (ExportDeclaration it) -> {
      return it.getExportedElement();
    };
    final Consumer<ModifiableElement> _function_2 = (ModifiableElement it) -> {
      it.getDeclaredModifiers().clear();
    };
    Iterables.<ModifiableElement>filter(ListExtensions.<ExportDeclaration, ExportableElement>map(this.<ExportDeclaration>collectNodes(this.getState().im, ExportDeclaration.class, false), _function_1), ModifiableElement.class).forEach(_function_2);
    final Consumer<ExportDeclaration> _function_3 = (ExportDeclaration it) -> {
      this.splitDefaultExportFromVarDecl(it);
    };
    this.<ExportDeclaration>collectNodes(this.getState().im, ExportDeclaration.class, false).forEach(_function_3);
    this.addEmptyImport("n4js-node");
  }
  
  private void transformImportDecl(final ImportDeclaration importDeclIM) {
    final TModule module = this.getState().info.getImportedModule(importDeclIM);
    final String actualModuleSpecifier = this.computeActualModuleSpecifier(module);
    final String actualModuleSpecifierNormalized = actualModuleSpecifier.replace("/./", "/");
    importDeclIM.setModuleSpecifierAsText(actualModuleSpecifierNormalized);
  }
  
  private String computeActualModuleSpecifier(final TModule module) {
    final ModuleSpecifierAdjustment moduleSpecifierAdjustment = this.getModuleSpecifierAdjustment(module);
    if (((moduleSpecifierAdjustment != null) && moduleSpecifierAdjustment.usePlainModuleSpecifier)) {
      return module.getModuleSpecifier();
    }
    final String completeModuleSpecifier = this.resourceNameComputer.getCompleteModuleSpecifier(module);
    IN4JSProject depProject = this.n4jsCore.findProject(module.eResource().getURI()).orNull();
    if (((depProject != null) && (depProject.getProjectType() == ProjectType.DEFINITION))) {
      final String definedPackageName = depProject.getDefinesPackageName();
      if ((definedPackageName != null)) {
        depProject = this.n4jsCore.findAllProjectMappings().get(definedPackageName);
      }
    }
    if ((depProject != null)) {
      final String projectName = depProject.getProjectName();
      String outputPath = depProject.getOutputPath();
      if (((projectName != null) && (outputPath != null))) {
        boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(outputPath);
        if (_isNullOrEmpty) {
          outputPath = "/";
        } else {
          boolean _startsWith = outputPath.startsWith("/");
          boolean _not = (!_startsWith);
          if (_not) {
            outputPath = ("/" + outputPath);
          }
          boolean _endsWith = outputPath.endsWith("/");
          boolean _not_1 = (!_endsWith);
          if (_not_1) {
            outputPath = (outputPath + "/");
          }
        }
        return ((projectName + outputPath) + completeModuleSpecifier);
      }
    }
    return completeModuleSpecifier;
  }
  
  /**
   * returns adjustments to be used based on the module loader specified for the provided module. May be null.
   */
  private ModuleSpecifierAdjustment getModuleSpecifierAdjustment(final TModule module) {
    Resource _eResource = null;
    if (module!=null) {
      _eResource=module.eResource();
    }
    URI _uRI = null;
    if (_eResource!=null) {
      _uRI=_eResource.getURI();
    }
    final URI resourceURI = _uRI;
    if ((resourceURI == null)) {
      return null;
    }
    final Optional<? extends IN4JSProject> project = this.n4jsCore.findProject(resourceURI);
    boolean _isPresent = project.isPresent();
    boolean _not = (!_isPresent);
    if (_not) {
      return null;
    }
    final ModuleLoader loader = project.get().getModuleLoader();
    if ((loader == null)) {
      return null;
    }
    final ModuleSpecifierAdjustment adjustment = N4JSLanguageConstants.MODULE_LOADER_PREFIXES.get(loader);
    return adjustment;
  }
  
  /**
   * Turns
   * <pre>
   * export default var|let|const C = ...
   * </pre>
   * into
   * <pre>
   * var|let|const C = ...
   * export default C;
   * </pre>
   */
  private void splitDefaultExportFromVarDecl(final ExportDeclaration exportDecl) {
    boolean _isDefaultExport = exportDecl.isDefaultExport();
    if (_isDefaultExport) {
      final ExportableElement exportedElement = exportDecl.getExportedElement();
      if ((exportedElement instanceof VariableStatement)) {
        boolean _isEmpty = IterableExtensions.isEmpty(Iterables.<VariableBinding>filter(((VariableStatement)exportedElement).getVarDeclsOrBindings(), VariableBinding.class));
        boolean _not = (!_isEmpty);
        if (_not) {
          throw new UnsupportedOperationException("unsupported: default-exported variable binding");
        }
        int _size = ((VariableStatement)exportedElement).getVarDeclsOrBindings().size();
        boolean _greaterThan = (_size > 1);
        if (_greaterThan) {
          throw new UnsupportedOperationException("unsupported: several default-exported variable declarations in a single export declaration");
        }
        VariableDeclarationOrBinding _head = IterableExtensions.<VariableDeclarationOrBinding>head(((VariableStatement)exportedElement).getVarDeclsOrBindings());
        final VariableDeclaration varDecl = ((VariableDeclaration) _head);
        final SymbolTableEntry varDeclSTE = this.findSymbolTableEntryForElement(varDecl, true);
        this.insertBefore(exportDecl, exportedElement);
        exportDecl.setDefaultExportedExpression(TranspilerBuilderBlocks._IdentRef(varDeclSTE));
      }
    }
  }
}
