/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.changes;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ExportedVariableStatement;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.ModifiableElement;
import org.eclipse.n4js.n4JS.ModifierUtils;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4JSFeatureUtils;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.N4Modifier;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.n4JS.TypeDefiningElement;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ui.changes.ChangeProvider;
import org.eclipse.n4js.ui.changes.IChange;
import org.eclipse.n4js.ui.changes.ICompositeChange;
import org.eclipse.n4js.utils.nodemodel.NodeModelAccess;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.ILeafNode;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Collection of high-level convenience methods for creating {@link IChange}s.
 * <p>
 * By convention, all methods take an {@link IXtextDocument} as first parameter and return an instance of IChange.
 * 
 * Other than ChangeProvider the methods in this class are thought to be a tool to handle
 * AST elements directly without dealing with their textual representation.
 */
@SuppressWarnings("all")
public class SemanticChangeProvider {
  /**
   * Retrieve internal annotation constant from AnnotationDefinition
   */
  private final static String INTERNAL_ANNOTATION = AnnotationDefinition.INTERNAL.name;
  
  @Inject
  private NodeModelAccess nodeModelAccess;
  
  @Inject
  private N4JSElementKeywordProvider elementKeywordProvider;
  
  /**
   * Return the IChange to set the access modifier for the given element.
   * 
   * @param document Document to modify
   * @param element Element to modify
   * @param modifier Modifier to set
   */
  public IChange setAccessModifiers(final IXtextDocument document, final ModifiableElement element, final N4Modifier modifier) throws BadLocationException {
    return this.setAccessModifiers(document, element, modifier, false);
  }
  
  /**
   * Return the IChange to set the access modifier for the given element with optional export modifier
   * 
   * @param document Document to modify
   * @param element Element to modify
   * @param modifier Modifier to set
   * @param export Optional export modifier
   */
  public IChange setAccessModifier(final IXtextDocument document, final TypeDefiningElement element, final N4Modifier modifier, final boolean export) throws BadLocationException {
    if (((!element.getDefinedType().isExported()) && export)) {
      return this.setAccessModifiers(document, ((ModifiableElement) element), modifier, true);
    }
    if ((element instanceof ModifiableElement)) {
      return this.setAccessModifiers(document, ((ModifiableElement) element), modifier, false);
    }
    return null;
  }
  
  /**
   * Return the IChange to set the access modifier for the given element with optional export modifier.
   *  No validation of export parameter. (e.g. members)
   * 
   * @param document Document to modify
   * @param element Element to modify
   * @param modifier Modifier to set
   * @param export Optional export modifier
   */
  private IChange setAccessModifiers(final IXtextDocument document, final ModifiableElement element, final N4Modifier modifier, final boolean export) {
    EList<N4Modifier> _declaredModifiers = element.getDeclaredModifiers();
    final List<N4Modifier> modifiers = new ArrayList<N4Modifier>(_declaredModifiers);
    final Function1<N4Modifier, Boolean> _function = (N4Modifier it) -> {
      boolean _isAccessModifier = ModifierUtils.isAccessModifier(it);
      return Boolean.valueOf((!_isAccessModifier));
    };
    final List<N4Modifier> nonAccessModifier = IterableExtensions.<N4Modifier>toList(IterableExtensions.<N4Modifier>filter(modifiers, _function));
    CollectionExtensions.<N4Modifier>addAll(nonAccessModifier, modifier);
    String exportPrefix = "";
    if (export) {
      exportPrefix = N4JSLanguageConstants.EXPORT_KEYWORD;
      int _length = ((Object[])Conversions.unwrapArray(nonAccessModifier, Object.class)).length;
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        String _exportPrefix = exportPrefix;
        exportPrefix = (_exportPrefix + " ");
      }
    }
    String _sortedModifierString = this.sortedModifierString(nonAccessModifier);
    String _plus = (exportPrefix + _sortedModifierString);
    return this.setModifiers(document, element, _plus);
  }
  
  /**
   * Return the IChange to add a custom modifier to the given element.
   * 
   * Note that all existing modifiers are maintained while the new custom modifier(s) are added at the beginning of the line.
   * No validation of a correct modifier order takes place.
   * 
   * @param document XtextDocument to modify
   * @param element ModifiableElement to declare as exported
   */
  public IChange addCustomModifier(final IXtextDocument document, final ModifiableElement element, final String modifier) {
    EList<N4Modifier> _declaredModifiers = element.getDeclaredModifiers();
    final ArrayList<N4Modifier> modifiers = new ArrayList<N4Modifier>(_declaredModifiers);
    String exportPrefix = modifier;
    int _length = ((Object[])Conversions.unwrapArray(modifiers, Object.class)).length;
    boolean _greaterThan = (_length > 0);
    if (_greaterThan) {
      String _exportPrefix = exportPrefix;
      exportPrefix = (_exportPrefix + " ");
    }
    String _sortedModifierString = this.sortedModifierString(modifiers);
    String _plus = (exportPrefix + _sortedModifierString);
    return this.setModifiers(document, element, _plus);
  }
  
  /**
   * Return the IChange to set the modifiers for the given Element. Replaces all modifiers with new ones.
   * 
   * @param document Document to modify
   * @param element Element to modify
   * @param modifier New modifiers
   */
  public IChange setModifiers(final IXtextDocument document, final EObject element, final String modifiers) {
    IChange _xblockexpression = null;
    {
      List _xifexpression = null;
      if ((element instanceof ModifiableElement)) {
        _xifexpression = ((ModifiableElement)element).getDeclaredModifiers();
      } else {
        _xifexpression = Collections.EMPTY_LIST;
      }
      List declaredModifiers = _xifexpression;
      String extra_whitespace = "";
      int delete_extra_whitespace = 0;
      int startOffset = this.modifierOffset(element);
      int endOffset = startOffset;
      final List _converted_declaredModifiers = (List)declaredModifiers;
      int _length = ((Object[])Conversions.unwrapArray(_converted_declaredModifiers, Object.class)).length;
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        final List _converted_declaredModifiers_1 = (List)declaredModifiers;
        int _length_1 = ((Object[])Conversions.unwrapArray(_converted_declaredModifiers_1, Object.class)).length;
        int _minus = (_length_1 - 1);
        final ILeafNode endNode = ModifierUtils.getNodeForModifier(((ModifiableElement) element), _minus);
        int _offset = endNode.getOffset();
        int _length_2 = endNode.getLength();
        int _plus = (_offset + _length_2);
        endOffset = _plus;
      } else {
        int _length_3 = modifiers.length();
        boolean _greaterThan_1 = (_length_3 > 0);
        if (_greaterThan_1) {
          extra_whitespace = " ";
        }
      }
      int _length_4 = modifiers.length();
      boolean _equals = (_length_4 == 0);
      if (_equals) {
        delete_extra_whitespace = 1;
      }
      final int modifierLength = ((endOffset - startOffset) + delete_extra_whitespace);
      _xblockexpression = ChangeProvider.replace(document, startOffset, modifierLength, (modifiers + extra_whitespace));
    }
    return _xblockexpression;
  }
  
  /**
   * Returns IChange to set the modifiers of the element
   * 
   * @param document Document to modify
   * @param element Element to modify
   * @param modifier List of N4Modifier to set
   */
  public IChange setModifiers(final IXtextDocument document, final ModifiableElement element, final List<N4Modifier> modifiers) throws BadLocationException {
    return this.setModifiers(document, element, this.sortedModifierString(modifiers));
  }
  
  /**
   * Returns IChange to add modifier to element.
   * 
   * @param document
   * 		The xtext document to make the changes in
   * @param element
   * 		The element to add the modifier to
   * @param modifier
   * 		The modifier to add
   */
  public IChange addModifier(final IXtextDocument document, final ModifiableElement element, final N4Modifier modifier) throws BadLocationException {
    EList<N4Modifier> _declaredModifiers = element.getDeclaredModifiers();
    ArrayList<N4Modifier> modifiers = new ArrayList<N4Modifier>(_declaredModifiers);
    modifiers.add(modifier);
    return this.setModifiers(document, element, this.sortedModifierString(modifiers));
  }
  
  /**
   * Returns IChange to remove modifier from element.
   * Only removes existing modifiers
   * 
   * @param document Document to modify
   * @param element Element to modify
   * @param modifier Modifier to remove
   */
  public IChange removeModifier(final IXtextDocument document, final ModifiableElement element, final N4Modifier modifier) throws BadLocationException {
    final Function1<N4Modifier, Boolean> _function = (N4Modifier it) -> {
      String _name = it.name();
      String _name_1 = modifier.name();
      return Boolean.valueOf((!Objects.equal(_name, _name_1)));
    };
    final List<N4Modifier> modifiers = IterableExtensions.<N4Modifier>toList(IterableExtensions.<N4Modifier>filter(element.getDeclaredModifiers(), _function));
    return this.setModifiers(document, element, this.sortedModifierString(modifiers));
  }
  
  /**
   * Returns IChange to add annotation to element. Only adds not yet existing annotations.
   * 
   * @param document Document to modify
   * @param element Element to modify
   * @param annotation Annotation to add
   */
  public IChange addAnnotation(final IXtextDocument document, final AnnotableElement element, final String annotation) throws BadLocationException {
    boolean _equals = annotation.equals(SemanticChangeProvider.INTERNAL_ANNOTATION);
    if (_equals) {
      return this.addInternalAnnotation(document, element);
    }
    EList<Annotation> _annotations = element.getAnnotations();
    ArrayList<Annotation> annotations = new ArrayList<Annotation>(_annotations);
    final Function1<Annotation, Boolean> _function = (Annotation it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equal(_name, annotation));
    };
    Annotation _findFirst = IterableExtensions.<Annotation>findFirst(annotations, _function);
    boolean _tripleEquals = (_findFirst == null);
    if (_tripleEquals) {
      ICompositeNode elementNode = NodeModelUtils.findActualNodeFor(element);
      return ChangeProvider.insertLineAbove(document, elementNode.getOffset(), ("@" + annotation), true);
    }
    return IChange.IDENTITY;
  }
  
  /**
   * Returns IChange to add @Internal annotation to the element.
   */
  private IChange addInternalAnnotation(final IXtextDocument document, final AnnotableElement element) {
    ICompositeChange _xblockexpression = null;
    {
      final Function1<Annotation, Boolean> _function = (Annotation it) -> {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, SemanticChangeProvider.INTERNAL_ANNOTATION));
      };
      Annotation _findFirst = IterableExtensions.<Annotation>findFirst(element.getAnnotations(), _function);
      boolean _tripleEquals = (_findFirst == null);
      if (_tripleEquals) {
        if ((element instanceof ModifiableElement)) {
          int offset = this.internalAnnotationOffset(((ModifiableElement)element));
          return ChangeProvider.replace(document, offset, 0, (("@" + SemanticChangeProvider.INTERNAL_ANNOTATION) + " "));
        }
      }
      _xblockexpression = IChange.IDENTITY;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the offset to place the @Internal annotation in the same line
   */
  private int internalAnnotationOffset(final ModifiableElement element) {
    if ((!(element instanceof AnnotableElement))) {
      throw new IllegalArgumentException("Can\'t compute @Internal offset for non-annotable element");
    }
    final EObject containerExportDeclaration = element.eContainer();
    if ((containerExportDeclaration instanceof ExportDeclaration)) {
      final ILeafNode node = this.nodeModelAccess.nodeForKeyword(containerExportDeclaration, N4JSLanguageConstants.EXPORT_KEYWORD);
      if ((node != null)) {
        return node.getOffset();
      } else {
        throw new NullPointerException("Failed to retrieve node for export keyword");
      }
    } else {
      return this.modifierOffset(element);
    }
  }
  
  /**
   * Return the ast node which holds the text represented by given keyword.
   * May be null. (e.g. keyword does not occur in element)
   */
  private INode astNodeForKeyword(final EObject element, final String keyword) {
    return this.nodeModelAccess.nodeForKeyword(element, keyword);
  }
  
  /**
   * Returns the modifier offset where modifiers are placed in front of a modifiable element, even if the element does
   * not have any modifiers.
   * 
   * @param element modifiable element to compute the modifier offset of
   * 
   * <p>Note:
   * - This does not include the export modifier as it is grammar wise no modifier. (offset is always after
   *   the export keyword for export declarations)
   * 
   * - Offset means that there is no additional space between the offset and following element.
   *   Therefore an additional space character has to be inserted in most of the cases
   * </p>
   */
  private int modifierOffset(final EObject element) {
    int offset = (-1);
    if (((element instanceof ModifiableElement) && (((Object[])Conversions.unwrapArray(((ModifiableElement) element).getDeclaredModifiers(), Object.class)).length > 0))) {
      offset = ModifierUtils.getNodeForModifier(((ModifiableElement) element), 0).getOffset();
    } else {
      int _switchResult = (int) 0;
      boolean _matched = false;
      if (element instanceof N4GetterDeclaration) {
        _matched=true;
        _switchResult = this.astNodeForKeyword(element, N4JSLanguageConstants.GET_KEYWORD).getOffset();
      }
      if (!_matched) {
        if (element instanceof N4SetterDeclaration) {
          _matched=true;
          _switchResult = this.astNodeForKeyword(element, N4JSLanguageConstants.SET_KEYWORD).getOffset();
        }
      }
      if (!_matched) {
        if (element instanceof FunctionDeclaration) {
          _matched=true;
        }
        if (!_matched) {
          if (element instanceof N4ClassDeclaration) {
            _matched=true;
          }
        }
        if (!_matched) {
          if (element instanceof N4InterfaceDeclaration) {
            _matched=true;
          }
        }
        if (!_matched) {
          if (element instanceof N4EnumDeclaration) {
            _matched=true;
          }
        }
        if (_matched) {
          _switchResult = this.astNodeForKeyword(element, this.elementKeywordProvider.keyword(element)).getOffset();
        }
      }
      if (!_matched) {
        if (element instanceof ExportedVariableStatement) {
          _matched=true;
          int _xblockexpression = (int) 0;
          {
            final INode exportKeyword = this.astNodeForKeyword(((ExportedVariableStatement)element).eContainer(), N4JSLanguageConstants.EXPORT_KEYWORD);
            int _offset = exportKeyword.getOffset();
            int _length = exportKeyword.getLength();
            int _plus = (_offset + _length);
            _xblockexpression = (_plus + 1);
          }
          _switchResult = _xblockexpression;
        }
      }
      if (!_matched) {
        if (element instanceof N4MethodDeclaration) {
          _matched=true;
          int _xblockexpression = (int) 0;
          {
            List<INode> nodes = null;
            nodes = NodeModelUtils.findNodesForFeature(element, N4JSPackage.Literals.GENERIC_DECLARATION__TYPE_VARS);
            boolean _isEmpty = nodes.isEmpty();
            if (_isEmpty) {
              nodes = NodeModelUtils.findNodesForFeature(element, N4JSPackage.Literals.TYPED_ELEMENT__BOGUS_TYPE_REF);
            }
            boolean _isEmpty_1 = nodes.isEmpty();
            if (_isEmpty_1) {
              nodes = NodeModelUtils.findNodesForFeature(element, N4JSPackage.Literals.FUNCTION_DEFINITION__GENERATOR);
            }
            boolean _isEmpty_2 = nodes.isEmpty();
            if (_isEmpty_2) {
              nodes = NodeModelUtils.findNodesForFeature(element, N4JSPackage.Literals.FUNCTION_DEFINITION__DECLARED_ASYNC);
            }
            boolean _isEmpty_3 = nodes.isEmpty();
            if (_isEmpty_3) {
              nodes = NodeModelUtils.findNodesForFeature(element, N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME);
            }
            int _xifexpression = (int) 0;
            if (((nodes != null) && (!nodes.isEmpty()))) {
              int _xblockexpression_1 = (int) 0;
              {
                final INode node = nodes.get(0);
                final EObject semElem = node.getSemanticElement();
                int _xifexpression_1 = (int) 0;
                if ((semElem instanceof TypeVariable)) {
                  int _offset = node.getOffset();
                  _xifexpression_1 = (_offset - 1);
                } else {
                  _xifexpression_1 = node.getOffset();
                }
                _xblockexpression_1 = _xifexpression_1;
              }
              _xifexpression = _xblockexpression_1;
            } else {
              _xifexpression = (-1);
            }
            _xblockexpression = _xifexpression;
          }
          _switchResult = _xblockexpression;
        }
      }
      if (!_matched) {
        if (element instanceof NamedElement) {
          _matched=true;
          int _xblockexpression = (int) 0;
          {
            final EStructuralFeature attr = N4JSFeatureUtils.attributeOfNameFeature(((NamedElement)element));
            final List<INode> nodes = NodeModelUtils.findNodesForFeature(element, attr);
            int _xifexpression = (int) 0;
            if (((nodes != null) && (!nodes.isEmpty()))) {
              _xifexpression = nodes.get(0).getOffset();
            } else {
              _xifexpression = (-1);
            }
            _xblockexpression = _xifexpression;
          }
          _switchResult = _xblockexpression;
        }
      }
      if (!_matched) {
        _switchResult = (-1);
      }
      offset = _switchResult;
    }
    if ((offset == (-1))) {
      throw new IllegalArgumentException("Couldn\'t determine element modifier offset");
    }
    return offset;
  }
  
  /**
   * Returns IChange to remove annotation from element. Only removes exisiting annotations.
   * 
   * @param document Document to modify
   * @param element Element to modify
   * @param annotation Annotation to remove
   */
  public IChange removeAnnotation(final IXtextDocument document, final AnnotableElement element, final String annotation) throws BadLocationException {
    AnnotableElement annotatedElement = element;
    Collection<Annotation> sourceList = null;
    Annotation targetAnnotation = null;
    final Function1<Annotation, Boolean> _function = (Annotation it) -> {
      return Boolean.valueOf(it.getName().equals(annotation));
    };
    Annotation _findFirst = IterableExtensions.<Annotation>findFirst(annotatedElement.getAnnotations(), _function);
    boolean _tripleNotEquals = (_findFirst != null);
    if (_tripleNotEquals) {
      sourceList = annotatedElement.getAnnotations();
      final Function1<Annotation, Boolean> _function_1 = (Annotation it) -> {
        return Boolean.valueOf(it.getName().equals(annotation));
      };
      targetAnnotation = IterableExtensions.<Annotation>findFirst(annotatedElement.getAnnotations(), _function_1);
    } else {
      if (((element.eContainer() instanceof ExportDeclaration) && 
        (IterableExtensions.<Annotation>findFirst(((ExportDeclaration) element.eContainer()).getAnnotations(), ((Function1<Annotation, Boolean>) (Annotation it) -> {
          return Boolean.valueOf(it.getName().equals(annotation));
        })) != null))) {
        EObject _eContainer = element.eContainer();
        sourceList = ((ExportDeclaration) _eContainer).getAnnotations();
        EObject _eContainer_1 = element.eContainer();
        final Function1<Annotation, Boolean> _function_2 = (Annotation it) -> {
          return Boolean.valueOf(it.getName().equals(annotation));
        };
        targetAnnotation = IterableExtensions.<Annotation>findFirst(((ExportDeclaration) _eContainer_1).getAnnotations(), _function_2);
      } else {
        return IChange.IDENTITY;
      }
    }
    ICompositeNode node = NodeModelUtils.findActualNodeFor(targetAnnotation);
    int offset = node.getOffset();
    int length = node.getLength();
    Annotation _head = IterableExtensions.<Annotation>head(sourceList);
    boolean _tripleEquals = (targetAnnotation == _head);
    if (_tripleEquals) {
      ICompositeNode containerNode = NodeModelUtils.findActualNodeFor(targetAnnotation.eContainer());
      offset = containerNode.getOffset();
      int _offset = node.getOffset();
      int _length = node.getLength();
      int _plus = (_offset + _length);
      int _offset_1 = containerNode.getOffset();
      int _minus = (_plus - _offset_1);
      length = _minus;
    }
    if (((element instanceof ModifiableElement) && 
      (document.getLineOfOffset(offset) == document.getLineOfOffset(this.modifierOffset(((ModifiableElement) element)))))) {
      final char trailingChar = document.getChar((offset + length));
      final char leadingChar = document.getChar((offset - 1));
      if ((Character.isWhitespace(leadingChar) && 
        Character.isWhitespace(trailingChar))) {
        length++;
      }
    }
    return ChangeProvider.removeText(document, offset, length, true);
  }
  
  /**
   * Extension method to sort modifiers
   */
  public String sortedModifierString(final List<N4Modifier> modifierList) {
    return IterableExtensions.join(ModifierUtils.getSortedModifiers(modifierList), " ");
  }
}
