/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.organize.imports;

import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4TypeDeclaration;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.organize.imports.InjectedTypesResolverUtility;
import org.eclipse.n4js.organize.imports.RefNameUtil;
import org.eclipse.n4js.organize.imports.ScriptDependency;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.ModuleNamespaceVirtualType;
import org.eclipse.n4js.ts.types.TAnnotableElement;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Static analysis for {@link Script} dependencies. Analyzes all identifiers in the {@link Script},
 * and on (on demand) {@link Type}s or {@link TypeRef}s. During analysis identifiers from {@link NamedImportSpecifier}
 * and {@link NamespaceImportSpecifier} are taken into account. Note that if result analysis does not contain descriptions matching
 * some of the imports, it means those imports are not used, and can be removed from script, or ignored during compilation.
 * 
 * Result of analysis is list of {@link ScriptDependency} instances describing what {@link EObject}s should be imported.
 * <ul>
 * Note:
 *  <li> declarations marked with "@ProvidedByRuntime" are ignored (they are never in the result)</li>
 *  <li> declarations marked with "@Global" are not ignored (can show up in the result)</li>
 * </ul>
 */
@SuppressWarnings("all")
public class ScriptDependencyResolver {
  /**
   * Resolves dependencies only from {@link IdentifierRef}s, no {@link Type}s or {@link TypeRef}s
   * are taken into account.
   */
  public static List<ScriptDependency> usedDependencies(final Script script) {
    List<ScriptDependency> _xblockexpression = null;
    {
      if ((null == script)) {
        return CollectionLiterals.<ScriptDependency>emptyList();
      }
      _xblockexpression = ScriptDependencyResolver.allRequiredExternalDeclaration(script, Collections.EMPTY_SET, Collections.EMPTY_SET);
    }
    return _xblockexpression;
  }
  
  /**
   * Resolves dependencies from {@link IdentifierRef}s and {@link Type}s that are
   * injected into local type declarations.
   */
  public static List<ScriptDependency> usedDependenciesWithInejctedTypes(final Script script) {
    List<ScriptDependency> _xblockexpression = null;
    {
      if ((null == script)) {
        return CollectionLiterals.<ScriptDependency>emptyList();
      }
      _xblockexpression = ScriptDependencyResolver.allRequiredExternalDeclaration(script, InjectedTypesResolverUtility.findAllInjected(script), Collections.EMPTY_SET);
    }
    return _xblockexpression;
  }
  
  /**
   * Resolves dependencies from {@link IdentifierRef}s and all {@link TypeRef}s.
   */
  public static List<ScriptDependency> usedDependenciesTypeRefs(final Script script) {
    List<ScriptDependency> _xblockexpression = null;
    {
      if ((null == script)) {
        return CollectionLiterals.<ScriptDependency>emptyList();
      }
      _xblockexpression = ScriptDependencyResolver.allRequiredExternalDeclaration(script, Collections.EMPTY_SET, IteratorExtensions.<TypeRef>toList(Iterators.<TypeRef>filter(script.eAllContents(), TypeRef.class)));
    }
    return _xblockexpression;
  }
  
  /**
   * Looks through all {@link IdentifierRef} for external dependencies
   * (from different module than currently analyzed script containing module).
   * Additionally looks through all types used as super types and implemented interfaces.
   * Not used types (see {@link #shouldBeImported}) are removed from external dependencies.
   * 
   * @param script to be analyzed
   * @param typesToBeIncluded force specific collection of {@link Type}s to be considered for as dependencies
   * @param typeRefsToBeIncluded force specific collection of {@link TypeRef}s to be considered for as dependencies
   */
  private static List<ScriptDependency> allRequiredExternalDeclaration(final Script script, final Collection<Type> typesToBeIncluded, final Collection<TypeRef> typeRefsToBeIncluded) {
    final List<N4TypeDeclaration> indirectlyImported = IteratorExtensions.<N4TypeDeclaration>toList(Iterators.<N4TypeDeclaration>filter(script.eAllContents(), N4TypeDeclaration.class));
    final List<IdentifierRef> identifierRefs = IteratorExtensions.<IdentifierRef>toList(Iterators.<IdentifierRef>filter(script.eAllContents(), IdentifierRef.class));
    final ArrayList<EObject> potentialDependencies = CollectionLiterals.<EObject>newArrayList();
    Iterables.<EObject>addAll(potentialDependencies, indirectlyImported);
    Iterables.<EObject>addAll(potentialDependencies, identifierRefs);
    Iterables.<EObject>addAll(potentialDependencies, typesToBeIncluded);
    Iterables.<EObject>addAll(potentialDependencies, typeRefsToBeIncluded);
    final Function1<NamedImportSpecifier, Boolean> _function = (NamedImportSpecifier it) -> {
      TExportableElement _importedElement = it.getImportedElement();
      return Boolean.valueOf((_importedElement != null));
    };
    final List<NamedImportSpecifier> namedImportSpecifiers = IteratorExtensions.<NamedImportSpecifier>toList(IteratorExtensions.<NamedImportSpecifier>filter(Iterators.<NamedImportSpecifier>filter(script.eAllContents(), NamedImportSpecifier.class), _function));
    final Function1<NamespaceImportSpecifier, Boolean> _function_1 = (NamespaceImportSpecifier it) -> {
      return Boolean.valueOf(false);
    };
    final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers = IteratorExtensions.<NamespaceImportSpecifier, Boolean>toInvertedMap(Iterators.<NamespaceImportSpecifier>filter(script.eAllContents(), NamespaceImportSpecifier.class), _function_1);
    final TModule baseModule = script.getModule();
    final Function1<EObject, Iterable<ScriptDependency>> _function_2 = (EObject it) -> {
      final Function1<NamedImportSpecifier, String> _function_3 = (NamedImportSpecifier it_1) -> {
        return it_1.getImportedElement().getName();
      };
      final Function1<EObject, Boolean> _function_4 = (EObject eo) -> {
        return Boolean.valueOf(ScriptDependencyResolver.shouldBeImported(baseModule, eo));
      };
      return ScriptDependencyResolver.handle(it, IterableExtensions.<String, NamedImportSpecifier>toMap(namedImportSpecifiers, _function_3), usedNamespaceSpecifiers, _function_4);
    };
    final Function1<ScriptDependency, String> _function_3 = (ScriptDependency it) -> {
      return it.localName;
    };
    List<ScriptDependency> depsImportedWithName = IterableExtensions.<ScriptDependency, String>sortBy(IterableExtensions.<ScriptDependency>toList(IterableExtensions.<ScriptDependency>toSet(IterableExtensions.<ScriptDependency>filterNull(Iterables.<ScriptDependency>concat(ListExtensions.<EObject, Iterable<ScriptDependency>>map(potentialDependencies, _function_2))))), _function_3);
    return depsImportedWithName;
  }
  
  /**
   * Checks if a given EObject should be imported.
   * 
   * <ul>
   * Evaluates to true if:
   * <li> provided EO is not from the module provided at creation time (that module is assumed to be one for which we analyze dependencies)</li>
   * <li> provided EO is not from built in types (usually n4ts files)</li>
   * <li> (in case of AST elements) is not annotated with {@link AnnotationDefinition.PROVIDED_BY_RUNTIME)</li>
   * <li> (in case of TS elements) providedByRuntime evaluates to false</li>
   * </ul>
   * 
   * @returns true if given EO should be imported
   */
  public static boolean shouldBeImported(final TModule baseModule, final EObject eo) {
    if ((eo instanceof ModuleNamespaceVirtualType)) {
      return true;
    }
    if ((eo instanceof AnnotableElement)) {
      boolean _hasAnnotation = AnnotationDefinition.PROVIDED_BY_RUNTIME.hasAnnotation(((AnnotableElement)eo));
      if (_hasAnnotation) {
        return false;
      }
    } else {
      if ((eo instanceof TAnnotableElement)) {
        boolean _hasAnnotation_1 = AnnotationDefinition.PROVIDED_BY_RUNTIME.hasAnnotation(((TAnnotableElement)eo));
        if (_hasAnnotation_1) {
          return false;
        } else {
          if ((eo instanceof Type)) {
            boolean _isProvidedByRuntime = ((Type)eo).isProvidedByRuntime();
            if (_isProvidedByRuntime) {
              return false;
            }
          } else {
            if ((eo instanceof TVariable)) {
              boolean _isProvidedByRuntime_1 = ((TVariable)eo).isProvidedByRuntime();
              if (_isProvidedByRuntime_1) {
                return false;
              }
            }
          }
        }
      }
    }
    boolean _and = false;
    boolean _and_1 = false;
    Resource _eResource = eo.eResource();
    URI _uRI = null;
    if (_eResource!=null) {
      _uRI=_eResource.getURI();
    }
    boolean _tripleNotEquals = (_uRI != null);
    if (!_tripleNotEquals) {
      _and_1 = false;
    } else {
      _and_1 = (eo.eResource().getURI().isFile() || eo.eResource().getURI().isPlatformResource());
    }
    if (!_and_1) {
      _and = false;
    } else {
      boolean _equals = eo.eResource().getURI().toString().equals(baseModule.eResource().getURI().toString());
      boolean _not = (!_equals);
      _and = _not;
    }
    return _and;
  }
  
  private static boolean isNamespaceDependencyHandlingNeeded(final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final TModule targMod) {
    final Function1<NamespaceImportSpecifier, Boolean> _function = (NamespaceImportSpecifier is) -> {
      EObject _eContainer = is.eContainer();
      TModule _module = ((ImportDeclaration) _eContainer).getModule();
      return Boolean.valueOf((_module == targMod));
    };
    return IterableExtensions.<NamespaceImportSpecifier>exists(usedNamespaceSpecifiers.keySet(), _function);
  }
  
  private static ScriptDependency createScriptDependency(final Type type, final Map<String, NamedImportSpecifier> nameToNamedImportSpecifiers, final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers) {
    ScriptDependency _xifexpression = null;
    boolean _containsKey = nameToNamedImportSpecifiers.containsKey(type.getName());
    if (_containsKey) {
      ScriptDependency _xblockexpression = null;
      {
        final NamedImportSpecifier nis = nameToNamedImportSpecifiers.get(type.getName());
        final TExportableElement identifiableElement = nis.getImportedElement();
        String _elvis = null;
        String _alias = nis.getAlias();
        if (_alias != null) {
          _elvis = _alias;
        } else {
          String _name = identifiableElement.getName();
          _elvis = _name;
        }
        String _name_1 = identifiableElement.getName();
        EObject _eContainer = identifiableElement.eContainer();
        _xblockexpression = new ScriptDependency(_elvis, _name_1, identifiableElement, ((TModule) _eContainer));
      }
      _xifexpression = _xblockexpression;
    } else {
      ScriptDependency _xifexpression_1 = null;
      boolean _isNamespaceDependencyHandlingNeeded = ScriptDependencyResolver.isNamespaceDependencyHandlingNeeded(usedNamespaceSpecifiers, type.getContainingModule());
      if (_isNamespaceDependencyHandlingNeeded) {
        _xifexpression_1 = ScriptDependencyResolver.createDependencyOnNamespace(usedNamespaceSpecifiers, type.getContainingModule());
      } else {
        String _name = type.getName();
        String _name_1 = type.getName();
        TModule _containingModule = type.getContainingModule();
        _xifexpression_1 = new ScriptDependency(_name, _name_1, type, _containingModule);
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  private static ScriptDependency createDependencyOnNamespace(final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final TModule targMod) {
    ScriptDependency _xblockexpression = null;
    {
      final Function1<NamespaceImportSpecifier, Boolean> _function = (NamespaceImportSpecifier is) -> {
        EObject _eContainer = is.eContainer();
        TModule _module = ((ImportDeclaration) _eContainer).getModule();
        return Boolean.valueOf((_module == targMod));
      };
      final NamespaceImportSpecifier is = IterableExtensions.<NamespaceImportSpecifier>findFirst(usedNamespaceSpecifiers.keySet(), _function);
      final Boolean used = usedNamespaceSpecifiers.get(is);
      ScriptDependency _xifexpression = null;
      if ((!(used).booleanValue())) {
        ScriptDependency _xblockexpression_1 = null;
        {
          usedNamespaceSpecifiers.put(is, Boolean.valueOf(true));
          String _alias = is.getAlias();
          _xblockexpression_1 = new ScriptDependency(_alias, null, 
            null, targMod);
        }
        _xifexpression = _xblockexpression_1;
      } else {
        _xifexpression = null;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private static Iterable<ScriptDependency> _handle(final EObject eo, final Map<String, NamedImportSpecifier> nameToNamedImportSpecifiers, final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final Function1<? super EObject, ? extends Boolean> compare) {
    return CollectionLiterals.<ScriptDependency>newArrayList();
  }
  
  private static Iterable<ScriptDependency> _handle(final TypeRef eo, final Map<String, NamedImportSpecifier> nameToNamedImportSpecifiers, final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final Function1<? super EObject, ? extends Boolean> compare) {
    return CollectionLiterals.<ScriptDependency>newArrayList();
  }
  
  private static Iterable<ScriptDependency> _handle(final Void eo, final Map<String, NamedImportSpecifier> nameToNamedImportSpecifiers, final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final Function1<? super EObject, ? extends Boolean> compare) {
    return CollectionLiterals.<ScriptDependency>newArrayList();
  }
  
  private static Iterable<ScriptDependency> _handle(final TFunction tFunction, final Map<String, NamedImportSpecifier> nameToNamedImportSpecifiers, final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final Function1<? super EObject, ? extends Boolean> compare) {
    return CollectionLiterals.<ScriptDependency>newArrayList();
  }
  
  private static Iterable<ScriptDependency> _handle(final N4ClassDeclaration eo, final Map<String, NamedImportSpecifier> nameToNamedImportSpecifiers, final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final Function1<? super EObject, ? extends Boolean> compare) {
    Iterable<ScriptDependency> _xblockexpression = null;
    {
      Type _definedType = eo.getDefinedType();
      final TClass tClass = ((TClass) _definedType);
      _xblockexpression = ScriptDependencyResolver.handle(tClass, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare);
    }
    return _xblockexpression;
  }
  
  private static Iterable<ScriptDependency> _handle(final N4InterfaceDeclaration eo, final Map<String, NamedImportSpecifier> nameToNamedImportSpecifiers, final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final Function1<? super EObject, ? extends Boolean> compare) {
    Iterable<ScriptDependency> _xblockexpression = null;
    {
      Type _definedType = eo.getDefinedType();
      final TInterface tInterface = ((TInterface) _definedType);
      _xblockexpression = ScriptDependencyResolver.handle(tInterface, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare);
    }
    return _xblockexpression;
  }
  
  private static Iterable<ScriptDependency> _handle(final ParameterizedTypeRef eo, final Map<String, NamedImportSpecifier> nameToNamedImportSpecifiers, final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final Function1<? super EObject, ? extends Boolean> compare) {
    if ((((eo.getDeclaredType() != null) && (eo.getDeclaredType().eContainer() instanceof TModule)) && 
      (compare.apply(eo.getDeclaredType())).booleanValue())) {
      final String typeName = RefNameUtil.findTypeName(eo);
      if ((typeName != null)) {
        String _name = eo.getDeclaredType().getName();
        Type _declaredType = eo.getDeclaredType();
        EObject _eContainer = eo.getDeclaredType().eContainer();
        ScriptDependency _scriptDependency = new ScriptDependency(typeName, _name, _declaredType, ((TModule) _eContainer));
        return CollectionLiterals.<ScriptDependency>newArrayList(_scriptDependency);
      }
    }
    return CollectionLiterals.<ScriptDependency>newArrayList();
  }
  
  /**
   * Resolves dependency from identifier reference.
   */
  private static Iterable<ScriptDependency> _handle(final IdentifierRef eo, final Map<String, NamedImportSpecifier> nameToNamedImportSpecifiers, final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final Function1<? super EObject, ? extends Boolean> compare) {
    IdentifiableElement _id = eo.getId();
    boolean _tripleEquals = (_id == null);
    if (_tripleEquals) {
      return CollectionLiterals.<ScriptDependency>newArrayList();
    }
    Boolean _apply = compare.apply(eo.getId());
    if ((_apply).booleanValue()) {
      String _findIdentifierName = RefNameUtil.findIdentifierName(eo);
      String _name = eo.getId().getName();
      IdentifiableElement _id_1 = eo.getId();
      EObject _eContainer = eo.getId().eContainer();
      ScriptDependency _scriptDependency = new ScriptDependency(_findIdentifierName, _name, _id_1, ((TModule) _eContainer));
      return CollectionLiterals.<ScriptDependency>newArrayList(_scriptDependency);
    } else {
      IdentifiableElement _id_2 = eo.getId();
      if ((_id_2 instanceof ModuleNamespaceVirtualType)) {
        IdentifiableElement _id_3 = eo.getId();
        final ModuleNamespaceVirtualType namespace = ((ModuleNamespaceVirtualType) _id_3);
        final TModule targMod = namespace.getModule();
        boolean _isNamespaceDependencyHandlingNeeded = ScriptDependencyResolver.isNamespaceDependencyHandlingNeeded(usedNamespaceSpecifiers, targMod);
        if (_isNamespaceDependencyHandlingNeeded) {
          return CollectionLiterals.<ScriptDependency>newArrayList(ScriptDependencyResolver.createDependencyOnNamespace(usedNamespaceSpecifiers, targMod));
        }
      }
    }
    return CollectionLiterals.<ScriptDependency>newArrayList();
  }
  
  private static Iterable<ScriptDependency> _handle(final TClass tClass, final Map<String, NamedImportSpecifier> nameToNamedImportSpecifiers, final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final Function1<? super EObject, ? extends Boolean> compare) {
    Iterable<ScriptDependency> _xblockexpression = null;
    {
      final ArrayList<Type> deps = new ArrayList<Type>();
      deps.add(tClass);
      final EList<ParameterizedTypeRef> interfaces = tClass.getImplementedInterfaceRefs();
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(interfaces);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        final Function1<ParameterizedTypeRef, Type> _function = (ParameterizedTypeRef it) -> {
          return it.getDeclaredType();
        };
        Iterables.<Type>addAll(deps, IterableExtensions.<Type>filterNull(ListExtensions.<ParameterizedTypeRef, Type>map(interfaces, _function)));
      }
      final ParameterizedTypeRef superClass = tClass.getSuperClassRef();
      if ((superClass != null)) {
        deps.add(superClass.getDeclaredType());
      }
      final Function1<Type, Boolean> _function_1 = (Type it) -> {
        return compare.apply(it);
      };
      final Function1<Type, ScriptDependency> _function_2 = (Type it) -> {
        return ScriptDependencyResolver.createScriptDependency(it, nameToNamedImportSpecifiers, usedNamespaceSpecifiers);
      };
      _xblockexpression = IterableExtensions.<Type, ScriptDependency>map(IterableExtensions.<Type>filter(deps, _function_1), _function_2);
    }
    return _xblockexpression;
  }
  
  private static Iterable<ScriptDependency> _handle(final TInterface tInterface, final Map<String, NamedImportSpecifier> nameToNamedImportSpecifiers, final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final Function1<? super EObject, ? extends Boolean> compare) {
    Iterable<ScriptDependency> _xblockexpression = null;
    {
      final ArrayList<Type> deps = new ArrayList<Type>();
      deps.add(tInterface);
      final EList<ParameterizedTypeRef> rs = tInterface.getSuperInterfaceRefs();
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(rs);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        final Function1<ParameterizedTypeRef, Type> _function = (ParameterizedTypeRef it) -> {
          return it.getDeclaredType();
        };
        Iterables.<Type>addAll(deps, IterableExtensions.<Type>filterNull(ListExtensions.<ParameterizedTypeRef, Type>map(rs, _function)));
      }
      final Function1<Type, Boolean> _function_1 = (Type it) -> {
        return compare.apply(it);
      };
      final Function1<Type, ScriptDependency> _function_2 = (Type it) -> {
        return ScriptDependencyResolver.createScriptDependency(it, nameToNamedImportSpecifiers, usedNamespaceSpecifiers);
      };
      _xblockexpression = IterableExtensions.<Type, ScriptDependency>map(IterableExtensions.<Type>filter(deps, _function_1), _function_2);
    }
    return _xblockexpression;
  }
  
  private static Iterable<ScriptDependency> handle(final EObject tClass, final Map<String, NamedImportSpecifier> nameToNamedImportSpecifiers, final Map<NamespaceImportSpecifier, Boolean> usedNamespaceSpecifiers, final Function1<? super EObject, ? extends Boolean> compare) {
    if (tClass instanceof TClass
         && compare != null) {
      return _handle((TClass)tClass, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare);
    } else if (tClass instanceof TInterface
         && compare != null) {
      return _handle((TInterface)tClass, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare);
    } else if (tClass instanceof N4ClassDeclaration
         && compare != null) {
      return _handle((N4ClassDeclaration)tClass, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare);
    } else if (tClass instanceof N4InterfaceDeclaration
         && compare != null) {
      return _handle((N4InterfaceDeclaration)tClass, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare);
    } else if (tClass instanceof TFunction
         && compare != null) {
      return _handle((TFunction)tClass, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare);
    } else if (tClass instanceof ParameterizedTypeRef
         && compare != null) {
      return _handle((ParameterizedTypeRef)tClass, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare);
    } else if (tClass instanceof IdentifierRef
         && compare != null) {
      return _handle((IdentifierRef)tClass, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare);
    } else if (tClass instanceof TypeRef
         && compare != null) {
      return _handle((TypeRef)tClass, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare);
    } else if (tClass != null
         && compare != null) {
      return _handle(tClass, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare);
    } else if (tClass == null
         && compare != null) {
      return _handle((Void)null, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(tClass, nameToNamedImportSpecifiers, usedNamespaceSpecifiers, compare).toString());
    }
  }
}
