/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.packagejson;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import java.util.Collection;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import org.eclipse.n4js.json.JSON.JSONDocument;
import org.eclipse.n4js.json.model.utils.JSONModelUtils;
import org.eclipse.n4js.packagejson.PackageJsonContentProvider;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectDescription.SourceContainerType;
import org.eclipse.n4js.utils.ProjectDescriptionLoader;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;

/**
 * Convenient builder for creating the N4JS package.json compliant {@link JSONDocument} model
 * instances or file content.#
 * 
 * Provides support for most supported N4JS-specific package.json properties.
 * See {@link ProjectDescriptionLoader} for details on all supported properties.
 */
@SuppressWarnings("all")
public class PackageJsonBuilder {
  /**
   * Creates a new builder instance.
   */
  public static PackageJsonBuilder newBuilder() {
    return new PackageJsonBuilder();
  }
  
  private String projectName;
  
  private ProjectType type;
  
  private String version;
  
  private SortedMap<String, String> dependencies;
  
  private String vendorId;
  
  private String vendorName;
  
  private String output;
  
  private Collection<String> providedRLs;
  
  private Collection<String> requiredRLs;
  
  private Collection<String> testedProjects;
  
  private String extendedRE;
  
  private Collection<String> implementedProjects;
  
  private String implementationId;
  
  private Map<SourceContainerType, String> sourceContainers;
  
  private PackageJsonBuilder() {
    this.type = ProjectType.VALIDATION;
    this.providedRLs = CollectionLiterals.<String>newArrayList();
    this.requiredRLs = CollectionLiterals.<String>newArrayList();
    TreeMap<String, String> _treeMap = new TreeMap<String, String>();
    this.dependencies = _treeMap;
    this.implementedProjects = CollectionLiterals.<String>newArrayList();
    this.testedProjects = CollectionLiterals.<String>newArrayList();
    this.sourceContainers = CollectionLiterals.<SourceContainerType, String>newHashMap();
  }
  
  /**
   * Builds the N4JS package.json file contents for a project with the given name.
   * 
   * @return the N4JS package.json file contents as a string.
   */
  public String build() {
    final JSONDocument document = this.buildModel();
    return JSONModelUtils.serializeJSON(document);
  }
  
  /**
   * Builds the N4JS package.json {@code JSONDocument} model representation.
   * 
   * @return the N4JS package.json {@code JSONDocument} representation.
   */
  public JSONDocument buildModel() {
    return PackageJsonContentProvider.getModel(
      Preconditions.<String>checkNotNull(this.projectName), 
      Optional.<String>fromNullable(this.version), 
      Preconditions.<ProjectType>checkNotNull(this.type), 
      Optional.<String>fromNullable(this.vendorId), 
      Optional.<String>fromNullable(this.vendorName), 
      Optional.<String>fromNullable(this.output), 
      Optional.<String>fromNullable(this.extendedRE), 
      this.dependencies, 
      this.providedRLs, 
      this.requiredRLs, 
      Optional.<String>fromNullable(this.implementationId), 
      this.implementedProjects, 
      this.testedProjects, 
      this.sourceContainers);
  }
  
  /**
   * Sets the project name and returns with the builder.
   * 
   * @param type The N4JS project name.
   * @return the builder.
   */
  public PackageJsonBuilder withName(final String name) {
    this.projectName = Preconditions.<String>checkNotNull(name);
    return this;
  }
  
  /**
   * Sets the project version and returns with the builder.
   * 
   * @param version The project version.
   * @return the builder.
   */
  public PackageJsonBuilder withVersion(final String version) {
    this.version = Preconditions.<String>checkNotNull(version);
    return this;
  }
  
  /**
   * Sets the project type and returns with the builder.
   * 
   * @param type the N4JS project type.
   * @return the builder.
   */
  public PackageJsonBuilder withType(final ProjectType type) {
    this.type = Preconditions.<ProjectType>checkNotNull(type);
    return this;
  }
  
  /**
   * Sets the vendorId and returns with the builder.
   * 
   * @param type The project's vendor ID.
   * @return the builder.
   */
  public PackageJsonBuilder withVendorId(final String vendorId) {
    this.vendorId = vendorId;
    return this;
  }
  
  /**
   * Sets the vendor name and returns with the builder.
   * 
   * @param type The project's vendor name.
   * @return the builder.
   */
  public PackageJsonBuilder withVendorName(final String vendorName) {
    this.vendorName = vendorName;
    return this;
  }
  
  /**
   * Sets the output folder and returns with the builder.
   * 
   * @param type The project's vendor name.
   * @return the builder.
   */
  public PackageJsonBuilder withOutput(final String output) {
    this.output = output;
    return this;
  }
  
  /**
   * Adds a source container with the given folder specifier and type.
   * 
   * @param type The source container type.
   * @param path The source container path.
   */
  public PackageJsonBuilder withSourceContainer(final SourceContainerType type, final String path) {
    this.sourceContainers.put(Preconditions.<SourceContainerType>checkNotNull(type), Preconditions.<String>checkNotNull(path));
    return this;
  }
  
  /**
   * Sets the extended runtime environment on the builder.
   * @param extendedRE the extended runtime environment. Optional. Can be {@code null}.
   * @return the builder.
   */
  public PackageJsonBuilder withExtendedRE(final String extendedRE) {
    this.extendedRE = extendedRE;
    return this;
  }
  
  /**
   * Adds a new provided runtime library to the current builder.
   * @param providedRL the provided runtime library to add to the current builder. Cannot be {@code null}.
   * @return the builder.
   */
  public PackageJsonBuilder withProvidedRL(final String providedRL) {
    this.providedRLs.add(Preconditions.<String>checkNotNull(providedRL));
    return this;
  }
  
  /**
   * Adds a new required runtime library to the current builder.
   * 
   * This method will add the required runtime library both to the "requiredRuntimeLibraries" section,
   * as well as the "dependencies" section.
   * 
   * @param requiredRL the required runtime library to add to the current builder. Cannot be {@code null}.
   * @return the builder.
   */
  public PackageJsonBuilder withRequiredRL(final String requiredRL) {
    this.requiredRLs.add(Preconditions.<String>checkNotNull(requiredRL));
    this.dependencies.put(requiredRL, "*");
    return this;
  }
  
  /**
   * Adds a new required runtime library with the given version constraint.
   * 
   * This method will add the required runtime library both to the "requiredRuntimeLibraries" section,
   * as well as the "dependencies" section.
   * 
   * @param requiredRL The project id of the required runtime library.
   * @param versionConstraint The version constraint to be used in the dependency section.
   */
  public PackageJsonBuilder withRequiredRL(final String requiredRL, final String versionConstraint) {
    this.requiredRLs.add(Preconditions.<String>checkNotNull(requiredRL));
    this.dependencies.put(requiredRL, versionConstraint);
    return this;
  }
  
  /**
   * Adds a new project dependency to the current builder.
   * @param projectDependency the project dependency to add to the current builder. Cannot be {@code null}.
   * @return the builder.
   */
  public PackageJsonBuilder withDependency(final String projectDependency) {
    this.dependencies.put(Preconditions.<String>checkNotNull(projectDependency), "*");
    return this;
  }
  
  /**
   * Adds a new project dependency to the current builder.
   * 
   * @param projectDependency The project dependency to add to the current builder. Cannot be {@code null}.
   * @param versionConstraint The version constraint of the added project dependency.
   * 
   * @return the builder.
   */
  public PackageJsonBuilder withDependency(final String projectDependency, final String versionConstraint) {
    this.dependencies.put(Preconditions.<String>checkNotNull(projectDependency), Preconditions.<String>checkNotNull(versionConstraint));
    return this;
  }
  
  /**
   * Adds the given project id to the list of tested projects.
   * 
   * This method also adds the given tested project as project dependency.
   */
  public PackageJsonBuilder withTestedProject(final String testedProject) {
    this.dependencies.put(Preconditions.<String>checkNotNull(testedProject), "*");
    this.testedProjects.add(Preconditions.<String>checkNotNull(testedProject));
    return this;
  }
  
  /**
   * Adds the given project id to the list of tested projects and to the list
   * of dependencies with the given version constraint.
   * 
   * This method also adds the given tested project as project dependency.
   */
  public PackageJsonBuilder withTestedProject(final String testedProject, final String version) {
    this.dependencies.put(Preconditions.<String>checkNotNull(testedProject), "*");
    this.testedProjects.add(Preconditions.<String>checkNotNull(testedProject));
    return this;
  }
  
  /**
   * Sets the implementation id to the current builder.
   * @param implementationId id for the implementations to choose from.
   * @return the builder.
   */
  public PackageJsonBuilder withImplementationId(final String implementationId) {
    this.implementationId = Preconditions.<String>checkNotNull(implementationId);
    return this;
  }
  
  /**
   * Adds a project to the list of implemented Projects.
   * NOTE: also call {@link withImplementationId()}
   * @param project id of the implemented API
   * @return the builder.
   */
  public PackageJsonBuilder withImplementedProject(final String implementationAPI) {
    this.implementedProjects.add(Preconditions.<String>checkNotNull(implementationAPI));
    return this;
  }
  
  @Override
  public String toString() {
    String _build = this.build();
    return ("!!! This is just a preview of the N4JS package.json file !!!\n" + _build);
  }
}
