/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesbuilder;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import org.eclipse.n4js.n4JS.ArrowFunction;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.GenericDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesbuilder.N4JSFormalParameterTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSTypesBuilderHelper;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Base class for functions and methods
 */
@SuppressWarnings("all")
class AbstractFunctionDefinitionTypesBuilder {
  @Inject
  @Extension
  private N4JSTypesBuilderHelper _n4JSTypesBuilderHelper;
  
  @Inject
  @Extension
  private N4JSFormalParameterTypesBuilder _n4JSFormalParameterTypesBuilder;
  
  protected void relinkFormalParameters(final TFunction functionType, final FunctionDefinition functionDef, final boolean preLinkingPhase) {
    final Function2<Integer, FormalParameter, Integer> _function = (Integer idx, FormalParameter fpar) -> {
      boolean _relinkFormalParameter = this._n4JSFormalParameterTypesBuilder.relinkFormalParameter(fpar, functionType, preLinkingPhase, (idx).intValue());
      if (_relinkFormalParameter) {
        return Integer.valueOf(((idx).intValue() + 1));
      }
      return idx;
    };
    IterableExtensions.<FormalParameter, Integer>fold(functionDef.getFpars(), Integer.valueOf(0), _function);
  }
  
  protected void addFormalParameters(final TFunction functionType, final FunctionDefinition functionDef, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    final Function1<FormalParameter, TFormalParameter> _function = (FormalParameter it) -> {
      return this._n4JSFormalParameterTypesBuilder.createFormalParameter(it, builtInTypeScope, preLinkingPhase);
    };
    Iterables.<TFormalParameter>addAll(functionType.getFpars(), 
      IterableExtensions.<TFormalParameter>filterNull(ListExtensions.<FormalParameter, TFormalParameter>map(functionDef.getFpars(), _function)));
  }
  
  /**
   * Transforms type variables from declaration (MethodDeclaration of FunctionDeclaration) to TFunction's type variables.
   */
  protected void addTypeVariables(final TFunction functionType, final GenericDeclaration genericDecl, final boolean preLinkingPhase) {
    this._n4JSTypesBuilderHelper.<TypeVariable>addCopyOfReferences(functionType.getTypeVars(), genericDecl.getTypeVars());
  }
  
  protected void setReturnType(final TGetter getterType, final N4GetterDeclaration getterDef, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    if ((!preLinkingPhase)) {
      TypeRef _xifexpression = null;
      TypeRef _declaredTypeRef = getterDef.getDeclaredTypeRef();
      boolean _tripleEquals = (_declaredTypeRef == null);
      if (_tripleEquals) {
        ParameterizedTypeRef _xifexpression_1 = null;
        if ((!preLinkingPhase)) {
          ParameterizedTypeRef _xifexpression_2 = null;
          boolean _isAbstract = getterType.isAbstract();
          if (_isAbstract) {
            _xifexpression_2 = builtInTypeScope.getAnyTypeRef();
          } else {
            _xifexpression_2 = this.inferReturnTypeFromReturnStatements(getterDef, builtInTypeScope);
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      } else {
        _xifexpression = getterDef.getDeclaredTypeRef();
      }
      final TypeRef inferredReturnTypeRef = _xifexpression;
      getterType.setDeclaredTypeRef(TypeUtils.<TypeRef>copyWithProxies(inferredReturnTypeRef));
    }
  }
  
  protected void setReturnType(final TFunction functionType, final FunctionDefinition functionDef, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    if ((!preLinkingPhase)) {
      TypeRef _xifexpression = null;
      TypeRef _returnTypeRef = functionDef.getReturnTypeRef();
      boolean _tripleEquals = (_returnTypeRef == null);
      if (_tripleEquals) {
        ParameterizedTypeRef _xifexpression_1 = null;
        if ((!preLinkingPhase)) {
          _xifexpression_1 = this.inferReturnTypeFromReturnStatements(functionDef, builtInTypeScope);
        }
        _xifexpression = _xifexpression_1;
      } else {
        _xifexpression = functionDef.getReturnTypeRef();
      }
      final TypeRef inferredReturnTypeRef = _xifexpression;
      functionType.setReturnTypeRef(TypeUtils.<TypeRef>copyWithProxies(inferredReturnTypeRef));
    }
  }
  
  /**
   * Poor man's return type inferencer
   */
  protected ParameterizedTypeRef inferReturnTypeFromReturnStatements(final FunctionDefinition definition, final BuiltInTypeScope builtInTypeScope) {
    final boolean hasNonVoidReturn = ((definition.getBody() != null) && definition.getBody().hasNonVoidReturn());
    if (hasNonVoidReturn) {
      return builtInTypeScope.getAnyTypeRef();
    } else {
      boolean _isSingleExprArrowFunction = this.isSingleExprArrowFunction(definition);
      if (_isSingleExprArrowFunction) {
        return builtInTypeScope.getAnyTypeRef();
      } else {
        return builtInTypeScope.getVoidTypeRef();
      }
    }
  }
  
  private boolean isSingleExprArrowFunction(final FunctionDefinition definition) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (definition instanceof ArrowFunction) {
      _matched=true;
      _switchResult = ((ArrowFunction)definition).isSingleExprImplicitReturn();
    }
    if (!_matched) {
      _switchResult = false;
    }
    return _switchResult;
  }
  
  /**
   * Poor man's return type inferencer
   */
  protected ParameterizedTypeRef inferReturnTypeFromReturnStatements(final N4GetterDeclaration definition, final BuiltInTypeScope builtInTypeScope) {
    ParameterizedTypeRef _xblockexpression = null;
    {
      final boolean hasNonVoidReturn = ((definition.getBody() != null) && definition.getBody().hasNonVoidReturn());
      ParameterizedTypeRef _xifexpression = null;
      if (hasNonVoidReturn) {
        _xifexpression = builtInTypeScope.getAnyTypeRef();
      } else {
        _xifexpression = builtInTypeScope.getVoidTypeRef();
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
}
