/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesbuilder;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import org.eclipse.n4js.n4JS.GenericDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDefinition;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.typesbuilder.N4JSFieldTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSGetterTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSMethodTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSSetterTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSTypesBuilderHelper;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Abstract base class for N4JSClassDeclarationTypesBuilder and N4JSInterfaceDeclarationTypesBuilder
 * to provide reusable bits and pieces.
 */
@SuppressWarnings("all")
abstract class N4JSClassifierDeclarationTypesBuilder {
  @Inject
  @Extension
  protected N4JSTypesBuilderHelper _n4JSTypesBuilderHelper;
  
  @Inject
  @Extension
  protected N4JSFieldTypesBuilder _n4JSFieldTypesBuilder;
  
  @Inject
  @Extension
  protected N4JSMethodTypesBuilder _n4JSMethodTypesBuilder;
  
  @Inject
  @Extension
  protected N4JSGetterTypesBuilder _n4JSGetterTypesBuilder;
  
  @Inject
  @Extension
  protected N4JSSetterTypesBuilder _n4JSSetterTypesBuilder;
  
  protected void addFields(final TClassifier classifier, final N4ClassifierDefinition definition, final boolean preLinkingPhase) {
    final Iterable<N4FieldDeclaration> n4Fields = Iterables.<N4FieldDeclaration>filter(definition.getOwnedMembers(), N4FieldDeclaration.class);
    final Function1<N4FieldDeclaration, TField> _function = (N4FieldDeclaration it) -> {
      return this._n4JSFieldTypesBuilder.createField(it, classifier, preLinkingPhase);
    };
    final Iterable<TField> fields = IterableExtensions.<TField>filterNull(IterableExtensions.<N4FieldDeclaration, TField>map(n4Fields, _function));
    Iterables.<TMember>addAll(classifier.getOwnedMembers(), fields);
  }
  
  protected void addMethods(final TClassifier classifier, final N4ClassifierDefinition definition, final boolean preLinkingPhase) {
    final Iterable<N4MethodDeclaration> n4Methods = Iterables.<N4MethodDeclaration>filter(definition.getOwnedMembers(), N4MethodDeclaration.class);
    final Function1<N4MethodDeclaration, TMethod> _function = (N4MethodDeclaration it) -> {
      return this._n4JSMethodTypesBuilder.createMethod(it, preLinkingPhase);
    };
    final Iterable<TMethod> methods = IterableExtensions.<TMethod>filterNull(IterableExtensions.<N4MethodDeclaration, TMethod>map(n4Methods, _function));
    Iterables.<TMember>addAll(classifier.getOwnedMembers(), methods);
    N4MethodDeclaration _ownedCallableCtor = definition.getOwnedCallableCtor();
    TMethod _createMethod = null;
    if (_ownedCallableCtor!=null) {
      _createMethod=this._n4JSMethodTypesBuilder.createMethod(_ownedCallableCtor, preLinkingPhase);
    }
    classifier.setCallableCtor(_createMethod);
  }
  
  protected void addGetters(final TClassifier classifier, final N4ClassifierDefinition definition, final boolean preLinkingPhase) {
    final Iterable<N4GetterDeclaration> n4Getters = Iterables.<N4GetterDeclaration>filter(definition.getOwnedMembers(), N4GetterDeclaration.class);
    final Function1<N4GetterDeclaration, TGetter> _function = (N4GetterDeclaration it) -> {
      return this._n4JSGetterTypesBuilder.createGetter(it, classifier, preLinkingPhase);
    };
    final Iterable<TGetter> getters = IterableExtensions.<TGetter>filterNull(IterableExtensions.<N4GetterDeclaration, TGetter>map(n4Getters, _function));
    Iterables.<TMember>addAll(classifier.getOwnedMembers(), getters);
  }
  
  protected void addSetters(final TClassifier classifier, final N4ClassifierDefinition definition, final boolean preLinkingPhase) {
    final Iterable<N4SetterDeclaration> n4Setters = Iterables.<N4SetterDeclaration>filter(definition.getOwnedMembers(), N4SetterDeclaration.class);
    final Function1<N4SetterDeclaration, TSetter> _function = (N4SetterDeclaration it) -> {
      return this._n4JSSetterTypesBuilder.createSetter(it, classifier, preLinkingPhase);
    };
    final Iterable<TSetter> setters = IterableExtensions.<TSetter>filterNull(IterableExtensions.<N4SetterDeclaration, TSetter>map(n4Setters, _function));
    Iterables.<TMember>addAll(classifier.getOwnedMembers(), setters);
  }
  
  protected void addTypeParameters(final TClassifier classifier, final GenericDeclaration decl, final boolean preLinkingPhase) {
    this._n4JSTypesBuilderHelper.<TypeVariable>addCopyOfReferences(classifier.getTypeVars(), decl.getTypeVars());
    for (int i = 0; (i < ((Object[])Conversions.unwrapArray(classifier.getTypeVars(), Object.class)).length); i++) {
      {
        TypeVariable typeVar = decl.getTypeVars().get(i);
        TypeVariable definedTypeVar = classifier.getTypeVars().get(i);
        typeVar.setDefinedTypeVariable(definedTypeVar);
      }
    }
  }
  
  void relinkClassifierAndMembers(final TClassifier classifier, final N4ClassifierDeclaration declaration, final boolean preLinkingPhase) {
    this._n4JSTypesBuilderHelper.ensureEqualName(declaration, classifier);
    N4MethodDeclaration _ownedCallableCtor = declaration.getOwnedCallableCtor();
    boolean _tripleNotEquals = (_ownedCallableCtor != null);
    if (_tripleNotEquals) {
      this._n4JSMethodTypesBuilder.relinkCallableCtor(declaration.getOwnedCallableCtor(), classifier, preLinkingPhase);
    }
    int memberIdx = 0;
    memberIdx = this.relinkFields(classifier, declaration, preLinkingPhase, memberIdx);
    memberIdx = this.relinkMethods(classifier, declaration, preLinkingPhase, memberIdx);
    memberIdx = this.relinkGetters(classifier, declaration, preLinkingPhase, memberIdx);
    memberIdx = this.relinkSetters(classifier, declaration, preLinkingPhase, memberIdx);
    classifier.setAstElement(declaration);
    declaration.setDefinedType(classifier);
  }
  
  protected int relinkFields(final TClassifier classifier, final N4ClassifierDefinition definition, final boolean preLinkingPhase, final int start) {
    final Function2<Integer, N4FieldDeclaration, Integer> _function = (Integer idx, N4FieldDeclaration fld) -> {
      boolean _relinkField = this._n4JSFieldTypesBuilder.relinkField(fld, classifier, preLinkingPhase, (idx).intValue());
      if (_relinkField) {
        return Integer.valueOf(((idx).intValue() + 1));
      }
      return idx;
    };
    return (int) IterableExtensions.<N4FieldDeclaration, Integer>fold(Iterables.<N4FieldDeclaration>filter(definition.getOwnedMembers(), N4FieldDeclaration.class), Integer.valueOf(start), _function);
  }
  
  protected int relinkMethods(final TClassifier classifier, final N4ClassifierDefinition definition, final boolean preLinkingPhase, final int start) {
    final Function2<Integer, N4MethodDeclaration, Integer> _function = (Integer idx, N4MethodDeclaration method) -> {
      boolean _relinkMethod = this._n4JSMethodTypesBuilder.relinkMethod(method, classifier, preLinkingPhase, (idx).intValue());
      if (_relinkMethod) {
        return Integer.valueOf(((idx).intValue() + 1));
      }
      return idx;
    };
    int result = (int) IterableExtensions.<N4MethodDeclaration, Integer>fold(Iterables.<N4MethodDeclaration>filter(definition.getOwnedMembers(), N4MethodDeclaration.class), Integer.valueOf(start), _function);
    return result;
  }
  
  protected int relinkGetters(final TClassifier classifier, final N4ClassifierDefinition definition, final boolean preLinkingPhase, final int start) {
    final Function2<Integer, N4GetterDeclaration, Integer> _function = (Integer idx, N4GetterDeclaration getter) -> {
      boolean _relinkGetter = this._n4JSGetterTypesBuilder.relinkGetter(getter, classifier, preLinkingPhase, (idx).intValue());
      if (_relinkGetter) {
        return Integer.valueOf(((idx).intValue() + 1));
      }
      return idx;
    };
    return (int) IterableExtensions.<N4GetterDeclaration, Integer>fold(Iterables.<N4GetterDeclaration>filter(definition.getOwnedMembers(), N4GetterDeclaration.class), Integer.valueOf(start), _function);
  }
  
  protected int relinkSetters(final TClassifier classifier, final N4ClassifierDefinition definition, final boolean preLinkingPhase, final int start) {
    final Function2<Integer, N4SetterDeclaration, Integer> _function = (Integer idx, N4SetterDeclaration setter) -> {
      boolean _relinkSetter = this._n4JSSetterTypesBuilder.relinkSetter(setter, classifier, preLinkingPhase, (idx).intValue());
      if (_relinkSetter) {
        return Integer.valueOf(((idx).intValue() + 1));
      }
      return idx;
    };
    return (int) IterableExtensions.<N4SetterDeclaration, Integer>fold(Iterables.<N4SetterDeclaration>filter(definition.getOwnedMembers(), N4SetterDeclaration.class), Integer.valueOf(start), _function);
  }
}
