/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesbuilder;

import com.google.common.base.Objects;
import com.google.common.base.Optional;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.ExportedVariableStatement;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.MethodDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassExpression;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.TypeDefiningElement;
import org.eclipse.n4js.naming.ModuleNameComputer;
import org.eclipse.n4js.naming.SpecifierConverter;
import org.eclipse.n4js.projectDescription.ModuleLoader;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.resource.N4JSResource;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.StructuralTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.ModuleNamespaceVirtualType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.n4js.typesbuilder.N4JSClassDeclarationTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSEnumDeclarationTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSFunctionDefinitionTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSInterfaceDeclarationTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSNamespaceImportTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSObjectLiteralTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSTypesBuilderHelper;
import org.eclipse.n4js.typesbuilder.N4JSTypesFromTypeRefBuilder;
import org.eclipse.n4js.typesbuilder.N4JSVariableStatementTypesBuilder;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.naming.IQualifiedNameConverter;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.parser.IParseResult;
import org.eclipse.xtext.resource.DerivedStateAwareResource;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * This class with its {@link N4JSTypesBuilder#createTModuleFromSource(DerivedStateAwareResource,boolean) createTModuleFromSource()}
 * method is the single entry point for the types builder package. The only exception is the public method
 * {@link N4JSFunctionDefinitionTypesBuilder#updateTFunction(FunctionExpression, FunctionTypeExprOrRef, TypeRef) updateTFunction()}
 * in N4JSFunctionDefinitionTypesBuilder, which is called from Xsemantics.
 * <p>
 * Derives the types model from the AST, i.e. creates a {@link TModule} with its contents
 * from a {@link Script} and its children. The types model is stored at the second index of the resource.
 * The types model contains for each exportable or type defining element a corresponding entry e.g.
 * for a {@link N4ClassDeclaration} a {@link TClass} is created. Later when linking only the types are
 * used in place of the original objects.
 * <p>
 * The types builder must not use type inference because this would lead to a resolution of lazy
 * cross-references at a too early stage. Instead, the types builder creates ComputedTypeRefs that
 * will later be resolved either on demand or by calling {@link N4JSResource#flattenModule()}.
 */
@SuppressWarnings("all")
public class N4JSTypesBuilder {
  @Inject(optional = true)
  private TypesFactory typesFactory = TypesFactory.eINSTANCE;
  
  @Inject
  @Extension
  private N4JSTypesBuilderHelper _n4JSTypesBuilderHelper;
  
  @Inject
  @Extension
  private N4JSClassDeclarationTypesBuilder _n4JSClassDeclarationTypesBuilder;
  
  @Inject
  @Extension
  private N4JSInterfaceDeclarationTypesBuilder _n4JSInterfaceDeclarationTypesBuilder;
  
  @Inject
  @Extension
  private N4JSEnumDeclarationTypesBuilder _n4JSEnumDeclarationTypesBuilder;
  
  @Inject
  @Extension
  private N4JSObjectLiteralTypesBuilder _n4JSObjectLiteralTypesBuilder;
  
  @Inject
  @Extension
  private N4JSFunctionDefinitionTypesBuilder _n4JSFunctionDefinitionTypesBuilder;
  
  @Inject
  @Extension
  private N4JSVariableStatementTypesBuilder _n4JSVariableStatementTypesBuilder;
  
  @Inject
  @Extension
  private N4JSTypesFromTypeRefBuilder _n4JSTypesFromTypeRefBuilder;
  
  @Inject
  @Extension
  private N4JSNamespaceImportTypesBuilder _n4JSNamespaceImportTypesBuilder;
  
  @Inject
  @Extension
  private ModuleNameComputer _moduleNameComputer;
  
  @Inject
  private IN4JSCore n4jscore;
  
  @Inject
  private IQualifiedNameConverter qualifiedNameConverter;
  
  @Inject
  private SpecifierConverter specifierConverter;
  
  @Inject
  protected JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * When demand-loading an AST for a resource that already has a TModule (usually retrieved from the index) by
   * calling SyntaxRelatedTElement#getAstElement(), we are facing a challenge: we could simply replace the original
   * TModule by a new TModule created in the same way as in the standard case of loading an empty resource from
   * source, i.e. with method {@link N4JSTypesBuilder#createTModuleFromSource(DerivedStateAwareResource, boolean)
   * #createTModuleFromSource()}. However, this would lead to the issue of all existing references to the original,
   * now replaced TModule to now have an invalid target object (not contained in a resource anymore, proxy resolution
   * impossible, etc.).
   * <p>
   * As a solution, this method provides a 2nd mode of the types builder in which not a new TModule is created from
   * the AST, but an existing TModule is reused, i.e. the types builder does not create anything but simply creates
   * the bidirectional links between AST nodes and TModule elements.
   * <p>
   * This method should be called after the AST has been loaded, with the original TModule at second position in the
   * resource's contents. If the AST and TModule were created from different versions of the source, checked via an
   * MD5 hash, or the rewiring fails for other reasons, an {@link IllegalStateException} is thrown. In that case, the
   * state of the AST and TModule are undefined (i.e. linking may have taken place partially).
   */
  public void relinkTModuleToSource(final DerivedStateAwareResource resource, final boolean preLinkingPhase) {
    final IParseResult parseResult = resource.getParseResult();
    if ((parseResult != null)) {
      EObject _rootASTElement = parseResult.getRootASTElement();
      final Script script = ((Script) _rootASTElement);
      EObject _get = resource.getContents().get(1);
      final TModule module = ((TModule) _get);
      final String astMD5New = N4JSASTUtils.md5Hex(resource);
      String _astMD5 = module.getAstMD5();
      boolean _notEquals = (!Objects.equal(astMD5New, _astMD5));
      if (_notEquals) {
        URI _uRI = resource.getURI();
        String _plus = ("cannot link existing TModule to new AST due to hash mismatch: " + _uRI);
        throw new IllegalStateException(_plus);
      }
      module.setReconciled(true);
      this._n4JSNamespaceImportTypesBuilder.relinkNamespaceTypes(script, module, preLinkingPhase);
      this.buildTypesFromTypeRefs(script, module, preLinkingPhase);
      this.relinkTypes(script, module, preLinkingPhase);
      module.setAstElement(script);
      script.setModule(module);
    } else {
      URI _uRI_1 = resource.getURI();
      String _plus_1 = ("resource has no parse result: " + _uRI_1);
      throw new IllegalStateException(_plus_1);
    }
  }
  
  /**
   * This method is the single entry point for the types builder package. The only exception is the public method
   * {@link N4JSFunctionDefinitionTypesBuilder#updateTFunction(FunctionExpression,FunctionTypeExprOrRef,TypeRef) updateTFunction()}
   * in N4JSFunctionDefinitionTypesBuilder, which is called from Xsemantics.
   * <p>
   * Creates an {@link TModule} with the module path name (@see {@link ModuleNameComputer#getModulePath(Resource)})
   * of the resource (replacing the dots with slashes). For all {@link ExportableElement} and
   * {@link TypeDefiningElement} corresponding types like {@link TClass}, {@link TInterface} are created and assigned
   * as containment references to {@link TModule}. Afterwards the so created {@link TModule} tree is browsed for
   * {@link TVariable}s which do not have a type reference yet. For these there right hand side expressions is
   * checked for a {@link TypeDefiningElement} for this a {@link ParameterizedTypeRef} is created having this element
   * as declared type. The parameterized type ref is then assigned as type ref to the {@link TVariable}.
   */
  public void createTModuleFromSource(final DerivedStateAwareResource resource, final boolean preLinkingPhase) {
    final IParseResult parseResult = resource.getParseResult();
    if ((parseResult != null)) {
      EObject _rootASTElement = parseResult.getRootASTElement();
      final Script script = ((Script) _rootASTElement);
      final TModule result = this.typesFactory.createTModule();
      result.setAstMD5(N4JSASTUtils.md5Hex(resource));
      result.setReconciled(false);
      QualifiedName qualifiedModuleName = this._moduleNameComputer.getQualifiedModuleName(resource);
      result.setQualifiedName(this.qualifiedNameConverter.toString(qualifiedModuleName));
      result.setPreLinkingPhase(preLinkingPhase);
      final Optional<? extends IN4JSProject> optionalProject = this.n4jscore.findProject(resource.getURI());
      boolean _isPresent = optionalProject.isPresent();
      if (_isPresent) {
        final IN4JSProject project = optionalProject.get();
        result.setProjectName(project.getProjectName());
        result.setVendorID(project.getVendorID());
        ModuleLoader _moduleLoader = project.getModuleLoader();
        String _literal = null;
        if (_moduleLoader!=null) {
          _literal=_moduleLoader.getLiteral();
        }
        result.setModuleLoader(_literal);
        final String mainModuleSpec = project.getMainModule();
        if ((mainModuleSpec != null)) {
          QualifiedName _qualifiedModuleName = this._moduleNameComputer.getQualifiedModuleName(resource);
          QualifiedName _qualifiedName = this.specifierConverter.toQualifiedName(mainModuleSpec);
          boolean _equals = Objects.equal(_qualifiedModuleName, _qualifiedName);
          result.setMainModule(_equals);
        }
      }
      this._n4JSTypesBuilderHelper.copyAnnotations(result, script, preLinkingPhase);
      this.buildNamespaceTypesFromModuleImports(script, result, preLinkingPhase);
      result.setN4jsdModule(this.jsVariantHelper.isExternalMode(script));
      result.setStaticPolyfillModule(N4JSLanguageUtils.isContainedInStaticPolyfillModule(result));
      result.setStaticPolyfillAware(N4JSLanguageUtils.isContainedInStaticPolyfillAware(result));
      this.buildTypesFromTypeRefs(script, result, preLinkingPhase);
      this.buildTypes(script, result, preLinkingPhase);
      result.setAstElement(script);
      script.setModule(result);
      ((N4JSResource) resource).sneakyAddToContent(result);
    } else {
      URI _uRI = resource.getURI();
      String _plus = (_uRI + " has no parse result.");
      throw new IllegalStateException(_plus);
    }
  }
  
  /**
   * Creates new {@link ModuleNamespaceVirtualType} instances for the namespace imports in {@code script}
   * and adds them to the given {@code target} module's {@link TModule#internalTypes}.
   */
  private void buildNamespaceTypesFromModuleImports(final Script script, final TModule target, final boolean preLinkingPhase) {
    if ((!preLinkingPhase)) {
      List<ImportDeclaration> _list = IterableExtensions.<ImportDeclaration>toList(Iterables.<ImportDeclaration>filter(script.getScriptElements(), ImportDeclaration.class));
      for (final ImportDeclaration importDeclaration : _list) {
        {
          final NamespaceImportSpecifier namespaceImport = this._n4JSNamespaceImportTypesBuilder.getNamespaceImportSpecifier(importDeclaration);
          if ((namespaceImport != null)) {
            Object _eGet = importDeclaration.eGet(N4JSPackage.eINSTANCE.getImportDeclaration_Module(), 
              false);
            final TModule importedModule = ((TModule) _eGet);
            EList<Type> _internalTypes = target.getInternalTypes();
            ModuleNamespaceVirtualType _createModuleNamespaceVirtualType = this._n4JSNamespaceImportTypesBuilder.createModuleNamespaceVirtualType(namespaceImport, importedModule);
            _internalTypes.add(_createModuleNamespaceVirtualType);
          }
        }
      }
    }
  }
  
  /**
   * Create types for those TypeRefs that define a type if they play the role of an AST node.
   * <p>
   * This has to be done up-front, because in the rest of the types builder code we do not know where such a TypeRef
   * shows up; to avoid having to check for them at every occurrence of a TypeRef, we do this ahead of the main types
   * builder phase.
   */
  private void buildTypesFromTypeRefs(final Script script, final TModule target, final boolean preLinkingPhase) {
    if ((!preLinkingPhase)) {
      List<TypeRef> _reverseView = ListExtensions.<TypeRef>reverseView(IteratorExtensions.<TypeRef>toList(Iterators.<TypeRef>filter(script.eAllContents(), TypeRef.class)));
      for (final TypeRef tr : _reverseView) {
        boolean _matched = false;
        if (tr instanceof StructuralTypeRef) {
          _matched=true;
          this._n4JSTypesFromTypeRefBuilder.createStructuralType(((StructuralTypeRef)tr), target);
        }
        if (!_matched) {
          if (tr instanceof FunctionTypeExpression) {
            _matched=true;
            this._n4JSTypesFromTypeRefBuilder.createTFunction(((FunctionTypeExpression)tr), target);
          }
        }
      }
    }
  }
  
  private void relinkTypes(final Script script, final TModule target, final boolean preLinkingPhase) {
    int topLevelTypesIdx = 0;
    int variableIndex = 0;
    Iterable<EObject> _iterable = IteratorExtensions.<EObject>toIterable(script.eAllContents());
    for (final EObject n : _iterable) {
      boolean _matched = false;
      if (n instanceof TypeDefiningElement) {
        _matched=true;
        topLevelTypesIdx = this.relinkType(n, target, preLinkingPhase, topLevelTypesIdx);
      }
      if (!_matched) {
        if (n instanceof ExportedVariableStatement) {
          _matched=true;
          variableIndex = this.relinkType(n, target, preLinkingPhase, variableIndex);
        }
      }
    }
  }
  
  protected int _relinkType(final TypeDefiningElement other, final TModule target, final boolean preLinkingPhase, final int idx) {
    EClass _eClass = null;
    if (other!=null) {
      _eClass=other.eClass();
    }
    String _name = _eClass.getName();
    String _plus = ("unknown subclass of TypeDefiningElement: " + _name);
    throw new IllegalArgumentException(_plus);
  }
  
  protected int _relinkType(final NamespaceImportSpecifier nsImpSpec, final TModule target, final boolean preLinkingPhase, final int idx) {
    return idx;
  }
  
  protected int _relinkType(final N4ClassDeclaration n4Class, final TModule target, final boolean preLinkingPhase, final int idx) {
    boolean _relinkTClass = this._n4JSClassDeclarationTypesBuilder.relinkTClass(n4Class, target, preLinkingPhase, idx);
    if (_relinkTClass) {
      return (idx + 1);
    }
    return idx;
  }
  
  protected int _relinkType(final N4ClassExpression n4Class, final TModule target, final boolean preLinkingPhase, final int idx) {
    this._n4JSClassDeclarationTypesBuilder.createTClass(n4Class, target, preLinkingPhase);
    return idx;
  }
  
  protected int _relinkType(final N4InterfaceDeclaration n4Interface, final TModule target, final boolean preLinkingPhase, final int idx) {
    boolean _relinkTInterface = this._n4JSInterfaceDeclarationTypesBuilder.relinkTInterface(n4Interface, target, preLinkingPhase, idx);
    if (_relinkTInterface) {
      return (idx + 1);
    }
    return idx;
  }
  
  protected int _relinkType(final N4EnumDeclaration n4Enum, final TModule target, final boolean preLinkingPhase, final int idx) {
    boolean _relinkTEnum = this._n4JSEnumDeclarationTypesBuilder.relinkTEnum(n4Enum, target, preLinkingPhase, idx);
    if (_relinkTEnum) {
      return (idx + 1);
    }
    return idx;
  }
  
  protected int _relinkType(final ObjectLiteral objectLiteral, final TModule target, final boolean preLinkingPhase, final int idx) {
    this._n4JSObjectLiteralTypesBuilder.createObjectLiteral(objectLiteral, target, preLinkingPhase);
    return idx;
  }
  
  protected int _relinkType(final MethodDeclaration n4MethodDecl, final TModule target, final boolean preLinkingPhase, final int idx) {
    return idx;
  }
  
  protected int _relinkType(final FunctionDeclaration n4FunctionDecl, final TModule target, final boolean preLinkingPhase, final int idx) {
    boolean _relinkTFunction = this._n4JSFunctionDefinitionTypesBuilder.relinkTFunction(n4FunctionDecl, target, preLinkingPhase, idx);
    if (_relinkTFunction) {
      return (idx + 1);
    }
    return idx;
  }
  
  /**
   * Function expressions are special, see {@link N4JSFunctionDefinitionTypesBuilder#createTFunction(FunctionExpression,TModule,boolean)}.
   */
  protected int _relinkType(final FunctionExpression n4FunctionExpr, final TModule target, final boolean preLinkingPhase, final int idx) {
    this._n4JSFunctionDefinitionTypesBuilder.createTFunction(n4FunctionExpr, target, preLinkingPhase);
    return idx;
  }
  
  protected int _relinkType(final ExportedVariableStatement n4VariableStatement, final TModule target, final boolean preLinkingPhase, final int idx) {
    return this._n4JSVariableStatementTypesBuilder.relinkVariableTypes(n4VariableStatement, target, preLinkingPhase, idx);
  }
  
  private void buildTypes(final Script script, final TModule target, final boolean preLinkingPhase) {
    Iterable<EObject> _iterable = IteratorExtensions.<EObject>toIterable(script.eAllContents());
    for (final EObject n : _iterable) {
      boolean _matched = false;
      if (n instanceof TypeDefiningElement) {
        _matched=true;
        this.createType(n, target, preLinkingPhase);
      }
      if (!_matched) {
        if (n instanceof ExportedVariableStatement) {
          _matched=true;
          this.createType(n, target, preLinkingPhase);
        }
      }
    }
  }
  
  protected void _createType(final TypeDefiningElement other, final TModule target, final boolean preLinkingPhase) {
    EClass _eClass = null;
    if (other!=null) {
      _eClass=other.eClass();
    }
    String _name = _eClass.getName();
    String _plus = ("unknown subclass of TypeDefiningElement: " + _name);
    throw new IllegalArgumentException(_plus);
  }
  
  protected void _createType(final NamespaceImportSpecifier nsImpSpec, final TModule target, final boolean preLinkingPhase) {
  }
  
  protected void _createType(final N4ClassDeclaration n4Class, final TModule target, final boolean preLinkingPhase) {
    this._n4JSClassDeclarationTypesBuilder.createTClass(n4Class, target, preLinkingPhase);
  }
  
  protected void _createType(final N4ClassExpression n4Class, final TModule target, final boolean preLinkingPhase) {
    this._n4JSClassDeclarationTypesBuilder.createTClass(n4Class, target, preLinkingPhase);
  }
  
  protected void _createType(final N4InterfaceDeclaration n4Interface, final TModule target, final boolean preLinkingPhase) {
    this._n4JSInterfaceDeclarationTypesBuilder.createTInterface(n4Interface, target, preLinkingPhase);
  }
  
  protected void _createType(final N4EnumDeclaration n4Enum, final TModule target, final boolean preLinkingPhase) {
    this._n4JSEnumDeclarationTypesBuilder.createTEnum(n4Enum, target, preLinkingPhase);
  }
  
  protected void _createType(final ObjectLiteral objectLiteral, final TModule target, final boolean preLinkingPhase) {
    this._n4JSObjectLiteralTypesBuilder.createObjectLiteral(objectLiteral, target, preLinkingPhase);
  }
  
  protected void _createType(final MethodDeclaration n4MethodDecl, final TModule target, final boolean preLinkingPhase) {
  }
  
  protected void _createType(final FunctionDeclaration n4FunctionDecl, final TModule target, final boolean preLinkingPhase) {
    this._n4JSFunctionDefinitionTypesBuilder.createTFunction(n4FunctionDecl, target, preLinkingPhase);
  }
  
  /**
   * Function expressions are special, see {@link N4JSFunctionDefinitionTypesBuilder#createTFunction(FunctionExpression,TModule,boolean)}.
   */
  protected void _createType(final FunctionExpression n4FunctionExpr, final TModule target, final boolean preLinkingPhase) {
    this._n4JSFunctionDefinitionTypesBuilder.createTFunction(n4FunctionExpr, target, preLinkingPhase);
  }
  
  protected void _createType(final ExportedVariableStatement n4VariableStatement, final TModule target, final boolean preLinkingPhase) {
    this._n4JSVariableStatementTypesBuilder.createVariableTypes(n4VariableStatement, target, preLinkingPhase);
  }
  
  protected int relinkType(final EObject n4Class, final TModule target, final boolean preLinkingPhase, final int idx) {
    if (n4Class instanceof N4ClassDeclaration) {
      return _relinkType((N4ClassDeclaration)n4Class, target, preLinkingPhase, idx);
    } else if (n4Class instanceof N4ClassExpression) {
      return _relinkType((N4ClassExpression)n4Class, target, preLinkingPhase, idx);
    } else if (n4Class instanceof N4InterfaceDeclaration) {
      return _relinkType((N4InterfaceDeclaration)n4Class, target, preLinkingPhase, idx);
    } else if (n4Class instanceof N4EnumDeclaration) {
      return _relinkType((N4EnumDeclaration)n4Class, target, preLinkingPhase, idx);
    } else if (n4Class instanceof ExportedVariableStatement) {
      return _relinkType((ExportedVariableStatement)n4Class, target, preLinkingPhase, idx);
    } else if (n4Class instanceof FunctionDeclaration) {
      return _relinkType((FunctionDeclaration)n4Class, target, preLinkingPhase, idx);
    } else if (n4Class instanceof FunctionExpression) {
      return _relinkType((FunctionExpression)n4Class, target, preLinkingPhase, idx);
    } else if (n4Class instanceof MethodDeclaration) {
      return _relinkType((MethodDeclaration)n4Class, target, preLinkingPhase, idx);
    } else if (n4Class instanceof ObjectLiteral) {
      return _relinkType((ObjectLiteral)n4Class, target, preLinkingPhase, idx);
    } else if (n4Class instanceof NamespaceImportSpecifier) {
      return _relinkType((NamespaceImportSpecifier)n4Class, target, preLinkingPhase, idx);
    } else if (n4Class instanceof TypeDefiningElement) {
      return _relinkType((TypeDefiningElement)n4Class, target, preLinkingPhase, idx);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(n4Class, target, preLinkingPhase, idx).toString());
    }
  }
  
  protected void createType(final EObject n4Class, final TModule target, final boolean preLinkingPhase) {
    if (n4Class instanceof N4ClassDeclaration) {
      _createType((N4ClassDeclaration)n4Class, target, preLinkingPhase);
      return;
    } else if (n4Class instanceof N4ClassExpression) {
      _createType((N4ClassExpression)n4Class, target, preLinkingPhase);
      return;
    } else if (n4Class instanceof N4InterfaceDeclaration) {
      _createType((N4InterfaceDeclaration)n4Class, target, preLinkingPhase);
      return;
    } else if (n4Class instanceof N4EnumDeclaration) {
      _createType((N4EnumDeclaration)n4Class, target, preLinkingPhase);
      return;
    } else if (n4Class instanceof ExportedVariableStatement) {
      _createType((ExportedVariableStatement)n4Class, target, preLinkingPhase);
      return;
    } else if (n4Class instanceof FunctionDeclaration) {
      _createType((FunctionDeclaration)n4Class, target, preLinkingPhase);
      return;
    } else if (n4Class instanceof FunctionExpression) {
      _createType((FunctionExpression)n4Class, target, preLinkingPhase);
      return;
    } else if (n4Class instanceof MethodDeclaration) {
      _createType((MethodDeclaration)n4Class, target, preLinkingPhase);
      return;
    } else if (n4Class instanceof ObjectLiteral) {
      _createType((ObjectLiteral)n4Class, target, preLinkingPhase);
      return;
    } else if (n4Class instanceof NamespaceImportSpecifier) {
      _createType((NamespaceImportSpecifier)n4Class, target, preLinkingPhase);
      return;
    } else if (n4Class instanceof TypeDefiningElement) {
      _createType((TypeDefiningElement)n4Class, target, preLinkingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(n4Class, target, preLinkingPhase).toString());
    }
  }
}
