/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import java.util.Objects;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.FieldAccessor;
import org.eclipse.n4js.ts.types.MemberAccessModifier;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TStructuralType;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.AndFunction1;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Utility class for filtering out structural members for different structural typing strategies.
 */
@SuppressWarnings("all")
public abstract class StructuralMembersPredicates {
  /**
   * Base predicate {@link Function1 function} for filtering out {@link TMember member}s which for sure cannot
   * be member of a structural part. Such as {@link TMember#isStatic() static} members or non-public ones.
   */
  private static class BaseStructuralMembersPredicate implements Function1<TMember, Boolean> {
    private final Type n4ObjectType;
    
    private final TMember object__proto__;
    
    private BaseStructuralMembersPredicate(final RuleEnvironment G) {
      this.n4ObjectType = Objects.<Type>requireNonNull(RuleEnvironmentExtensions.n4ObjectType(G));
      final Function1<TMember, Boolean> _function = (TMember it) -> {
        String _name = it.getName();
        return Boolean.valueOf(com.google.common.base.Objects.equal(_name, N4JSLanguageConstants.PROPERTY__PROTO__NAME));
      };
      this.object__proto__ = Objects.<TMember>requireNonNull(
        IterableExtensions.<TMember>findFirst(RuleEnvironmentExtensions.objectType(G).getOwnedMembers(), _function));
    }
    
    @Override
    public Boolean apply(final TMember it) {
      boolean _hasStructuralTypeContainer = this.hasStructuralTypeContainer(it);
      boolean _not = (!_hasStructuralTypeContainer);
      if (_not) {
        boolean _isPublicVisible = this.isPublicVisible(it);
        boolean _not_1 = (!_isPublicVisible);
        if (_not_1) {
          return Boolean.valueOf(false);
        }
        boolean _hasN4ObjectTypeContainer = this.hasN4ObjectTypeContainer(it);
        if (_hasN4ObjectTypeContainer) {
          return Boolean.valueOf(false);
        }
        boolean _isStatic = it.isStatic();
        if (_isStatic) {
          return Boolean.valueOf(false);
        }
        if ((it == this.object__proto__)) {
          return Boolean.valueOf(false);
        }
      }
      return Boolean.valueOf(true);
    }
    
    private boolean hasStructuralTypeContainer(final TMember it) {
      ContainerType<?> _containingType = it.getContainingType();
      return (_containingType instanceof TStructuralType);
    }
    
    private boolean hasN4ObjectTypeContainer(final TMember it) {
      ContainerType<?> _containingType = it.getContainingType();
      return (_containingType == this.n4ObjectType);
    }
    
    private boolean isPublicVisible(final TMember it) {
      int _compareTo = MemberAccessModifier.PUBLIC_INTERNAL.compareTo(it.getMemberAccessModifier());
      return (0 >= _compareTo);
    }
  }
  
  /**
   * Accepts all members but the {@code constructor}s.
   */
  public final static AndFunction1<TMember> MEMBERS_PREDICATE = AndFunction1.<TMember>conjunctionOf(
    ((Function1<TMember, Boolean>) (TMember it) -> {
      boolean _isConstructor = it.isConstructor();
      return Boolean.valueOf((!_isConstructor));
    }));
  
  /**
   * Predicate for accepting all field structural type members. Such as data {@link TField field}s and field {@link FieldAccessor accessor}s.
   */
  public final static AndFunction1<TMember> FIELDS_PREDICATE = AndFunction1.<TMember>conjunctionOf(
    ((Function1<TMember, Boolean>) (TMember it) -> {
      return Boolean.valueOf(((it instanceof TField) || (it instanceof FieldAccessor)));
    }));
  
  /**
   * Predicate for accepting readable fields such as {@link TMember#isReadable() readable} {@link TField data field}s
   * and {@link TGetter getter}s.
   */
  public final static AndFunction1<TMember> READABLE_FIELDS_PREDICATE = AndFunction1.<TMember>conjunctionOf(
    ((Function1<TMember, Boolean>) (TMember it) -> {
      return Boolean.valueOf((((it instanceof TField) && it.isReadable()) || (it instanceof TGetter)));
    }));
  
  /**
   * Predicate for writable fields. Accepts {@link TMember#isWriteable() writable} data {@link TField field}s and
   * {@link TSetter setter}s. Also accepts {@link TField#isFinal() @Final} fields without any {@link TField#isHasExpression()
   * initializer expression}.
   */
  public final static AndFunction1<TMember> WRITABLE_FIELDS_PREDICATE = AndFunction1.<TMember>conjunctionOf(
    ((Function1<TMember, Boolean>) (TMember it) -> {
      boolean _xifexpression = false;
      if ((it instanceof TField)) {
        _xifexpression = (((TField)it).isWriteable() || (((TField)it).isFinal() && (!((TField)it).isHasExpression())));
      } else {
        _xifexpression = (it instanceof TSetter);
      }
      return Boolean.valueOf(_xifexpression);
    }));
  
  /**
   * Predicate for write-only fields. Accepts only {@link TGetter getter}s.
   */
  public final static AndFunction1<TMember> GETTERS_PREDICATE = AndFunction1.<TMember>conjunctionOf(
    ((Function1<TMember, Boolean>) (TMember it) -> {
      return Boolean.valueOf((it instanceof TGetter));
    }));
  
  /**
   * Accepts only writable {@link FieldAccessor field accessor}s; {@link TSetter setter}s.
   */
  public final static AndFunction1<TMember> SETTERS_PREDICATE = AndFunction1.<TMember>conjunctionOf(
    ((Function1<TMember, Boolean>) (TMember it) -> {
      return Boolean.valueOf((it instanceof TSetter));
    }));
  
  /**
   * Creates and returns with a new predicate instance that can be used to filter out all members of a type that
   * cannot be a member of any structural types. For instance each {@link TMember#isStatic() static} and/or non-public
   * members will be filtered out.
   */
  public static AndFunction1<TMember> createBaseStructuralMembersPredicate(final RuleEnvironment it) {
    StructuralMembersPredicates.BaseStructuralMembersPredicate _baseStructuralMembersPredicate = new StructuralMembersPredicates.BaseStructuralMembersPredicate(it);
    return AndFunction1.<TMember>conjunctionOf(((Function1<TMember, Boolean>) _baseStructuralMembersPredicate));
  }
}
