/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.util.AbstractCompleteHierarchyTraverser;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Collects all members, including inherited members, of a type and omits some members overridden in the type hierarchy.
 * That is, members overridden in the type hierarchy are removed and replaced by the overridden members. However,
 * this is done lazily, that is not all cases are considered, as the result is further validated anyway.
 * That is,
 * <ul>
 * <li>data fields and corresponding accessors are not recognized to override each other
 * <li>methods (and fields) inherited from interfaces are all added and not filtered.
 * <li>private members are added to result (which is required for validation: cannot override private member etc.)
 * </ul>
 * Thus, this collector should only be used for validation purposes.
 */
@SuppressWarnings("all")
public class LazyOverrideAwareMemberCollector extends AbstractCompleteHierarchyTraverser<List<TMember>> {
  private final boolean includeImplicitSuperTypes;
  
  private List<TMember> result;
  
  private final RuleEnvironment G;
  
  private final boolean onlyInheritedMembers;
  
  /**
   * Collects all members, including owned members and members defined in implicite super types.
   */
  public static List<TMember> collectAllMembers(final ContainerType<?> type) {
    return new LazyOverrideAwareMemberCollector(type, true, false).getResult();
  }
  
  /**
   * Collects all inherited members, including members defined in implicit super types; owned members are omitted.
   */
  public static List<TMember> collectAllInheritedMembers(final ContainerType<?> type) {
    return new LazyOverrideAwareMemberCollector(type, true, true).getResult();
  }
  
  /**
   * Collects all declared members, including owned members; members defined in implicit super types are omitted.
   */
  public static List<TMember> collectAllDeclaredMembers(final ContainerType<?> type) {
    return new LazyOverrideAwareMemberCollector(type, false, false).getResult();
  }
  
  /**
   * Collects all declared inherited members; owned members and members defined in implicit super types are omitted.
   */
  public static List<TMember> collectAllDeclaredInheritedMembers(final ContainerType<?> type) {
    return new LazyOverrideAwareMemberCollector(type, false, true).getResult();
  }
  
  private List<TMember> createResultInstance() {
    return Lists.<TMember>newLinkedList();
  }
  
  /**
   * Creates a new collector with optional support for implicit super types, better use static helper methods.
   * 
   * @param type
   *            the base type. Must be contained in a resource set if <code>includeImplicitSuperTypes</code> is set to
   *            <code>true</code>.
   * @param includeImplicitSuperTypes
   *            if true also members of implicit super types will be collected; otherwise only members of declared
   *            super types are included.
   * @param onlyInheritedMembers
   * 			if true, owned members of type are ignore, that is only inherited members are collected
   * @throws IllegalArgumentException
   *             if <code>includeImplicitSuperTypes</code> is set to <code>true</code> and <code>type</code> is not
   *             contained in a properly initialized N4JS resource set.
   */
  private LazyOverrideAwareMemberCollector(final ContainerType<?> type, final boolean includeImplicitSuperTypes, final boolean onlyInheritedMembers) {
    super(type);
    this.onlyInheritedMembers = onlyInheritedMembers;
    this.includeImplicitSuperTypes = includeImplicitSuperTypes;
    this.result = this.createResultInstance();
    RuleEnvironment _xifexpression = null;
    if (includeImplicitSuperTypes) {
      _xifexpression = RuleEnvironmentExtensions.newRuleEnvironment(type);
    } else {
      _xifexpression = null;
    }
    this.G = _xifexpression;
  }
  
  @Override
  protected List<TMember> doGetResult() {
    return this.result;
  }
  
  @Override
  protected void doProcess(final ContainerType<?> type) {
    if ((type instanceof TClassifier)) {
      final EList<TMember> ownedMembers = ((TClassifier)type).getOwnedMembers();
      this.result.addAll(ownedMembers);
    }
  }
  
  protected boolean processAndReplace(final TClassifier type) {
    boolean _xblockexpression = false;
    {
      final EList<TMember> ownedMembers = type.getOwnedMembers();
      final Iterator<TMember> iterInherited = this.result.iterator();
      while (iterInherited.hasNext()) {
        {
          final TMember inherited = iterInherited.next();
          final Function1<TMember, Boolean> _function = (TMember it) -> {
            return Boolean.valueOf(((Objects.equal(it.getName(), inherited.getName()) && (it.getMemberType() == inherited.getMemberType())) && (Boolean.valueOf(it.isStatic()) == Boolean.valueOf(inherited.isStatic()))));
          };
          boolean _exists = IterableExtensions.<TMember>exists(ownedMembers, _function);
          if (_exists) {
            iterInherited.remove();
          }
        }
      }
      _xblockexpression = this.result.addAll(ownedMembers);
    }
    return _xblockexpression;
  }
  
  /**
   * Does not add owned members and add inherited members overridden aware.
   */
  @Override
  public Boolean caseTClass(final TClass object) {
    boolean _tryNext = this.guard.tryNext(object);
    if (_tryNext) {
      final List<TMember> parentResult = this.result;
      if ((object != this.bottomType)) {
        this.result = this.createResultInstance();
      }
      this.doProcess(this.getSuperTypes(object));
      this.doProcess(object.getImplementedInterfaceRefs());
      if (((!this.onlyInheritedMembers) || (!Objects.equal(object, this.bottomType)))) {
        this.processAndReplace(object);
      }
      if ((parentResult != this.result)) {
        parentResult.addAll(this.result);
        this.result = parentResult;
      }
    }
    return Boolean.FALSE;
  }
  
  /**
   * Does not add owned members and add inherited members overridden aware.
   */
  @Override
  public Boolean caseTInterface(final TInterface object) {
    boolean _tryNext = this.guard.tryNext(object);
    if (_tryNext) {
      final List<TMember> parentResult = this.result;
      if ((object != this.bottomType)) {
        this.result = this.createResultInstance();
      }
      this.doProcess(object.getSuperInterfaceRefs());
      this.doProcess(object.getSuperInterfaceRefs());
      if (((!this.onlyInheritedMembers) || (!Objects.equal(object, this.bottomType)))) {
        this.processAndReplace(object);
      }
      if ((parentResult != this.result)) {
        parentResult.addAll(this.result);
        this.result = parentResult;
      }
    }
    return Boolean.FALSE;
  }
  
  @Override
  protected List<ParameterizedTypeRef> getImplicitSuperTypes(final Type t) {
    if (this.includeImplicitSuperTypes) {
      final List<ParameterizedTypeRef> implSuperTypeRefs = new ArrayList<ParameterizedTypeRef>();
      List<ParameterizedTypeRef> _collectAllImplicitSuperTypes = RuleEnvironmentExtensions.collectAllImplicitSuperTypes(this.G, 
        TypeUtils.createTypeRef(t));
      for (final TypeRef currTypeRef : _collectAllImplicitSuperTypes) {
        implSuperTypeRefs.add(((ParameterizedTypeRef) currTypeRef));
      }
      return implSuperTypeRefs;
    } else {
      return Collections.<ParameterizedTypeRef>emptyList();
    }
  }
}
