/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.AnnotationArgument;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDefinition;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.NewExpression;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.ThisLiteral;
import org.eclipse.n4js.n4JS.TypeRefAnnotationArgument;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.typeRefs.UnknownTypeRef;
import org.eclipse.n4js.ts.types.BuiltInType;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TAnnotation;
import org.eclipse.n4js.ts.types.TAnnotationArgument;
import org.eclipse.n4js.ts.types.TAnnotationTypeRefArgument;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TN4Classifier;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.types.VirtualBaseType;
import org.eclipse.n4js.ts.types.util.AllSuperTypesCollector;
import org.eclipse.n4js.ts.types.util.SuperInterfacesIterable;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelper;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.xtext.scoping.IEObjectDescriptionWithError;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.scoping.IScopeProvider;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Validations related to dependency injection (covering annotations and instantiations).
 * <p>
 * For other DI-related validations (covering invocations of N4Injector methods) see {@link N4JSInjectorCallsitesValidator}
 */
@SuppressWarnings("all")
public class N4JSDependencyInjectionValidator extends AbstractN4JSDeclarativeValidator {
  /**
   * Record-like class to track the association between a binding and the UseBinder annotation
   * that brings it into the configuration of an injector of interest (not shown).
   */
  public static class Binding_UseBinder {
    private final TAnnotation bindingTAnn;
    
    private final Annotation useBinderAnn;
    
    public Binding_UseBinder(final TAnnotation bindingTAnn, final Annotation useBinderAnn) {
      this.bindingTAnn = bindingTAnn;
      this.useBinderAnn = useBinderAnn;
    }
  }
  
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  @Inject
  @Extension
  private ContainerTypesHelper _containerTypesHelper;
  
  @Inject
  @Extension
  private IScopeProvider _iScopeProvider;
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  /**
   * In case the argument refers to a class declaration, return its TClass. Otherwise null.
   */
  public static TClass tClassOf(final TypeRef ref) {
    if ((null != ref)) {
      Type _declaredType = ref.getDeclaredType();
      if ((_declaredType instanceof TClass)) {
        Type _declaredType_1 = ref.getDeclaredType();
        return ((TClass) _declaredType_1);
      }
    }
    return null;
  }
  
  /**
   * "new C" is invalid if:
   * <ul>
   * <li>C declares or inherits some member(s) annotated (at)Inject.</li>
   * <li>C or a super-type is marked (at)Injected, ie an API project marks a type so that implementing projects may subclass it adding injected members.</li>
   * </ul>
   */
  @Check
  public void checkNewExpression(final NewExpression newExpression) {
    Expression _callee = null;
    if (newExpression!=null) {
      _callee=newExpression.getCallee();
    }
    boolean _tripleEquals = (_callee == null);
    if (_tripleEquals) {
      return;
    }
    final TypeRef typeRef = this.ts.tau(newExpression.getCallee());
    if ((typeRef == null)) {
      return;
    }
    if ((typeRef instanceof UnknownTypeRef)) {
      return;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(newExpression);
    Type _xifexpression = null;
    if ((typeRef instanceof TypeTypeRef)) {
      _xifexpression = this.tsh.getStaticType(G, ((TypeTypeRef)typeRef));
    } else {
      _xifexpression = null;
    }
    final Type staticType = _xifexpression;
    if (((staticType == null) || staticType.eIsProxy())) {
      return;
    }
    if ((!(staticType instanceof TClass))) {
      return;
    }
    final TClass tClazz = ((TClass) staticType);
    boolean _requiresInjection = N4JSDependencyInjectionValidator.requiresInjection(tClazz);
    if (_requiresInjection) {
      this.addIssue(IssueCodes.getMessageForDI_MUST_BE_INJECTED(tClazz.getTypeAsString()), newExpression, N4JSPackage.eINSTANCE.getNewExpression_Callee(), IssueCodes.DI_MUST_BE_INJECTED);
    }
    boolean _isMarkedInjected = N4JSDependencyInjectionValidator.isMarkedInjected(tClazz);
    if (_isMarkedInjected) {
      this.addIssue(IssueCodes.getMessageForDI_API_INJECTED(), newExpression, N4JSPackage.eINSTANCE.getNewExpression_Callee(), IssueCodes.DI_API_INJECTED);
    }
  }
  
  @Check
  public void checkDependencyInjectionAnnotation(final Annotation ann) {
    String _name = ann.getName();
    boolean _matched = false;
    if (Objects.equal(_name, AnnotationDefinition.GENERATE_INJECTOR.name)) {
      _matched=true;
      this.internalCheckAnnotationInjector(ann);
    }
    if (!_matched) {
      if (Objects.equal(_name, AnnotationDefinition.BINDER.name)) {
        _matched=true;
        this.internalCheckAnnotationBinder(ann);
      }
    }
    if (!_matched) {
      if (Objects.equal(_name, AnnotationDefinition.BIND.name)) {
        _matched=true;
        this.internalCheckAnnotationBind(ann);
      }
    }
    if (!_matched) {
      if (Objects.equal(_name, AnnotationDefinition.WITH_PARENT_INJECTOR.name)) {
        _matched=true;
        this.internalCheckAnnotationWithParentInjector(ann);
      }
    }
    if (!_matched) {
      if (Objects.equal(_name, AnnotationDefinition.USE_BINDER.name)) {
        _matched=true;
        this.internalCheckAnnotationUseBinder(ann);
      }
    }
    if (!_matched) {
      if (Objects.equal(_name, AnnotationDefinition.PROVIDES.name)) {
        _matched=true;
        this.internalCheckAnnotationProvides(ann);
      }
    }
    if (!_matched) {
      if (Objects.equal(_name, AnnotationDefinition.INJECT.name)) {
        _matched=true;
        this.internalCheckAnnotationInject(ann);
      }
    }
    if (!_matched) {
      if (Objects.equal(_name, AnnotationDefinition.INJECTED.name)) {
        _matched=true;
        this.internalCheckAnnotationInjected(ann);
      }
    }
  }
  
  /**
   * In addition to the validations invoked directly from this method,
   * further validations (applicable to the formal params of an injected ctor)
   * are checked in {@link #internalCheckAnnotationInject}.
   */
  @Check
  public Object checkCtor(final N4MethodDeclaration it) {
    Object _xifexpression = null;
    if (((((null != it) && it.isConstructor()) && this.internalCheckCtorReferencesInjectedFields(it)) && this.internalCheckCtorInjectedWhenParentInjected(it))) {
      _xifexpression = null;
    }
    return _xifexpression;
  }
  
  /**
   * A constructor that's not annotated (at)Inject (in our case, the argument)
   * breaks the injection-chain if some inherited constructors exist
   * that are marked (at)Inject.
   * 
   * TODO clarification: is it "inherited constructors exist" or "other constructors --- inherited or not ---"
   * 
   * @return true iff no issues were found.
   */
  private boolean internalCheckCtorInjectedWhenParentInjected(final N4MethodDeclaration ctor) {
    boolean _hasAnnotation = AnnotationDefinition.INJECT.hasAnnotation(ctor);
    boolean _not = (!_hasAnnotation);
    if (_not) {
      N4ClassifierDefinition _owner = null;
      if (ctor!=null) {
        _owner=ctor.getOwner();
      }
      Type _definedType = null;
      if (_owner!=null) {
        _definedType=_owner.getDefinedType();
      }
      final Type currentType = _definedType;
      if ((currentType instanceof ContainerType<?>)) {
        final Function1<TMethod, Boolean> _function = (TMethod it) -> {
          return Boolean.valueOf(((it.isConstructor() && (it != ctor.getDefinedType())) && AnnotationDefinition.INJECT.hasAnnotation(it)));
        };
        final Iterable<TMethod> injectedParentInjectors = IterableExtensions.<TMethod>filter(Iterables.<TMethod>filter(this._containerTypesHelper.fromContext(ctor).allMembers(((ContainerType<?>)currentType)), TMethod.class), _function);
        boolean _isEmpty = IterableExtensions.isEmpty(injectedParentInjectors);
        boolean _not_1 = (!_isEmpty);
        if (_not_1) {
          final String currentName = ((ContainerType<?>)currentType).getName();
          TMethod _get = ((TMethod[])Conversions.unwrapArray(injectedParentInjectors, TMethod.class))[0];
          ContainerType<?> _containingType = null;
          if (_get!=null) {
            _containingType=_get.getContainingType();
          }
          String _name = null;
          if (_containingType!=null) {
            _name=_containingType.getName();
          }
          final String superName = _name;
          this.addIssue(IssueCodes.getMessageForDI_CTOR_BREAKS_INJECTION_CHAIN(superName, currentName), ctor, N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.DI_CTOR_BREAKS_INJECTION_CHAIN);
          return false;
        }
      }
    }
    return true;
  }
  
  /**
   * Scan the statements of ctor, focusing on field-reads of the form "this.f" where f is injected. For each of them:
   * <ul>
   * <il>the ctor itself must be marked (at)Inject,</il>
   * <il>and a formal param must exist whose type is (a subtype of) that of the field "f" being read.</il>
   * </ul>
   * 
   * @return true iff no issues were found.
   */
  private boolean internalCheckCtorReferencesInjectedFields(final N4MethodDeclaration ctor) {
    Block _body = ctor.getBody();
    boolean _tripleEquals = (null == _body);
    if (_tripleEquals) {
      return true;
    }
    final AtomicBoolean valid = new AtomicBoolean(true);
    final Procedure1<Statement> _function = (Statement stmt) -> {
      final Procedure1<EObject> _function_1 = (EObject propAcc) -> {
        boolean _isPropAccessOfInterest = this.isPropAccessOfInterest(propAcc);
        if (_isPropAccessOfInterest) {
          boolean _holdsCtorReferencesInjectedField = this.holdsCtorReferencesInjectedField(ctor, ((ParameterizedPropertyAccessExpression) propAcc));
          boolean _not = (!_holdsCtorReferencesInjectedField);
          if (_not) {
            valid.set(false);
          }
        }
      };
      IteratorExtensions.<EObject>forEach(EcoreUtil.<EObject>getAllContents(stmt, false), _function_1);
    };
    IteratorExtensions.<Statement>forEach(ctor.getBody().getAllStatements(), _function);
    return valid.get();
  }
  
  /**
   * For a given field-read of the form "this.f" where f is injected. Check whether:
   * <ul>
   * <il>the ctor itself is marked (at)Inject,</il>
   * <il>a formal param must exist whose type is (a subtype of) that of the field "f" being read.</il>
   * </ul>
   * 
   * @return true iff no issues were found.
   */
  private boolean holdsCtorReferencesInjectedField(final N4MethodDeclaration ctor, final ParameterizedPropertyAccessExpression propAccess) {
    boolean isValid = true;
    IdentifiableElement _property = propAccess.getProperty();
    final TField accessedField = ((TField) _property);
    final boolean isInjectedCtor = AnnotationDefinition.INJECT.hasAnnotation(ctor);
    if ((!isInjectedCtor)) {
      this.addIssue(IssueCodes.getMessageForDI_FIELD_IS_NOT_INJECTED_YET(accessedField.getName()), propAccess, IssueCodes.DI_FIELD_IS_NOT_INJECTED_YET);
      isValid = false;
    } else {
      final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(ctor);
      final Function1<FormalParameter, TypeRef> _function = (FormalParameter it) -> {
        return it.getDeclaredTypeRef();
      };
      final Function1<TypeRef, Boolean> _function_1 = (TypeRef it) -> {
        boolean _or = false;
        Type _declaredType = it.getDeclaredType();
        TypeRef _typeRef = null;
        if (accessedField!=null) {
          _typeRef=accessedField.getTypeRef();
        }
        Type _declaredType_1 = null;
        if (_typeRef!=null) {
          _declaredType_1=_typeRef.getDeclaredType();
        }
        boolean _tripleEquals = (_declaredType == _declaredType_1);
        if (_tripleEquals) {
          _or = true;
        } else {
          TypeRef _typeRef_1 = null;
          if (accessedField!=null) {
            _typeRef_1=accessedField.getTypeRef();
          }
          boolean _subtypeSucceeded = this.ts.subtypeSucceeded(G, it, _typeRef_1);
          _or = _subtypeSucceeded;
        }
        return Boolean.valueOf(_or);
      };
      final boolean someParamSubtypesFieldType = IterableExtensions.<TypeRef>exists(ListExtensions.<FormalParameter, TypeRef>map(ctor.getFpars(), _function), _function_1);
      if ((!someParamSubtypesFieldType)) {
        this.addIssue(IssueCodes.getMessageForDI_FIELD_IS_NOT_INJECTED_YET(accessedField.getName()), propAccess, IssueCodes.DI_FIELD_IS_NOT_INJECTED_YET);
        isValid = false;
      }
    }
    return isValid;
  }
  
  /**
   * Is the argument a field-read of the form "this.f" where f is injected?
   */
  private boolean isPropAccessOfInterest(final EObject eo) {
    if ((!(eo instanceof ParameterizedPropertyAccessExpression))) {
      return false;
    }
    final ParameterizedPropertyAccessExpression propAcc = ((ParameterizedPropertyAccessExpression) eo);
    IdentifiableElement _property = propAcc.getProperty();
    if ((_property instanceof TField)) {
      boolean _isLhsInFieldAssignment = this.isLhsInFieldAssignment(propAcc);
      boolean _not = (!_isLhsInFieldAssignment);
      if (_not) {
        Expression _target = propAcc.getTarget();
        if ((_target instanceof ThisLiteral)) {
          IdentifiableElement _property_1 = propAcc.getProperty();
          final TField accessedField = ((TField) _property_1);
          return (AnnotationDefinition.INJECT.hasAnnotation(accessedField) && this.isVisibleAt(accessedField, propAcc));
        }
      }
    }
    return false;
  }
  
  /**
   * Does the argument occur as LHS in a field assignment?
   */
  private boolean isLhsInFieldAssignment(final ParameterizedPropertyAccessExpression it) {
    boolean _xblockexpression = false;
    {
      EObject _eContainer = it.eContainer();
      if ((_eContainer instanceof AssignmentExpression)) {
        EObject _eContainer_1 = it.eContainer();
        final AssignmentExpression assignExp = ((AssignmentExpression) _eContainer_1);
        if (((assignExp.getLhs() == it) && (it.getProperty() instanceof TField))) {
          return true;
        }
      }
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  private boolean isVisibleAt(final TField field, final ParameterizedPropertyAccessExpression expr) {
    final IScope scope = this._iScopeProvider.getScope(expr, N4JSPackage.eINSTANCE.getParameterizedPropertyAccessExpression_Property());
    boolean _isErrorDescription = IEObjectDescriptionWithError.isErrorDescription(scope.getSingleElement(field));
    return (!_isErrorDescription);
  }
  
  /**
   * A class annotated (at)GenerateInjector:
   * <ul>
   * <li>may not extend another class</li>
   * <li>may not use binders that, taken together, contain duplicates</li>
   * <li>the constructor of DIC is either arg-less or injected</li>
   * </ul>
   * 
   * As per Sec. 11.2.1 "DI Components and Injectors"
   * 
   * IDE-1563 is about relaxing the first constraint above:
   * once a class is marked as injector, all its subclasses must be injectors too.
   */
  private void internalCheckAnnotationInjector(final Annotation ann) {
    final N4ClassDeclaration injtorClassDecl = N4JSDependencyInjectionValidator.getAnnotatedClass(ann);
    if ((null == injtorClassDecl)) {
      return;
    }
    ParameterizedTypeRef _superClassRef = injtorClassDecl.getSuperClassRef();
    boolean _tripleNotEquals = (_superClassRef != null);
    if (_tripleNotEquals) {
      this.addIssue(IssueCodes.getMessageForDI_ANN_INJECTOR_EXTENDS(), ann, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_INJECTOR_EXTENDS);
    }
    final Iterable<Annotation> usedBindersAnns = AnnotationDefinition.USE_BINDER.getAllAnnotations(injtorClassDecl);
    final ArrayList<N4JSDependencyInjectionValidator.Binding_UseBinder> collectedBindings = new ArrayList<N4JSDependencyInjectionValidator.Binding_UseBinder>();
    final Consumer<Annotation> _function = (Annotation usedBinderAnn) -> {
      final TypeRef binderTypeRef = N4JSDependencyInjectionValidator.getArgAsTypeRef(usedBinderAnn, 0);
      final TClass binderTClass = N4JSDependencyInjectionValidator.tClassOf(binderTypeRef);
      if ((null != binderTClass)) {
        final Iterable<TAnnotation> tbindings = AnnotationDefinition.BIND.getAllAnnotations(binderTClass);
        final Consumer<TAnnotation> _function_1 = (TAnnotation tbinding) -> {
          N4JSDependencyInjectionValidator.Binding_UseBinder _binding_UseBinder = new N4JSDependencyInjectionValidator.Binding_UseBinder(tbinding, usedBinderAnn);
          collectedBindings.add(_binding_UseBinder);
        };
        tbindings.forEach(_function_1);
      }
    };
    usedBindersAnns.forEach(_function);
    this.internalCheckNoDupBindingsAcrossBinders(collectedBindings, RuleEnvironmentExtensions.newRuleEnvironment(injtorClassDecl));
    final N4MethodDeclaration injtorCtor = injtorClassDecl.getOwnedCtor();
    if (((null != injtorCtor) && (!injtorCtor.getFpars().isEmpty()))) {
      boolean _hasAnnotation = AnnotationDefinition.INJECT.hasAnnotation(injtorCtor);
      boolean _not = (!_hasAnnotation);
      if (_not) {
        this.addIssue(IssueCodes.getMessageForDI_ANN_INJECTOR_CTOR_MUST_BE_INJECT(), injtorCtor, IssueCodes.DI_ANN_INJECTOR_CTOR_MUST_BE_INJECT);
      }
    }
  }
  
  /**
   * In source code, dicTClass stands for a class marked (at)GenerateInjector
   * with one or more (at)UseBinder(binderClass_i) annotations.
   * This method returns those binder classes.
   * 
   * TODO reuse this method in N4JSDependencyInjectionValidator.internalCheckAnnotationInjector
   */
  public static Iterable<TClass> usedBindersOf(final TClass dicTClass) {
    Iterable<TClass> _xblockexpression = null;
    {
      final Iterable<TAnnotation> usedBindersAnns = AnnotationDefinition.USE_BINDER.getAllAnnotations(dicTClass);
      final Function1<TAnnotation, TClass> _function = (TAnnotation usedBinderAnn) -> {
        TClass _xblockexpression_1 = null;
        {
          final TypeRef binderTypeRef = N4JSDependencyInjectionValidator.getArgAsTypeRef(usedBinderAnn, 0);
          _xblockexpression_1 = N4JSDependencyInjectionValidator.tClassOf(binderTypeRef);
        }
        return _xblockexpression_1;
      };
      _xblockexpression = IterableExtensions.<TAnnotation, TClass>map(usedBindersAnns, _function);
    }
    return _xblockexpression;
  }
  
  /**
   * All of the given bindings are part of the configuration of a single injector.
   * Duplicate keys are detected, and errors are issued at the (a)UseBinder annotations
   * that brought the duplicate bindings into the configuration.
   */
  private void internalCheckNoDupBindingsAcrossBinders(final List<N4JSDependencyInjectionValidator.Binding_UseBinder> collectedBindings, final RuleEnvironment G) {
    final HashMap<TypeRef, Annotation> seen = new HashMap<TypeRef, Annotation>();
    final Consumer<N4JSDependencyInjectionValidator.Binding_UseBinder> _function = (N4JSDependencyInjectionValidator.Binding_UseBinder bindingTAnnAndItsUseAnn) -> {
      final TAnnotation bindingTAnn = bindingTAnnAndItsUseAnn.bindingTAnn;
      final Annotation useBinderAnn = bindingTAnnAndItsUseAnn.useBinderAnn;
      final TypeRef extra = N4JSDependencyInjectionValidator.getArgAsTypeRef(bindingTAnn, 0);
      boolean _hasStructuralFlavor = N4JSDependencyInjectionValidator.hasStructuralFlavor(extra);
      boolean _not = (!_hasStructuralFlavor);
      if (_not) {
        final Annotation dupBinding = this.getDuplicate(extra, seen, G);
        if ((null != dupBinding)) {
          this.addIssue(IssueCodes.getMessageForDI_ANN_DUPLICATE_BINDING(), useBinderAnn, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_DUPLICATE_BINDING);
          this.addIssue(IssueCodes.getMessageForDI_ANN_DUPLICATE_BINDING(), dupBinding, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_DUPLICATE_BINDING);
        } else {
          seen.put(extra, useBinderAnn);
        }
      }
    };
    collectedBindings.forEach(_function);
  }
  
  /**
   * The N4ClassDefinition annotated by (at)Binder:
   * <ul>
   * <li>must not be abstract</li>
   * <li>must not extend another class</li>
   * <li>must not be annotated with (at)GenerateInjector</li>
   * </ul>
   * Additionally, duplicate bindings aren't allowed (ie, different (at)Bind annotations with the same key).
   * This method looks for duplicate bindings within the binder of interest.
   * Method {@link #internalCheckAnnotationInjector} considers all binders used by the given injector and detects duplicate bindings across all of them.
   * 
   * As per Sec. 11.2.6.4 "N4JS DI (at)Binder".
   */
  private void internalCheckAnnotationBinder(final Annotation binderAnn) {
    final N4ClassDeclaration binderClassDecl = N4JSDependencyInjectionValidator.getAnnotatedClass(binderAnn);
    if (((null == binderClassDecl) || binderClassDecl.isAbstract())) {
      this.addIssue(IssueCodes.getMessageForDI_ANN_BINDER_NOT_APPLICABLE(), binderAnn, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_BINDER_NOT_APPLICABLE);
      return;
    }
    ParameterizedTypeRef _superClassRef = binderClassDecl.getSuperClassRef();
    boolean _tripleNotEquals = (_superClassRef != null);
    if (_tripleNotEquals) {
      this.addIssue(IssueCodes.getMessageForDI_ANN_BINDER_EXTENDS(), binderAnn, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_BINDER_EXTENDS);
    }
    final TClass binderTClazz = binderClassDecl.getDefinedTypeAsClass();
    boolean _hasAnnotation = AnnotationDefinition.GENERATE_INJECTOR.hasAnnotation(binderTClazz);
    if (_hasAnnotation) {
      this.addIssue(IssueCodes.getMessageForDI_ANN_BINDER_AND_INJECTOR_DONT_GO_TOGETHER(), binderAnn, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_BINDER_AND_INJECTOR_DONT_GO_TOGETHER);
    }
    this.internalCheckNoDupBindings(AnnotationDefinition.BIND.getAllAnnotations(binderClassDecl), RuleEnvironmentExtensions.newRuleEnvironment(binderClassDecl));
  }
  
  /**
   * The arguments are (at)Bind annotations. In case two different bindings share the same key an issue is reported.
   */
  private void internalCheckNoDupBindings(final Iterable<Annotation> bindings, final RuleEnvironment G) {
    final HashMap<TypeRef, Annotation> seen = new HashMap<TypeRef, Annotation>();
    final Consumer<Annotation> _function = (Annotation binding) -> {
      final TypeRef extra = N4JSDependencyInjectionValidator.getArgAsTypeRef(binding, 0);
      boolean _hasStructuralFlavor = N4JSDependencyInjectionValidator.hasStructuralFlavor(extra);
      boolean _not = (!_hasStructuralFlavor);
      if (_not) {
        final Annotation dupBinding = this.getDuplicate(extra, seen, G);
        if ((null != dupBinding)) {
          this.addIssue(IssueCodes.getMessageForDI_ANN_DUPLICATE_BINDING(), binding, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_DUPLICATE_BINDING);
          this.addIssue(IssueCodes.getMessageForDI_ANN_DUPLICATE_BINDING(), dupBinding, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_DUPLICATE_BINDING);
        } else {
          seen.put(extra, binding);
        }
      }
    };
    bindings.forEach(_function);
  }
  
  /**
   * Is the just-encountered key ("extra") among those already seen? If so, return the annotation for its duplicate
   * (ie, the binding or useBinder annotation that brought that duplicate into the configuration).
   * Otherwise null.
   */
  private Annotation getDuplicate(final TypeRef extra, final HashMap<TypeRef, Annotation> seen, final RuleEnvironment G) {
    final Iterator<Map.Entry<TypeRef, Annotation>> iter = seen.entrySet().iterator();
    while (iter.hasNext()) {
      {
        final Map.Entry<TypeRef, Annotation> oldEntry = iter.next();
        final TypeRef old = oldEntry.getKey();
        boolean _equaltypeSucceeded = this.ts.equaltypeSucceeded(G, old, extra);
        if (_equaltypeSucceeded) {
          return oldEntry.getValue();
        }
      }
    }
    return null;
  }
  
  private static boolean hasStructuralFlavor(final TypeRef typeRef) {
    final TypingStrategy ts = typeRef.getTypingStrategy();
    return (Objects.equal(TypingStrategy.STRUCTURAL, ts) || Objects.equal(TypingStrategy.STRUCTURAL_FIELDS, ts));
  }
  
  /**
   * Bind annotation:
   * <ul>
   * <li>occurs in tandem with (at)Binder on the ExportDeclaration or N4ClassDeclaration it annotates</li>
   * <li>its first argument (key) denotes an injectable type</li>
   * <li>its second argument (target) subtypes the first one, is itself injectable, and moreover is non-abstract</li>
   * </ul>
   */
  private void internalCheckAnnotationBind(final Annotation ann) {
    boolean _and = false;
    if (!((this.holdsAnnotatedTClassIsAnnotatedWith(ann, AnnotationDefinition.BINDER) && this.holdsIsInjectableType(ann, N4JSDependencyInjectionValidator.getArgAsTypeRef(ann, 0))) && this.holdsSecondArgIsSubtypeOfFirstArg(ann))) {
      _and = false;
    } else {
      boolean _holdsHasValidTargetType = this.holdsHasValidTargetType(ann);
      _and = _holdsHasValidTargetType;
    }
  }
  
  /**
   * WithParentInjector annotation:
   * <ul>
   * <li>occurs in tandem with (at)GenerateInjector on the ExportDeclaration or N4ClassDeclaration it annotates</li>
   * <li>its first argument refers to an injector</li>
   * <li>holdsNoCycleBetweenUsedInjectors</li>
   * </ul>
   */
  private void internalCheckAnnotationWithParentInjector(final Annotation ann) {
    boolean _and = false;
    if (!(this.holdsAnnotatedTClassIsAnnotatedWith(ann, AnnotationDefinition.GENERATE_INJECTOR) && this.holdsArgumentIsTypeRefToTClassAnnotatedWith(ann, AnnotationDefinition.GENERATE_INJECTOR))) {
      _and = false;
    } else {
      boolean _holdsNoCycleBetweenUsedInjectors = this.holdsNoCycleBetweenUsedInjectors(ann);
      _and = _holdsNoCycleBetweenUsedInjectors;
    }
  }
  
  private boolean holdsNoCycleBetweenUsedInjectors(final Annotation it) {
    Type type = this.typeOfUseInjector(it);
    if ((null != type)) {
      final ArrayList<String> visitedTypes = CollectionLiterals.<String>newArrayList(type.getName());
      final Function1<TAnnotation, Boolean> _function = (TAnnotation it_1) -> {
        String _name = it_1.getName();
        return Boolean.valueOf(Objects.equal(AnnotationDefinition.WITH_PARENT_INJECTOR.name, _name));
      };
      TAnnotation annotation = IterableExtensions.<TAnnotation>findFirst(type.getAnnotations(), _function);
      while ((null != annotation)) {
        {
          type = this.typeOfUseInjector(annotation);
          if ((null == type)) {
            annotation = null;
          } else {
            final int indexOf = visitedTypes.indexOf(type.getName());
            if ((indexOf > (-1))) {
              StringConcatenation _builder = new StringConcatenation();
              String _join = IterableExtensions.join(visitedTypes, " > ");
              _builder.append(_join);
              _builder.append(" > ");
              String _name = type.getName();
              _builder.append(_name);
              this.addIssue(
                IssueCodes.getMessageForDI_ANN_USE_INJECTOR_CYCLE(_builder), it, 
                N4JSPackage.eINSTANCE.getAnnotation_Name(), 
                IssueCodes.DI_ANN_USE_INJECTOR_CYCLE);
              return false;
            }
            visitedTypes.add(type.getName());
            final Function1<TAnnotation, Boolean> _function_1 = (TAnnotation it_1) -> {
              String _name_1 = it_1.getName();
              return Boolean.valueOf(Objects.equal(AnnotationDefinition.WITH_PARENT_INJECTOR.name, _name_1));
            };
            annotation = IterableExtensions.<TAnnotation>findFirst(type.getAnnotations(), _function_1);
          }
        }
      }
    }
    return true;
  }
  
  private Type typeOfUseInjector(final Annotation it) {
    if (((Objects.equal(AnnotationDefinition.WITH_PARENT_INJECTOR.name, it.getName()) && (!IterableExtensions.isNullOrEmpty(it.getArgs()))) && (IterableExtensions.<AnnotationArgument>head(it.getArgs()) instanceof TypeRefAnnotationArgument))) {
      AnnotationArgument _head = IterableExtensions.<AnnotationArgument>head(it.getArgs());
      final TypeRefAnnotationArgument arg = ((TypeRefAnnotationArgument) _head);
      TypeRef _typeRef = arg.getTypeRef();
      if ((_typeRef instanceof ParameterizedTypeRef)) {
        TypeRef _typeRef_1 = arg.getTypeRef();
        return ((ParameterizedTypeRef) _typeRef_1).getDeclaredType();
      }
    }
    return null;
  }
  
  private Type typeOfUseInjector(final TAnnotation it) {
    if (((Objects.equal(AnnotationDefinition.WITH_PARENT_INJECTOR.name, it.getName()) && (!IterableExtensions.isNullOrEmpty(it.getArgs()))) && (IterableExtensions.<TAnnotationArgument>head(it.getArgs()) instanceof TAnnotationTypeRefArgument))) {
      TAnnotationArgument _head = IterableExtensions.<TAnnotationArgument>head(it.getArgs());
      final TAnnotationTypeRefArgument arg = ((TAnnotationTypeRefArgument) _head);
      TypeRef _typeRef = arg.getTypeRef();
      if ((_typeRef instanceof ParameterizedTypeRef)) {
        TypeRef _typeRef_1 = arg.getTypeRef();
        return ((ParameterizedTypeRef) _typeRef_1).getDeclaredType();
      }
    }
    return null;
  }
  
  /**
   * UseBinder annotation:
   * <ul>
   * <li>occurs in tandem with (at)GenerateInjector on the class it annotates</li>
   * <li>its first argument refers to a binder</li>
   * </ul>
   */
  private void internalCheckAnnotationUseBinder(final Annotation ann) {
    boolean _and = false;
    boolean _holdsAnnotatedTClassIsAnnotatedWith = this.holdsAnnotatedTClassIsAnnotatedWith(ann, AnnotationDefinition.GENERATE_INJECTOR);
    if (!_holdsAnnotatedTClassIsAnnotatedWith) {
      _and = false;
    } else {
      boolean _holdsArgumentIsTypeRefToTClassAnnotatedWith = this.holdsArgumentIsTypeRefToTClassAnnotatedWith(ann, AnnotationDefinition.BINDER);
      _and = _holdsArgumentIsTypeRefToTClassAnnotatedWith;
    }
  }
  
  /**
   * Provides annotates a method that:
   * <ul>
   * <li>belongs to a binder</li>
   * <li>has zero or more params whose types are all injectable</li>
   * <li>has a non-void, injectable return type</li>
   * </ul>
   */
  private void internalCheckAnnotationProvides(final Annotation ann) {
    boolean _and = false;
    boolean _holdsAnnotatedTMethodIsContainedInTClassAnnotatedWith = this.holdsAnnotatedTMethodIsContainedInTClassAnnotatedWith(ann, AnnotationDefinition.BINDER);
    if (!_holdsAnnotatedTMethodIsContainedInTClassAnnotatedWith) {
      _and = false;
    } else {
      boolean _holdsAnnotatedTMethodHasCorrectSignature = this.holdsAnnotatedTMethodHasCorrectSignature(ann);
      _and = _holdsAnnotatedTMethodHasCorrectSignature;
    }
  }
  
  /**
   * Inject annotation marks a field or constructor that may not belong to an interface and that:
   * <ul>
   * <li>for a field: has injectable type and neither such type nor any of its super-classes is an injector</li>
   * <li>for a method or constructor:
   * <ul>
   * <li>has zero or more params whose types are all injectable (the return type doesn't matter)</li>
   * <li>no param is variadic</li>
   * </ul>
   * </li>
   * </ul>
   * Method injection is not supported yet (an error is issued upon attempting to inject a method).
   * 
   * TODO injection point bindings need to be resolvable
   * (that check requires determining at compile time the injector in effect).
   */
  private void internalCheckAnnotationInject(final Annotation ann) {
    final EObject annElem = ann.getAnnotatedElement();
    final EObject annElemCont = annElem.eContainer();
    if ((annElemCont instanceof N4InterfaceDeclaration)) {
      this.addIssue(IssueCodes.getMessageForDI_ANN_INTERFACE_INJECTION_NOT_SUPPORTED(), ann, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_INTERFACE_INJECTION_NOT_SUPPORTED);
    }
    if ((annElem instanceof N4FieldDeclaration)) {
      TMember _definedTypeElement = ((N4FieldDeclaration)annElem).getDefinedTypeElement();
      if ((_definedTypeElement instanceof TField)) {
        TMember _definedTypeElement_1 = ((N4FieldDeclaration)annElem).getDefinedTypeElement();
        final TField field = ((TField) _definedTypeElement_1);
        final TypeRef ref = field.getTypeRef();
        TClass clazz = N4JSDependencyInjectionValidator.tClassOf(ref);
        if ((null != clazz)) {
          boolean valid = true;
          while ((null != clazz)) {
            if ((!valid)) {
              clazz = null;
            } else {
              boolean _hasAnnotation = AnnotationDefinition.GENERATE_INJECTOR.hasAnnotation(clazz);
              boolean _not = (!_hasAnnotation);
              valid = _not;
              if ((!valid)) {
                this.addIssue(
                  IssueCodes.getMessageForDI_ANN_INJECTOR_CANNOT_BE_INJECTED_INTO_INJECTOR(), ann, 
                  N4JSPackage.eINSTANCE.getAnnotation_Name(), 
                  IssueCodes.DI_ANN_INJECTOR_CANNOT_BE_INJECTED_INTO_INJECTOR);
              }
              ParameterizedTypeRef _superClassRef = null;
              if (clazz!=null) {
                _superClassRef=clazz.getSuperClassRef();
              }
              Type _declaredType = null;
              if (_superClassRef!=null) {
                _declaredType=_superClassRef.getDeclaredType();
              }
              final Type type = _declaredType;
              TClass _xifexpression = null;
              if ((type instanceof TClass)) {
                _xifexpression = ((TClass)type);
              } else {
                _xifexpression = null;
              }
              clazz = _xifexpression;
            }
          }
          if ((!valid)) {
            return;
          }
        }
      }
    }
    TMember _xifexpression = null;
    if ((annElem instanceof N4MemberDeclaration)) {
      _xifexpression = ((N4MemberDeclaration)annElem).getDefinedTypeElement();
    } else {
      _xifexpression = null;
    }
    final TMember defMember = _xifexpression;
    boolean _matched = false;
    if (defMember instanceof TField) {
      _matched=true;
      this.holdsIsInjectableType(ann, ((TField)defMember).getTypeRef(), ((TField)defMember).getName());
    }
    if (!_matched) {
      if (defMember instanceof TMethod) {
        _matched=true;
        boolean _isConstructor = ((TMethod)defMember).isConstructor();
        if (_isConstructor) {
          final Consumer<TFormalParameter> _function = (TFormalParameter it) -> {
            this.holdsIsInjectableType(ann, it);
          };
          ((TMethod)defMember).getFpars().forEach(_function);
        } else {
          this.addIssue(IssueCodes.getMessageForDI_ANN_INJECT_METHOD_NOT_SUPPORTED_YET(), ann, N4JSPackage.eINSTANCE.getAnnotation_Name(), 
            IssueCodes.DI_ANN_INJECT_METHOD_NOT_SUPPORTED_YET);
        }
      }
    }
  }
  
  /**
   * The class marked by the given annotation should also carry another (required) annotation.
   * For example, an (at)Bind annotation may only mark a class that's also annotated (at)Binder.
   * 
   * @return true iff no issues were found
   */
  private boolean holdsAnnotatedTClassIsAnnotatedWith(final Annotation ann, final AnnotationDefinition requiredDef) {
    final N4ClassDeclaration classDecl = N4JSDependencyInjectionValidator.getAnnotatedClass(ann);
    Type _definedType = null;
    if (classDecl!=null) {
      _definedType=classDecl.getDefinedType();
    }
    final Type tClass = _definedType;
    if (((tClass != null) && (!requiredDef.hasAnnotation(tClass)))) {
      this.addIssue(IssueCodes.getMessageForDI_ANN_ONLY_ON_CLASS_ANNOTATED_WITH(ann.getName(), requiredDef.name), ann, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_ONLY_ON_CLASS_ANNOTATED_WITH);
      return false;
    }
    return true;
  }
  
  /**
   * The method marked by the given annotation belong to a class that in turn should carry another (required) annotation.
   * For example, an (at)Provides annotation may only mark a method of an (at)Binder class.
   * 
   * @return true iff no issues were found
   */
  private boolean holdsAnnotatedTMethodIsContainedInTClassAnnotatedWith(final Annotation ann, final AnnotationDefinition requiredDef) {
    final N4MethodDeclaration methodDecl = N4JSDependencyInjectionValidator.getAnnotatedMethod(ann);
    Type _definedType = null;
    if (methodDecl!=null) {
      _definedType=methodDecl.getDefinedType();
    }
    EObject _eContainer = null;
    if (_definedType!=null) {
      _eContainer=_definedType.eContainer();
    }
    final EObject tClass = _eContainer;
    if (((tClass instanceof TClass) && (!requiredDef.hasAnnotation(((TClass) tClass))))) {
      this.addIssue(IssueCodes.getMessageForDI_ANN_ONLY_ON_METHOD_IN_CLASS_ANNOTATED_WITH(ann.getName(), requiredDef.name), ann, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_ONLY_ON_METHOD_IN_CLASS_ANNOTATED_WITH);
      return false;
    }
    return true;
  }
  
  /**
   * The method annotated by the argument:
   * <ul>
   * <li>has zero or more params whose types are all injectable</li>
   * <li>has a non-void, injectable return type</li>
   * </ul>
   * 
   * @return true iff no issues were found
   */
  private boolean holdsAnnotatedTMethodHasCorrectSignature(final Annotation ann) {
    final N4MethodDeclaration methodDecl = N4JSDependencyInjectionValidator.getAnnotatedMethod(ann);
    Type _definedType = null;
    if (methodDecl!=null) {
      _definedType=methodDecl.getDefinedType();
    }
    final Type tMethod = _definedType;
    if ((tMethod instanceof TMethod)) {
      final Function1<TFormalParameter, Boolean> _function = (TFormalParameter fpar) -> {
        boolean _isInjectableType = N4JSDependencyInjectionValidator.isInjectableType(fpar.getTypeRef());
        return Boolean.valueOf((!_isInjectableType));
      };
      final Iterable<TFormalParameter> nonInjectableParams = IterableExtensions.<TFormalParameter>filter(((TMethod)tMethod).getFpars(), _function);
      final Consumer<TFormalParameter> _function_1 = (TFormalParameter fpar) -> {
        String _typeRefAsString = fpar.getTypeRef().getTypeRefAsString();
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("at ");
        String _name = fpar.getName();
        _builder.append(_name);
        this.addIssue(
          IssueCodes.getMessageForDI_NOT_INJECTABLE(_typeRefAsString, _builder), 
          fpar.getAstElement(), 
          TypesPackage.eINSTANCE.getIdentifiableElement_Name(), 
          IssueCodes.DI_NOT_INJECTABLE);
      };
      nonInjectableParams.forEach(_function_1);
      boolean _isEmpty = IterableExtensions.isEmpty(nonInjectableParams);
      boolean _not = (!_isEmpty);
      if (_not) {
        return false;
      }
      final TypeRef retTR = ((TMethod)tMethod).getReturnTypeRef();
      final boolean isVoidOrOptional = (TypeUtils.isVoid(retTR) || ((TMethod)tMethod).isReturnValueOptional());
      if (isVoidOrOptional) {
        this.addIssue(IssueCodes.getMessageForDI_ANN_PROVIDES_METHOD_MUST_RETURN_VALUE(), methodDecl, N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.DI_ANN_PROVIDES_METHOD_MUST_RETURN_VALUE);
        return false;
      }
      return this.holdsIsInjectableType(ann, ((TMethod)tMethod).getReturnTypeRef());
    }
    return true;
  }
  
  /**
   * The annotation's first argument is expected to refer to a class marked as requested.
   * If not, issue an error.
   */
  private boolean holdsArgumentIsTypeRefToTClassAnnotatedWith(final Annotation ann, final AnnotationDefinition requiredDef) {
    AnnotationArgument _xifexpression = null;
    boolean _isEmpty = ann.getArgs().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      _xifexpression = ann.getArgs().get(0);
    } else {
      _xifexpression = null;
    }
    final AnnotationArgument arg = _xifexpression;
    if ((null == arg)) {
      return true;
    }
    TypeRef _xifexpression_1 = null;
    if ((arg instanceof TypeRefAnnotationArgument)) {
      _xifexpression_1 = ((TypeRefAnnotationArgument)arg).getTypeRef();
    } else {
      _xifexpression_1 = null;
    }
    final TypeRef argTypeRef = _xifexpression_1;
    boolean _isTypeRefToTClassAnnotatedWith = N4JSDependencyInjectionValidator.isTypeRefToTClassAnnotatedWith(argTypeRef, requiredDef);
    boolean _not_1 = (!_isTypeRefToTClassAnnotatedWith);
    if (_not_1) {
      this.addIssue(IssueCodes.getMessageForDI_ANN_ARG_MUST_BE_ANNOTATED_WITH(ann.getName(), requiredDef.name), arg, IssueCodes.DI_ANN_ARG_MUST_BE_ANNOTATED_WITH);
      return false;
    }
    return true;
  }
  
  /**
   * Is the type-declaration annotated as requested?
   */
  private static boolean isTypeRefToTClassAnnotatedWith(final TypeRef typeRef, final AnnotationDefinition requiredAnn) {
    final Type tclazz = N4JSDependencyInjectionValidator.tClassOf(typeRef);
    return ((null != tclazz) && requiredAnn.hasAnnotation(tclazz));
  }
  
  /**
   * The (binding) annotation's second arg (target) is expected to subtype the first arg (key).
   * If not, issue an error.
   */
  private boolean holdsSecondArgIsSubtypeOfFirstArg(final Annotation ann) {
    Resource _eResource = null;
    if (ann!=null) {
      _eResource=ann.eResource();
    }
    ResourceSet _resourceSet = null;
    if (_eResource!=null) {
      _resourceSet=_eResource.getResourceSet();
    }
    boolean _tripleEquals = (_resourceSet == null);
    if (_tripleEquals) {
      return true;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(ann);
    final TypeRef arg0TypeRef = N4JSDependencyInjectionValidator.getArgAsTypeRef(ann, 0);
    final TypeRef arg1TypeRef = N4JSDependencyInjectionValidator.getArgAsTypeRef(ann, 1);
    if ((((arg0TypeRef != null) && (arg1TypeRef != null)) && (!this.ts.subtypeSucceeded(G, arg1TypeRef, arg0TypeRef)))) {
      this.addIssue(IssueCodes.getMessageForDI_ANN_BIND_SECOND_MUST_BE_SUBTYPE_FIRST(ann.getName()), ann.getArgs().get(1), IssueCodes.DI_ANN_BIND_SECOND_MUST_BE_SUBTYPE_FIRST);
      return false;
    }
    return true;
  }
  
  /**
   * The given formal param belongs to an injected constructor. Such param is expected to be non-variadic, non-optional,
   * and have an injectable type. If not, issue an error.
   */
  private boolean holdsIsInjectableType(final Annotation ann, final TFormalParameter it) {
    if ((null == it)) {
      return true;
    }
    boolean _isVariadicOrOptional = it.isVariadicOrOptional();
    if (_isVariadicOrOptional) {
      this.addIssue(IssueCodes.getMessageForDI_VARARGS_NOT_INJECTABLE(), it.getAstElement(), 
        TypesPackage.eINSTANCE.getIdentifiableElement_Name(), IssueCodes.DI_VARARGS_NOT_INJECTABLE);
      return false;
    }
    return this.holdsIsInjectableType(ann, it.getTypeRef(), it.getName());
  }
  
  private boolean holdsIsInjectableType(final Annotation ann, final TypeRef typeRef) {
    return this.holdsIsInjectableType(ann, typeRef, ((String) null));
  }
  
  /**
   * A Bind annotation must have an injectable and moreover non-abstract target.
   * 
   * @return true iff no issues were found
   */
  private boolean holdsHasValidTargetType(final Annotation bindAnn) {
    final TypeRef targetTypeRef = N4JSDependencyInjectionValidator.getArgAsTypeRef(bindAnn, 1);
    if ((targetTypeRef == null)) {
      return false;
    }
    boolean _holdsIsInjectableType = this.holdsIsInjectableType(bindAnn, targetTypeRef, ((String) null));
    boolean _not = (!_holdsIsInjectableType);
    if (_not) {
      return false;
    }
    boolean _isConcrete = N4JSDependencyInjectionValidator.isConcrete(targetTypeRef);
    boolean _not_1 = (!_isConcrete);
    if (_not_1) {
      this.addIssue(IssueCodes.getMessageForDI_ANN_BIND_ABSTRACT_TARGET(), bindAnn, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_BIND_ABSTRACT_TARGET);
      return false;
    }
    return true;
  }
  
  /**
   * Does the argument denote a concrete class? (ie, non-abstract class, non-interface)
   */
  private static boolean isConcrete(final TypeRef typeRef) {
    return N4JSDependencyInjectionValidator.isConcrete(typeRef.getDeclaredType());
  }
  
  /**
   * Is the argument a concrete class? (ie, non-abstract class, non-interface)
   */
  private static boolean isConcrete(final Type declType) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (declType instanceof TClass) {
      _matched=true;
      boolean _isAbstract = ((TClass)declType).isAbstract();
      _switchResult = (!_isAbstract);
    }
    if (!_matched) {
      _switchResult = false;
    }
    return _switchResult;
  }
  
  /**
   * The typeRef argument is expected to denote an injectable type.
   * If not, issue an error at the position of the annotation.
   * 
   * @return true iff no issues were found
   */
  private boolean holdsIsInjectableType(final Annotation ann, final TypeRef typeRef, final String name) {
    if ((typeRef == null)) {
      return true;
    }
    Type _declaredType = typeRef.getDeclaredType();
    boolean _tripleNotEquals = (_declaredType != null);
    if (_tripleNotEquals) {
      boolean _eIsProxy = typeRef.getDeclaredType().eIsProxy();
      if (_eIsProxy) {
        return true;
      }
    }
    boolean _isInjectableType = N4JSDependencyInjectionValidator.isInjectableType(typeRef);
    boolean _not = (!_isInjectableType);
    if (_not) {
      String _xifexpression = null;
      if ((null == name)) {
        StringConcatenation _builder = new StringConcatenation();
        _xifexpression = _builder.toString();
      } else {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("at ");
        _builder_1.append(name);
        _xifexpression = _builder_1.toString();
      }
      final String description = _xifexpression;
      this.addIssue(IssueCodes.getMessageForDI_NOT_INJECTABLE(typeRef.getTypeRefAsString(), description), ann, IssueCodes.DI_NOT_INJECTABLE);
      return false;
    }
    return true;
  }
  
  /**
   * See N4JS specification, Section 11.1.5 "N4JS DI Injectable Values".
   * 
   * Summing up: for a value to injectable it must:
   * <ul>
   * <li>either conform to a user-defined class or interface (a non-parameterized one, that is),</li>
   * <li>or conform to Provider-of-T where T is injectable itself</li>
   * </ul>
   */
  public static boolean isInjectableType(final TypeArgument typeArg) {
    if ((!(typeArg instanceof ParameterizedTypeRef))) {
      return false;
    }
    final ParameterizedTypeRef typeRef = ((ParameterizedTypeRef) typeArg);
    final Type declType = typeRef.getDeclaredType();
    if ((declType == null)) {
      return false;
    }
    if ((declType instanceof PrimitiveType)) {
      return false;
    }
    if ((((declType instanceof BuiltInType) || (declType instanceof TObjectPrototype)) || (declType instanceof VirtualBaseType))) {
      return false;
    }
    if ((!(declType instanceof TN4Classifier))) {
      return false;
    }
    if (((declType instanceof TN4Classifier) && (!N4JSDependencyInjectionValidator.isProviderType(declType)))) {
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(declType.getTypeVars());
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        return false;
      }
    }
    boolean _isProviderType = N4JSDependencyInjectionValidator.isProviderType(declType);
    if (_isProviderType) {
      Iterable<TypeRef> targs = Iterables.<TypeRef>filter(typeRef.getTypeArgs(), TypeRef.class);
      int _size = IterableExtensions.size(targs);
      boolean _tripleEquals = (_size == 1);
      if (_tripleEquals) {
        boolean _isInjectableType = N4JSDependencyInjectionValidator.isInjectableType(IterableExtensions.<TypeRef>head(targs));
        if (_isInjectableType) {
          return true;
        }
      }
      return IterableExtensions.isEmpty(targs);
    }
    TypingStrategy _typingStrategy = typeRef.getTypingStrategy();
    boolean _tripleEquals_1 = (TypingStrategy.STRUCTURAL_FIELDS == _typingStrategy);
    if (_tripleEquals_1) {
      return false;
    }
    if ((declType instanceof TypeVariable)) {
      return false;
    }
    return true;
  }
  
  /**
   * Is the argument (a subtype of) the built-in N4Provider interface?
   */
  private static boolean isProviderType(final Type type) {
    boolean _xifexpression = false;
    if ((type instanceof TN4Classifier)) {
      boolean _xblockexpression = false;
      {
        final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(type);
        TInterface _n4ProviderType = RuleEnvironmentExtensions.n4ProviderType(G);
        boolean _tripleEquals = (_n4ProviderType == type);
        if (_tripleEquals) {
          return true;
        }
        final Function1<TInterface, Boolean> _function = (TInterface iface) -> {
          TInterface _n4ProviderType_1 = RuleEnvironmentExtensions.n4ProviderType(G);
          return Boolean.valueOf((iface == _n4ProviderType_1));
        };
        _xblockexpression = IterableExtensions.<TInterface>exists(SuperInterfacesIterable.of(((TClassifier)type)), _function);
      }
      _xifexpression = _xblockexpression;
    } else {
      _xifexpression = false;
    }
    return _xifexpression;
  }
  
  /**
   * The given annotation marks an (exported) class declaration, return it of so, otherwise null.
   */
  private static N4ClassDeclaration getAnnotatedClass(final Annotation ann) {
    EObject elem = ann.getAnnotatedElement();
    if ((elem instanceof ExportDeclaration)) {
      elem = ((ExportDeclaration)elem).getExportedElement();
    }
    if ((elem instanceof N4ClassDeclaration)) {
      return ((N4ClassDeclaration)elem);
    }
    return null;
  }
  
  /**
   * The given annotation marks a method declaration, return it of so, otherwise null.
   */
  private static N4MethodDeclaration getAnnotatedMethod(final Annotation ann) {
    EObject elem = ann.getAnnotatedElement();
    if ((elem instanceof N4MethodDeclaration)) {
      return ((N4MethodDeclaration)elem);
    }
    return null;
  }
  
  /**
   * If the annotation argument at the given index is a TypeRef, then returns
   * that type reference; otherwise <code>null</code>.
   */
  private static TypeRef getArgAsTypeRef(final Annotation ann, final int index) {
    AnnotationArgument _xifexpression = null;
    int _size = ann.getArgs().size();
    boolean _lessThan = (index < _size);
    if (_lessThan) {
      _xifexpression = ann.getArgs().get(index);
    } else {
      _xifexpression = null;
    }
    final AnnotationArgument arg = _xifexpression;
    if ((arg instanceof TypeRefAnnotationArgument)) {
      return ((TypeRefAnnotationArgument)arg).getTypeRef();
    }
    return null;
  }
  
  /**
   * If the annotation argument at the given index is a TypeRef, then returns
   * that type reference; otherwise <code>null</code>.
   */
  private static TypeRef getArgAsTypeRef(final TAnnotation ann, final int index) {
    TAnnotationArgument _xifexpression = null;
    int _size = ann.getArgs().size();
    boolean _lessThan = (index < _size);
    if (_lessThan) {
      _xifexpression = ann.getArgs().get(index);
    } else {
      _xifexpression = null;
    }
    final TAnnotationArgument arg = _xifexpression;
    if ((arg instanceof TAnnotationTypeRefArgument)) {
      return ((TAnnotationTypeRefArgument)arg).getTypeRef();
    }
    return null;
  }
  
  /**
   * Does the given type inherit or declare some members annotated with (at)Inject?
   */
  public static boolean requiresInjection(final TClass type) {
    final Function1<TClassifier, Boolean> _function = (TClassifier t) -> {
      final Function1<TMember, Boolean> _function_1 = (TMember m) -> {
        return Boolean.valueOf(AnnotationDefinition.INJECT.hasAnnotation(m));
      };
      return Boolean.valueOf(IterableExtensions.<TMember>exists(t.getOwnedMembers(), _function_1));
    };
    return IterableExtensions.<TClassifier>exists(AllSuperTypesCollector.collect(type), _function);
  }
  
  /**
   * Is the given type (or some super-type) marked (at)Injected?
   */
  public static boolean isMarkedInjected(final TClass type) {
    final Function1<TClassifier, Boolean> _function = (TClassifier t) -> {
      return Boolean.valueOf(AnnotationDefinition.INJECTED.hasAnnotation(t));
    };
    return IterableExtensions.<TClassifier>exists(AllSuperTypesCollector.collect(type), _function);
  }
  
  /**
   * Injected annotates a class (abstract or not) defined in an API project.
   */
  private void internalCheckAnnotationInjected(final Annotation ann) {
    final N4ClassDeclaration classDecl = N4JSDependencyInjectionValidator.getAnnotatedClass(ann);
    boolean _isInjectedApplicable = N4JSDependencyInjectionValidator.isInjectedApplicable(classDecl);
    boolean _not = (!_isInjectedApplicable);
    if (_not) {
      this.addIssue(IssueCodes.getMessageForDI_ANN_INJECTED_NOT_APPLICABLE(), ann, N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.DI_ANN_INJECTED_NOT_APPLICABLE);
      return;
    }
  }
  
  /**
   * Is it allowed to mark the given class declaration (at)Injected?
   */
  private static boolean isInjectedApplicable(final N4ClassDeclaration classDecl) {
    return ((null != classDecl) && (null != classDecl.getDefinedTypeAsClass()));
  }
}
