/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.collect.Iterables;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.generator.GeneratorOption;
import org.eclipse.n4js.n4JS.AwaitExpression;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.ReturnStatement;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TransformationDependency;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Transform async functions to an ES2015 equivalent (still requires generator functions and 'yield').
 * <p>
 * For example, changes the following code
 * <pre>
 * async function funDef(p1, p2) {
 *     await foo();
 * }
 * </pre>
 * to
 * <pre>
 * function funDef(p1, p2) {
 *     return $spawn(function *() {
 *         yield foo();
 *     }.apply(this, arguments));
 * };
 * </pre>
 */
@TransformationDependency.Optional(GeneratorOption.AsyncAwait)
@SuppressWarnings("all")
public class AsyncAwaitTransformation extends Transformation {
  @Override
  public void assertPreConditions() {
  }
  
  @Override
  public void assertPostConditions() {
    final Function1<FunctionOrFieldAccessor, Boolean> _function = (FunctionOrFieldAccessor it) -> {
      return Boolean.valueOf(it.isAsync());
    };
    this.assertFalse("there should not be any async functions left in the intermediate model", 
      IterableExtensions.<FunctionOrFieldAccessor>exists(this.<FunctionOrFieldAccessor>collectNodes(this.getState().im, FunctionOrFieldAccessor.class, true), _function));
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final Function1<FunctionDefinition, Boolean> _function = (FunctionDefinition it) -> {
      return Boolean.valueOf(it.isAsync());
    };
    final Consumer<FunctionDefinition> _function_1 = (FunctionDefinition it) -> {
      this.transformAsyncFunction(it);
    };
    IterableExtensions.<FunctionDefinition>filter(this.<FunctionDefinition>collectNodes(this.getState().im, FunctionDefinition.class, true), _function).forEach(_function_1);
  }
  
  private void transformAsyncFunction(final FunctionDefinition funDef) {
    boolean _isAsync = funDef.isAsync();
    boolean _not = (!_isAsync);
    if (_not) {
      throw new IllegalArgumentException("given function definition must be asynchronous");
    }
    funDef.setDeclaredAsync(false);
    final Block body = funDef.getBody();
    final List<ReturnStatement> returnStmnts = IteratorExtensions.<ReturnStatement>toList(body.getAllReturnStatements());
    final Consumer<ReturnStatement> _function = (ReturnStatement returnStmnt) -> {
      final Expression expr = returnStmnt.getExpression();
      if ((expr instanceof AwaitExpression)) {
        final Expression innerExpr = ((AwaitExpression)expr).getExpression();
        ((AwaitExpression)expr).setExpression(null);
        this.replace(expr, innerExpr);
      }
    };
    returnStmnts.forEach(_function);
    FunctionExpression __FunExpr = TranspilerBuilderBlocks._FunExpr(false);
    final Procedure1<FunctionExpression> _function_1 = (FunctionExpression innerFun) -> {
      innerFun.setGenerator(true);
      EList<Statement> _statements = innerFun.getBody().getStatements();
      EList<Statement> _statements_1 = body.getStatements();
      Iterables.<Statement>addAll(_statements, _statements_1);
    };
    FunctionExpression _doubleArrow = ObjectExtensions.<FunctionExpression>operator_doubleArrow(__FunExpr, _function_1);
    final ParameterizedCallExpression invoke$spawn = TranspilerBuilderBlocks._CallExpr(
      TranspilerBuilderBlocks._IdentRef(this.steFor_$spawn()), 
      TranspilerBuilderBlocks._CallExpr(
        TranspilerBuilderBlocks._PropertyAccessExpr(_doubleArrow, 
          this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.functionType(this.getState().G), "apply", false, false, true)), 
        TranspilerBuilderBlocks._ThisLiteral(), 
        TranspilerBuilderBlocks._IdentRef(this.steFor_arguments())));
    boolean _isEmpty = body.getStatements().isEmpty();
    boolean _not_1 = (!_isEmpty);
    if (_not_1) {
      throw new IllegalStateException();
    }
    EList<Statement> _statements = body.getStatements();
    ReturnStatement __ReturnStmnt = TranspilerBuilderBlocks._ReturnStmnt(invoke$spawn);
    _statements.add(__ReturnStmnt);
    final Consumer<AwaitExpression> _function_2 = (AwaitExpression it) -> {
      this.replace(it, TranspilerBuilderBlocks._YieldExpr(it.getExpression()));
    };
    this.<AwaitExpression>collectNodes(body, AwaitExpression.class, true).forEach(_function_2);
  }
}
