/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.base.Optional;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.ModuleSpecifierAdjustment;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.n4JS.AdditiveOperator;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.BinaryLogicalOperator;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.CommaExpression;
import org.eclipse.n4js.n4JS.EqualityOperator;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.ExportedVariableStatement;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ExpressionStatement;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.ParenExpression;
import org.eclipse.n4js.n4JS.PostfixExpression;
import org.eclipse.n4js.n4JS.PostfixOperator;
import org.eclipse.n4js.n4JS.PrimaryExpression;
import org.eclipse.n4js.n4JS.ReturnStatement;
import org.eclipse.n4js.n4JS.ScriptElement;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.ThrowStatement;
import org.eclipse.n4js.n4JS.UnaryExpression;
import org.eclipse.n4js.n4JS.UnaryOperator;
import org.eclipse.n4js.n4JS.VariableBinding;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableDeclarationOrBinding;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.n4idl.transpiler.utils.N4IDLTranspilerUtils;
import org.eclipse.n4js.projectDescription.ModuleLoader;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TransformationDependency;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.es.assistants.DestructuringAssistant;
import org.eclipse.n4js.transpiler.es.transform.DestructuringTransformation;
import org.eclipse.n4js.transpiler.es.transform.internal.ImportAssignment;
import org.eclipse.n4js.transpiler.es.transform.internal.ImportEntry;
import org.eclipse.n4js.transpiler.es.transform.internal.NamedImportAssignment;
import org.eclipse.n4js.transpiler.es.transform.internal.NamespaceImportAssignment;
import org.eclipse.n4js.transpiler.im.IdentifierRef_IM;
import org.eclipse.n4js.transpiler.im.ParameterizedPropertyAccessExpression_IM;
import org.eclipse.n4js.transpiler.im.ReferencingElementExpression_IM;
import org.eclipse.n4js.transpiler.im.Script_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryInternal;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.versions.VersionableUtils;
import org.eclipse.n4js.utils.ResourceNameComputer;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Module/Script wrapping transformation.
 */
@TransformationDependency.ExcludesAfter(DestructuringTransformation.class)
@SuppressWarnings("all")
public class ModuleWrappingTransformation extends Transformation {
  @Inject
  private IN4JSCore n4jsCore;
  
  @Inject
  private ResourceNameComputer resourceNameComputer;
  
  @Inject
  private DestructuringAssistant destructuringAssistant;
  
  private final Set<SymbolTableEntry> exportedSTEs = CollectionLiterals.<SymbolTableEntry>newLinkedHashSet();
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void assertPreConditions() {
    final Function1<ImportDeclaration, Boolean> _function = (ImportDeclaration it) -> {
      TModule _importedModule = this.getState().info.getImportedModule(it);
      return Boolean.valueOf((_importedModule != null));
    };
    this.assertTrue("every import declaration should have an imported module", 
      IteratorExtensions.<ImportDeclaration>forall(Iterators.<ImportDeclaration>filter(this.getState().im.eAllContents(), ImportDeclaration.class), _function));
  }
  
  @Override
  public void assertPostConditions() {
    int _size = this.getState().im.getScriptElements().size();
    boolean _equals = (_size == 1);
    this.assertTrue("wrapped module has a single statement.", _equals);
  }
  
  @Override
  public void transform() {
    final Script_IM script_im = this.getState().im;
    final List<ScriptElement> content_im = script_im.getScriptElements();
    this.replaceExportStatementsAndExctractCalls(content_im);
    final LinkedHashMap<String, ImportEntry> importSetterMap = this.processImports(content_im);
    final List<Statement> activeStatements = CollectionLiterals.<Statement>newArrayList();
    ParameterizedCallExpression __CallExpr = TranspilerBuilderBlocks._CallExpr();
    final Procedure1<ParameterizedCallExpression> _function = (ParameterizedCallExpression it) -> {
      ParameterizedPropertyAccessExpression_IM __PropertyAccessExpr = TranspilerBuilderBlocks._PropertyAccessExpr();
      final Procedure1<ParameterizedPropertyAccessExpression_IM> _function_1 = (ParameterizedPropertyAccessExpression_IM it_1) -> {
        it_1.setTarget(TranspilerBuilderBlocks._IdentRef(this.steFor_System()));
        it_1.setProperty_IM(this.steFor_register());
      };
      ParameterizedPropertyAccessExpression_IM _doubleArrow = ObjectExtensions.<ParameterizedPropertyAccessExpression_IM>operator_doubleArrow(__PropertyAccessExpr, _function_1);
      it.setTarget(_doubleArrow);
      EList<Argument> _arguments = it.getArguments();
      ArrayLiteral __ArrLit = TranspilerBuilderBlocks._ArrLit();
      final Procedure1<ArrayLiteral> _function_2 = (ArrayLiteral it_1) -> {
        EList<ArrayElement> _elements = it_1.getElements();
        final Function1<ImportEntry, ArrayElement> _function_3 = (ImportEntry itx) -> {
          ArrayElement __ArrayElement = TranspilerBuilderBlocks._ArrayElement(TranspilerBuilderBlocks._StringLiteral(itx.getActualModuleSpecifier()));
          final Procedure1<ArrayElement> _function_4 = (ArrayElement it_2) -> {
            this.getState().tracer.copyTrace(itx.getToBeReplacedImportDeclaration(), it_2);
          };
          return ObjectExtensions.<ArrayElement>operator_doubleArrow(__ArrayElement, _function_4);
        };
        Iterable<ArrayElement> _map = IterableExtensions.<ImportEntry, ArrayElement>map(importSetterMap.values(), _function_3);
        Iterables.<ArrayElement>addAll(_elements, _map);
      };
      ArrayLiteral _doubleArrow_1 = ObjectExtensions.<ArrayLiteral>operator_doubleArrow(__ArrLit, _function_2);
      Argument __Argument = TranspilerBuilderBlocks._Argument(_doubleArrow_1);
      _arguments.add(__Argument);
      EList<Argument> _arguments_1 = it.getArguments();
      FunctionExpression __FunExpr = TranspilerBuilderBlocks._FunExpr(false);
      final Procedure1<FunctionExpression> _function_3 = (FunctionExpression it_1) -> {
        EList<FormalParameter> _fpars = it_1.getFpars();
        FormalParameter __Fpar = TranspilerBuilderBlocks._Fpar();
        final Procedure1<FormalParameter> _function_4 = (FormalParameter it_2) -> {
          it_2.setName(this.steFor_$n4Export().getName());
        };
        FormalParameter _doubleArrow_2 = ObjectExtensions.<FormalParameter>operator_doubleArrow(__Fpar, _function_4);
        _fpars.add(_doubleArrow_2);
        Block __Block = TranspilerBuilderBlocks._Block();
        final Procedure1<Block> _function_5 = (Block it_2) -> {
          final Pair<VariableStatement, List<ExpressionStatement>> hoist = this.hoistedVariablesAndInits(content_im);
          if ((hoist != null)) {
            EList<Statement> _statements = it_2.getStatements();
            VariableStatement _key = hoist.getKey();
            _statements.add(_key);
            EList<Statement> _statements_1 = it_2.getStatements();
            List<ExpressionStatement> _value = hoist.getValue();
            Iterables.<Statement>addAll(_statements_1, _value);
            List<ExpressionStatement> _value_1 = hoist.getValue();
            Iterables.<Statement>addAll(activeStatements, _value_1);
          }
          EList<Statement> _statements_2 = it_2.getStatements();
          final Function1<ImportEntry, FunctionExpression> _function_6 = (ImportEntry it_3) -> {
            return this.importFE(it_3);
          };
          final Function1<FunctionExpression, ArrayElement> _function_7 = (FunctionExpression it_3) -> {
            return TranspilerBuilderBlocks._ArrayElement(it_3);
          };
          FunctionExpression __FunExpr_1 = TranspilerBuilderBlocks._FunExpr(false);
          final Procedure1<FunctionExpression> _function_8 = (FunctionExpression it_3) -> {
            final List<Statement> intoExecute = IterableExtensions.<Statement>toList(Iterables.<Statement>filter(content_im, Statement.class));
            Iterables.<Statement>addAll(activeStatements, intoExecute);
            EList<Statement> _statements_3 = it_3.getBody().getStatements();
            Iterables.<Statement>addAll(_statements_3, intoExecute);
          };
          FunctionExpression _doubleArrow_3 = ObjectExtensions.<FunctionExpression>operator_doubleArrow(__FunExpr_1, _function_8);
          ReturnStatement __ReturnStmnt = TranspilerBuilderBlocks._ReturnStmnt(
            TranspilerBuilderBlocks._ObjLit(
              TranspilerBuilderBlocks._PropertyNameValuePair("setters", 
                TranspilerBuilderBlocks._ArrLit(
                  ((ArrayElement[])Conversions.unwrapArray(IterableExtensions.<FunctionExpression, ArrayElement>map(IterableExtensions.<ImportEntry, FunctionExpression>map(importSetterMap.values(), _function_6), _function_7), ArrayElement.class)))), 
              TranspilerBuilderBlocks._PropertyNameValuePair("execute", _doubleArrow_3)));
          _statements_2.add(__ReturnStmnt);
        };
        Block _doubleArrow_3 = ObjectExtensions.<Block>operator_doubleArrow(__Block, _function_5);
        it_1.setBody(_doubleArrow_3);
      };
      FunctionExpression _doubleArrow_2 = ObjectExtensions.<FunctionExpression>operator_doubleArrow(__FunExpr, _function_3);
      Argument __Argument_1 = TranspilerBuilderBlocks._Argument(_doubleArrow_2);
      _arguments_1.add(__Argument_1);
    };
    final ParameterizedCallExpression call_System_dot_register_Expr = ObjectExtensions.<ParameterizedCallExpression>operator_doubleArrow(__CallExpr, _function);
    final ExpressionStatement exprStatementSystem = TranspilerBuilderBlocks._ExprStmnt(call_System_dot_register_Expr);
    final Statement exprStatement = this.doWrapInCJSpatch(exprStatementSystem);
    script_im.getScriptElements().clear();
    EList<ScriptElement> _scriptElements = script_im.getScriptElements();
    _scriptElements.add(exprStatement);
    this.transFormExportExpressions(activeStatements);
  }
  
  /**
   * FunctionExpression for import used inside of the setters-array
   */
  private FunctionExpression importFE(final ImportEntry entry) {
    FunctionExpression _xblockexpression = null;
    {
      final String fparName = "$exports";
      final SymbolTableEntryInternal steFpar = this.getSymbolTableEntryInternal(fparName, true);
      FunctionExpression __FunExpr = TranspilerBuilderBlocks._FunExpr(false);
      final Procedure1<FunctionExpression> _function = (FunctionExpression it) -> {
        EList<FormalParameter> _fpars = it.getFpars();
        FormalParameter __Fpar = TranspilerBuilderBlocks._Fpar();
        final Procedure1<FormalParameter> _function_1 = (FormalParameter it_1) -> {
          it_1.setName(fparName);
        };
        FormalParameter _doubleArrow = ObjectExtensions.<FormalParameter>operator_doubleArrow(__Fpar, _function_1);
        _fpars.add(_doubleArrow);
        Block __Block = TranspilerBuilderBlocks._Block();
        final Procedure1<Block> _function_2 = (Block it_1) -> {
          EList<Statement> _statements = it_1.getStatements();
          String _actualModuleSpecifier = entry.getActualModuleSpecifier();
          String _plus = ("// " + _actualModuleSpecifier);
          ExpressionStatement __ExprStmnt = TranspilerBuilderBlocks._ExprStmnt(TranspilerBuilderBlocks._Snippet(_plus));
          _statements.add(__ExprStmnt);
          for (final Iterator<ImportAssignment> iter = entry.getVariableSTE_actualName().iterator(); iter.hasNext();) {
            {
              final ImportAssignment current = iter.next();
              final IdentifierRef_IM refToFPar = TranspilerBuilderBlocks._IdentRef(steFpar);
              ReferencingElementExpression_IM _xifexpression = null;
              if ((current instanceof NamespaceImportAssignment)) {
                _xifexpression = refToFPar;
              } else {
                ParameterizedPropertyAccessExpression_IM _xifexpression_1 = null;
                if ((current instanceof NamedImportAssignment)) {
                  ParameterizedPropertyAccessExpression_IM __PropertyAccessExpr = TranspilerBuilderBlocks._PropertyAccessExpr();
                  final Procedure1<ParameterizedPropertyAccessExpression_IM> _function_3 = (ParameterizedPropertyAccessExpression_IM it_2) -> {
                    it_2.setProperty_IM(this.getEntryForNamedImportedElement(((NamedImportAssignment)current).getSte()));
                    it_2.setTarget(refToFPar);
                  };
                  _xifexpression_1 = ObjectExtensions.<ParameterizedPropertyAccessExpression_IM>operator_doubleArrow(__PropertyAccessExpr, _function_3);
                } else {
                  String _simpleName = current.getClass().getSimpleName();
                  String _plus_1 = ("unsupported subclass of ImportAssignment: " + _simpleName);
                  throw new IllegalStateException(_plus_1);
                }
                _xifexpression = _xifexpression_1;
              }
              final Expression rhs = _xifexpression;
              EList<Statement> _statements_1 = it_1.getStatements();
              ExpressionStatement __ExprStmnt_1 = TranspilerBuilderBlocks._ExprStmnt(TranspilerBuilderBlocks._AssignmentExpr(TranspilerBuilderBlocks._IdentRef(current.getSte()), rhs));
              final Procedure1<ExpressionStatement> _function_4 = (ExpressionStatement it_2) -> {
                this.getState().tracer.copyTrace(current.getToBeReplacedImportSpecifier(), it_2);
              };
              ExpressionStatement _doubleArrow_1 = ObjectExtensions.<ExpressionStatement>operator_doubleArrow(__ExprStmnt_1, _function_4);
              _statements_1.add(_doubleArrow_1);
            }
          }
        };
        Block _doubleArrow_1 = ObjectExtensions.<Block>operator_doubleArrow(__Block, _function_2);
        it.setBody(_doubleArrow_1);
        this.getState().tracer.copyTrace(entry.getToBeReplacedImportDeclaration(), it);
      };
      _xblockexpression = ObjectExtensions.<FunctionExpression>operator_doubleArrow(__FunExpr, _function);
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the STE to use to import a named element from another module.
   */
  private SymbolTableEntry getEntryForNamedImportedElement(final SymbolTableEntryOriginal importedElementEntry) {
    final IdentifiableElement originalTarget = importedElementEntry.getOriginalTarget();
    boolean _isTVersionable = VersionableUtils.isTVersionable(originalTarget);
    if (_isTVersionable) {
      return this.getSymbolTableEntryInternal(N4IDLTranspilerUtils.getVersionedInternalName(originalTarget), true);
    }
    return this.getSymbolTableEntryInternal(importedElementEntry.getExportedName(), true);
  }
  
  /**
   * Goes over imports from content_im and collect imported things in a meta-data-structure
   * in map[module_name -> ImportEntry]
   * 
   * This method builds up the data structure to handle imports but doesn't change the intermediate model.
   * 
   * @param contents_im script-content of intermediate model
   * @return map of completeModuleSpecifier to ImportEntries.
   */
  private LinkedHashMap<String, ImportEntry> processImports(final List<ScriptElement> contents_im) {
    final LinkedHashMap<String, ImportEntry> map = CollectionLiterals.<String, ImportEntry>newLinkedHashMap();
    for (final Iterator<ScriptElement> iter = contents_im.iterator(); iter.hasNext();) {
      {
        final ScriptElement elementIM = iter.next();
        if ((elementIM instanceof ImportDeclaration)) {
          final TModule module = this.getState().info.getImportedModule(((ImportDeclaration)elementIM));
          final String actualModuleSpecifier = this.computeActualModuleSpecifier(module);
          ImportEntry moduleEntry = map.get(actualModuleSpecifier);
          if ((moduleEntry == null)) {
            ArrayList<ImportAssignment> _newArrayList = CollectionLiterals.<ImportAssignment>newArrayList();
            ImportEntry _importEntry = new ImportEntry(actualModuleSpecifier, _newArrayList, ((ImportDeclaration)elementIM));
            moduleEntry = _importEntry;
            map.put(actualModuleSpecifier, moduleEntry);
          }
          final ImportEntry finalModuleEntry = moduleEntry;
          final Consumer<ImportSpecifier> _function = (ImportSpecifier it) -> {
            boolean _matched = false;
            if (it instanceof NamespaceImportSpecifier) {
              _matched=true;
              final SymbolTableEntryOriginal nisSTE = this.findSymbolTableEntryForNamespaceImport(((NamespaceImportSpecifier)it));
              List<ImportAssignment> _variableSTE_actualName = finalModuleEntry.getVariableSTE_actualName();
              NamespaceImportAssignment _namespaceImportAssignment = new NamespaceImportAssignment(nisSTE, ((NamespaceImportSpecifier)it));
              _variableSTE_actualName.add(_namespaceImportAssignment);
            }
            if (!_matched) {
              if (it instanceof NamedImportSpecifier) {
                _matched=true;
                final SymbolTableEntryOriginal ste = this.findSymbolTableEntryForNamedImport(((NamedImportSpecifier)it));
                if ((ste != null)) {
                  List<ImportAssignment> _variableSTE_actualName = finalModuleEntry.getVariableSTE_actualName();
                  String _alias = ((NamedImportSpecifier)it).getAlias();
                  NamedImportAssignment _namedImportAssignment = new NamedImportAssignment(ste, _alias, ((NamedImportSpecifier)it));
                  _variableSTE_actualName.add(_namedImportAssignment);
                }
              }
            }
          };
          ((ImportDeclaration)elementIM).getImportSpecifiers().forEach(_function);
        }
      }
    }
    return map;
  }
  
  private String computeActualModuleSpecifier(final TModule module) {
    final String completeModuleSpecifier = this.resourceNameComputer.getCompleteModuleSpecifier(module);
    final ModuleSpecifierAdjustment moduleSpecifierAdjustment = this.getModuleSpecifierAdjustment(module);
    if (((moduleSpecifierAdjustment != null) && moduleSpecifierAdjustment.usePlainModuleSpecifier)) {
      String _moduleSpecifier = module.getModuleSpecifier();
      return ((moduleSpecifierAdjustment.prefix + "/") + _moduleSpecifier);
    }
    String specifier = completeModuleSpecifier;
    IN4JSProject depProject = this.n4jsCore.findProject(module.eResource().getURI()).orNull();
    if ((((depProject != null) && (depProject.getProjectType() == ProjectType.DEFINITION)) && (depProject.getDefinesPackageName() != null))) {
      depProject = this.n4jsCore.findAllProjectMappings().get(depProject.getDefinesPackageName());
    }
    if ((depProject != null)) {
      final String projectRelativeSegment = depProject.getOutputPath();
      final Path depLocation = depProject.getLocationPath();
      if ((depLocation != null)) {
        final String depLocationString = depLocation.toString();
        final String depProjecOutputPath = depProject.getLocationPath().resolve(projectRelativeSegment).normalize().toString();
        int _length = depLocationString.length();
        int _length_1 = depProject.getProjectName().length();
        int _minus = (_length - _length_1);
        final String depRelativeSpecifier = depProjecOutputPath.substring(_minus);
        specifier = ((depRelativeSpecifier + "/") + completeModuleSpecifier);
      }
    }
    if ((moduleSpecifierAdjustment != null)) {
      return ((moduleSpecifierAdjustment.prefix + "/") + specifier);
    }
    return specifier;
  }
  
  /**
   * Removes "export" :
   * Iterates over scriptContent_im and replaces each export-statement with it's containing statement.
   * @param scriptContent_im current intermediate top-level elements
   * @return exportMethodCalls - container collecting the newly created "$n4Export()" calls - detached from intermediate model.
   */
  private void replaceExportStatementsAndExctractCalls(final List<ScriptElement> scriptContent_im) {
    for (int i = 0; (i < scriptContent_im.size()); i++) {
      {
        final ScriptElement orig = scriptContent_im.get(i);
        if ((orig instanceof ExportDeclaration)) {
          ExportableElement _exportedElement = ((ExportDeclaration)orig).getExportedElement();
          final ExportedVariableStatement exportedVarStmt = ((ExportedVariableStatement) _exportedElement);
          if ((exportedVarStmt != null)) {
            final VariableStatement bareVarStmt = this.removeExport(exportedVarStmt);
            final Function1<VariableDeclaration, SymbolTableEntry> _function = (VariableDeclaration it) -> {
              return this.findSymbolTableEntryForElement(it, true);
            };
            final Consumer<SymbolTableEntry> _function_1 = (SymbolTableEntry it) -> {
              this.exportedSTEs.add(it);
            };
            ListExtensions.<VariableDeclaration, SymbolTableEntry>map(bareVarStmt.getVarDecl(), _function).forEach(_function_1);
          }
        }
      }
    }
  }
  
  /**
   * Precondition: The scriptContent only contains VariableDeclarations and ImportStatements.
   * 
   * All hoisted variables will be extracted and returned in a new VariableStatement
   * 
   * Removed things:
   * - Import Declaration
   * - Variable declarations without initializers (they are hoisted)
   * 
   * Moved things:
   * - initializer expressions will be extracted from declarations.
   * 
   * @return the statement with all hoisted variable AND the list of initializers to be hoisted
   */
  private Pair<VariableStatement, List<ExpressionStatement>> hoistedVariablesAndInits(final List<ScriptElement> scriptContent) {
    final Map<VariableStatement, List<ExpressionStatement>> mapVarStatement2replacer = CollectionLiterals.<VariableStatement, List<ExpressionStatement>>newLinkedHashMap();
    final Function1<ScriptElement, Iterable<VariableDeclaration>> _function = (ScriptElement scriptElement) -> {
      Iterable<VariableDeclaration> _switchResult = null;
      boolean _matched = false;
      if (scriptElement instanceof VariableStatement) {
        _matched=true;
        Iterable<VariableDeclaration> _xblockexpression = null;
        {
          final Iterable<Pair<List<VariableDeclaration>, ExpressionStatement>> pairsOfDeclarationAndInitialsers = this.toHoistDeclarations(((VariableStatement)scriptElement).getVarDeclsOrBindings());
          final Function1<Pair<List<VariableDeclaration>, ExpressionStatement>, ExpressionStatement> _function_1 = (Pair<List<VariableDeclaration>, ExpressionStatement> it) -> {
            return it.getValue();
          };
          final List<ExpressionStatement> listOfInitExpressionStmts = IterableExtensions.<ExpressionStatement>toList(IterableExtensions.<ExpressionStatement>filterNull(IterableExtensions.<Pair<List<VariableDeclaration>, ExpressionStatement>, ExpressionStatement>map(pairsOfDeclarationAndInitialsers, _function_1)));
          boolean _isEmpty = listOfInitExpressionStmts.isEmpty();
          boolean _not = (!_isEmpty);
          if (_not) {
            mapVarStatement2replacer.put(((VariableStatement)scriptElement), listOfInitExpressionStmts);
          }
          final Function1<Pair<List<VariableDeclaration>, ExpressionStatement>, List<VariableDeclaration>> _function_2 = (Pair<List<VariableDeclaration>, ExpressionStatement> it) -> {
            return it.getKey();
          };
          _xblockexpression = Iterables.<VariableDeclaration>concat(IterableExtensions.<Pair<List<VariableDeclaration>, ExpressionStatement>, List<VariableDeclaration>>map(pairsOfDeclarationAndInitialsers, _function_2));
        }
        _switchResult = _xblockexpression;
      }
      if (!_matched) {
        if (scriptElement instanceof ImportDeclaration) {
          _matched=true;
          _switchResult = this.toHoistDeclaration(((ImportDeclaration)scriptElement));
        }
      }
      if (!_matched) {
        _switchResult = null;
      }
      final Iterable<VariableDeclaration> ret = _switchResult;
      return ret;
    };
    final Iterable<VariableDeclaration> allHoisted = IterableExtensions.<VariableDeclaration>toList(Iterables.<VariableDeclaration>concat(IterableExtensions.<Iterable<VariableDeclaration>>filterNull(ListExtensions.<ScriptElement, Iterable<VariableDeclaration>>map(scriptContent, _function))));
    VariableStatement __VariableStatement = TranspilerBuilderBlocks._VariableStatement();
    final Procedure1<VariableStatement> _function_1 = (VariableStatement it) -> {
      EList<VariableDeclarationOrBinding> _varDeclsOrBindings = it.getVarDeclsOrBindings();
      Iterables.<VariableDeclarationOrBinding>addAll(_varDeclsOrBindings, allHoisted);
    };
    final VariableStatement varStmtHoistedVariables = ObjectExtensions.<VariableStatement>operator_doubleArrow(__VariableStatement, _function_1);
    for (int i = (scriptContent.size() - 1); (i >= 0); i--) {
      {
        ScriptElement scriptElement = scriptContent.get(i);
        if ((scriptElement instanceof ImportDeclaration)) {
          this.remove(scriptElement);
        } else {
          if ((scriptElement instanceof VariableStatement)) {
            boolean _containsKey = mapVarStatement2replacer.containsKey(scriptElement);
            boolean _not = (!_containsKey);
            if (_not) {
              this.remove(scriptElement);
            }
          }
        }
      }
    }
    final List<ExpressionStatement> hoistedInitializer = CollectionLiterals.<ExpressionStatement>newArrayList();
    final Consumer<Map.Entry<VariableStatement, List<ExpressionStatement>>> _function_2 = (Map.Entry<VariableStatement, List<ExpressionStatement>> it) -> {
      this.replace(it.getKey(), ((Statement[])Conversions.unwrapArray(it.getValue(), Statement.class)));
      final Function1<ExpressionStatement, Boolean> _function_3 = (ExpressionStatement it_1) -> {
        return Boolean.valueOf(this.getState().info.isInitializerOfHoistedVariable(it_1));
      };
      Iterable<ExpressionStatement> _filter = IterableExtensions.<ExpressionStatement>filter(it.getValue(), _function_3);
      Iterables.<ExpressionStatement>addAll(hoistedInitializer, _filter);
    };
    mapVarStatement2replacer.entrySet().forEach(_function_2);
    boolean _isEmpty = varStmtHoistedVariables.getVarDeclsOrBindings().isEmpty();
    if (_isEmpty) {
      return null;
    }
    return Pair.<VariableStatement, List<ExpressionStatement>>of(varStmtHoistedVariables, hoistedInitializer);
  }
  
  /**
   * Decouples VarDeclarations and their initializer expressions. Returns them as a Pair. Value can be {@code null}
   */
  private Iterable<Pair<List<VariableDeclaration>, ExpressionStatement>> toHoistDeclarations(final List<VariableDeclarationOrBinding> varDeclsOrBindings) {
    final Function1<VariableDeclarationOrBinding, Pair<List<VariableDeclaration>, ExpressionStatement>> _function = (VariableDeclarationOrBinding entry) -> {
      Pair<List<VariableDeclaration>, ExpressionStatement> _switchResult = null;
      boolean _matched = false;
      if (entry instanceof VariableDeclaration) {
        _matched=true;
        _switchResult = this.hoistEntry(((VariableDeclaration)entry));
      }
      if (!_matched) {
        if (entry instanceof VariableBinding) {
          _matched=true;
          _switchResult = this.hoistEntry(((VariableBinding)entry));
        }
      }
      if (!_matched) {
        String _simpleName = VariableDeclarationOrBinding.class.getSimpleName();
        String _plus = ("unknown subclass of " + _simpleName);
        String _plus_1 = (_plus + ": ");
        String _simpleName_1 = entry.getClass().getSimpleName();
        String _plus_2 = (_plus_1 + _simpleName_1);
        throw new IllegalStateException(_plus_2);
      }
      return _switchResult;
    };
    return ListExtensions.<VariableDeclarationOrBinding, Pair<List<VariableDeclaration>, ExpressionStatement>>map(varDeclsOrBindings, _function);
  }
  
  /**
   * Decouple VariableDeclaration and initializer.
   * If an initializer is given it will be wrapped into a new ExpressionStatement.
   */
  private Pair<List<VariableDeclaration>, ExpressionStatement> hoistEntry(final VariableDeclaration vDeclIM) {
    ExpressionStatement _xifexpression = null;
    Expression _expression = vDeclIM.getExpression();
    boolean _tripleNotEquals = (_expression != null);
    if (_tripleNotEquals) {
      ExpressionStatement _xblockexpression = null;
      {
        final Expression initExpr = vDeclIM.getExpression();
        vDeclIM.setExpression(null);
        final SymbolTableEntry ste = this.findSymbolTableEntryForElement(vDeclIM, true);
        final ExpressionStatement stmt = TranspilerBuilderBlocks._ExprStmnt(TranspilerBuilderBlocks._AssignmentExpr(TranspilerBuilderBlocks._IdentRef(ste), initExpr));
        boolean _isToHoist = this.getState().info.isToHoist(vDeclIM);
        if (_isToHoist) {
          this.getState().info.markAsInitializerOfHoistedVariable(stmt);
        }
        this.getState().tracer.copyTrace(vDeclIM, stmt);
        _xblockexpression = stmt;
      }
      _xifexpression = _xblockexpression;
    }
    final ExpressionStatement exprStmt = _xifexpression;
    return Pair.<List<VariableDeclaration>, ExpressionStatement>of(Collections.<VariableDeclaration>unmodifiableList(CollectionLiterals.<VariableDeclaration>newArrayList(vDeclIM)), exprStmt);
  }
  
  private Pair<List<VariableDeclaration>, ExpressionStatement> hoistEntry(final VariableBinding binding) {
    EList<VariableDeclaration> _variableDeclarations = binding.getVariableDeclarations();
    ExpressionStatement _convertDestructBindingToDestructAssignment = this.convertDestructBindingToDestructAssignment(binding);
    return Pair.<List<VariableDeclaration>, ExpressionStatement>of(_variableDeclarations, _convertDestructBindingToDestructAssignment);
  }
  
  /**
   * Creates a list of VariableDeclarations without initialisers. Used in hoisting.
   */
  private Iterable<VariableDeclaration> toHoistDeclaration(final ImportDeclaration importDecl) {
    final Function1<ImportSpecifier, VariableDeclaration> _function = (ImportSpecifier it) -> {
      VariableDeclaration _switchResult = null;
      boolean _matched = false;
      if (it instanceof NamespaceImportSpecifier) {
        _matched=true;
        _switchResult = this.namespaceToHoistDeclaration(((NamespaceImportSpecifier)it));
      }
      if (!_matched) {
        if (it instanceof NamedImportSpecifier) {
          _matched=true;
          _switchResult = this.namedImportsToHoistDeclaration(((NamedImportSpecifier)it));
        }
      }
      return _switchResult;
    };
    return ListExtensions.<ImportSpecifier, VariableDeclaration>map(importDecl.getImportSpecifiers(), _function);
  }
  
  /**
   * Creates a single VariableDeclaration without initialiser.
   */
  private VariableDeclaration namedImportsToHoistDeclaration(final NamedImportSpecifier nis) {
    VariableDeclaration __VariableDeclaration = TranspilerBuilderBlocks._VariableDeclaration(this.findSymbolTableEntryForNamedImport(nis).getName());
    final Procedure1<VariableDeclaration> _function = (VariableDeclaration it) -> {
      this.getState().tracer.copyTrace(nis, it);
    };
    return ObjectExtensions.<VariableDeclaration>operator_doubleArrow(__VariableDeclaration, _function);
  }
  
  /**
   * Creates a single VariableDeclaration without initialiser.
   */
  private VariableDeclaration namespaceToHoistDeclaration(final NamespaceImportSpecifier nsImport) {
    VariableDeclaration __VariableDeclaration = TranspilerBuilderBlocks._VariableDeclaration(nsImport.getAlias());
    final Procedure1<VariableDeclaration> _function = (VariableDeclaration it) -> {
      this.getState().tracer.copyTrace(nsImport, it);
    };
    return ObjectExtensions.<VariableDeclaration>operator_doubleArrow(__VariableDeclaration, _function);
  }
  
  private void transFormExportExpressions(final List<Statement> list) {
    final Iterable<Statement> toProcess = Iterables.<Statement>concat(Collections.<Statement>unmodifiableList(CollectionLiterals.<Statement>newArrayList()), list);
    final Consumer<Statement> _function = (Statement it) -> {
      final Consumer<Expression> _function_1 = (Expression it_1) -> {
        this.inferExportCall(it_1);
      };
      this.<Expression>collectNodes(it, Expression.class, true).forEach(_function_1);
    };
    toProcess.forEach(_function);
  }
  
  private void _inferExportCall(final PostfixExpression expr) {
    final Expression subExpr = expr.getExpression();
    if ((subExpr instanceof IdentifierRef_IM)) {
      final SymbolTableEntry ste = ((IdentifierRef_IM)subExpr).getRewiredTarget();
      boolean _isExported = this.isExported(ste);
      if (_isExported) {
        final EObject container = expr.eContainer();
        boolean _isAppendableStatement = ModuleWrappingTransformation.isAppendableStatement(container);
        if (_isAppendableStatement) {
          this.insertAfter(container, TranspilerBuilderBlocks._ExprStmnt(this.createExportExpression(ste)));
        } else {
          PostfixOperator _op = expr.getOp();
          if (_op != null) {
            switch (_op) {
              case INC:
                this.postfixReplacement(expr, ste, AdditiveOperator.SUB);
                break;
              case DEC:
                this.postfixReplacement(expr, ste, AdditiveOperator.ADD);
                break;
              default:
                break;
            }
          }
        }
      }
    }
  }
  
  private final void postfixReplacement(final PostfixExpression expr, final SymbolTableEntry ste, final AdditiveOperator op) {
    final ParenExpression replaceExp = TranspilerBuilderBlocks._Parenthesis(
      TranspilerBuilderBlocks._CommaExpression(TranspilerBuilderBlocks._N4ExportExpr(ste, 
        TranspilerBuilderBlocks._Parenthesis(
          TranspilerBuilderBlocks._CommaExpression(TranspilerBuilderBlocks._ObjLit(), this.__NSSafe_IdentRef(ste))), this.steFor_$n4Export()), 
        TranspilerBuilderBlocks._AdditiveExpression(this.__NSSafe_IdentRef(ste), op, TranspilerBuilderBlocks._IntLiteral(1))));
    final Procedure1<PostfixExpression> _function = (PostfixExpression pe) -> {
      Expression _expression = replaceExp.getExpression();
      Expression _get = ((CommaExpression) _expression).getExprs().get(0);
      Expression _expression_1 = ((ParameterizedCallExpression) _get).getArguments().get(1).getExpression();
      Expression _expression_2 = ((ParenExpression) _expression_1).getExpression();
      final Procedure1<CommaExpression> _function_1 = (CommaExpression it) -> {
        it.getExprs().set(0, pe);
      };
      ObjectExtensions.<CommaExpression>operator_doubleArrow(
        ((CommaExpression) _expression_2), _function_1);
      return;
    };
    final Procedure1<PostfixExpression> initFunction = _function;
    this.<PostfixExpression>wrapExistingExpression(expr, replaceExp, initFunction);
  }
  
  private void _inferExportCall(final UnaryExpression expr) {
    final Expression subExpr = expr.getExpression();
    if ((subExpr instanceof IdentifierRef_IM)) {
      final SymbolTableEntry ste = ((IdentifierRef_IM)subExpr).getRewiredTarget();
      boolean _isExported = this.isExported(ste);
      if (_isExported) {
        final EObject container = expr.eContainer();
        boolean _isAppendableStatement = ModuleWrappingTransformation.isAppendableStatement(container);
        if (_isAppendableStatement) {
          this.insertAfter(container, TranspilerBuilderBlocks._ExprStmnt(this.createExportExpression(ste)));
        } else {
          UnaryOperator _op = expr.getOp();
          if (_op != null) {
            switch (_op) {
              case INC:
                this.exprReplacement(expr, ste);
                break;
              case DEC:
                this.exprReplacement(expr, ste);
                break;
              default:
                break;
            }
          } else {
          }
        }
      }
    }
  }
  
  private void _inferExportCall(final AssignmentExpression expr) {
    final Expression lhs = expr.getLhs();
    boolean _matched = false;
    if (lhs instanceof IdentifierRef_IM) {
      _matched=true;
      final SymbolTableEntry ste = ((IdentifierRef_IM)lhs).getRewiredTarget();
      boolean _isExported = this.isExported(ste);
      if (_isExported) {
        final EObject container = expr.eContainer();
        boolean _isAppendableStatement = ModuleWrappingTransformation.isAppendableStatement(container);
        if (_isAppendableStatement) {
          this.insertAfter(container, TranspilerBuilderBlocks._ExprStmnt(this.createExportExpression(ste)));
        } else {
          this.exprReplacement(expr, ste);
        }
      }
    }
    if (!_matched) {
    }
  }
  
  /**
   * The EObject is NOT a {@link ReturnStatement} nor {@link ThrowStatement}
   */
  private static boolean isAppendableStatement(final EObject container) {
    return (((container instanceof Statement) && (!(container instanceof ReturnStatement))) && (!(container instanceof ThrowStatement)));
  }
  
  /**
   * Reusing an expression {@code expr1} of IM wrapped into a comma-expression including call to $n4Export:
   * <pre>
   * ....expr1/setting valueof x/.... -->  ....($n4Export("x",expr1),x)....
   * </pre>
   * 
   * Only applicable if {@code expr1} evaluates (while actually setting) to the new value of x.
   * This is true to Assignment- and Unary-Expressions <b>but not</b> for PostfixExprssion
   * 
   * <p>
   * Note: only <pre>++x</pre> and <pre>--x</pre> are setting the value of <pre>x</pre>.
   * other Unary-Expressions are not setting the value. Caller needs to take care of this, as
   * expression operator is not checked in this method.
   * 
   * <p>
   * Generates a ParenExpression and a callback-function
   * 
   * After inserting the ParenExpression into the old place of expr1,
   * executing the callback with parameter expr1 will insert expr1 as a child at the correct place.
   * 
   * @param ste symboltableentry for {@code x}
   */
  private final void exprReplacement(final Expression expr, final SymbolTableEntry ste) {
    final ParenExpression replaceExp = TranspilerBuilderBlocks._Parenthesis(
      TranspilerBuilderBlocks._CommaExpression(
        TranspilerBuilderBlocks._N4ExportExpr(ste, TranspilerBuilderBlocks._ObjLit(), this.steFor_$n4Export()), this.__NSSafe_IdentRef(ste)));
    final Procedure1<Expression> _function = (Expression ae) -> {
      Expression _expression = replaceExp.getExpression();
      Expression _get = ((CommaExpression) _expression).getExprs().get(0);
      final Procedure1<ParameterizedCallExpression> _function_1 = (ParameterizedCallExpression it) -> {
        it.getArguments().set(1, TranspilerBuilderBlocks._Argument(ae));
      };
      ObjectExtensions.<ParameterizedCallExpression>operator_doubleArrow(
        ((ParameterizedCallExpression) _get), _function_1);
      return;
    };
    final Procedure1<Expression> initFunc = _function;
    this.<Expression>wrapExistingExpression(expr, replaceExp, initFunc);
  }
  
  private void _inferExportCall(final Expression expression) {
  }
  
  private boolean isExported(final SymbolTableEntry ste) {
    return this.exportedSTEs.contains(ste);
  }
  
  /**
   * returns adjustments to be used based on the module loader specified for the provided module. May be null.
   */
  private ModuleSpecifierAdjustment getModuleSpecifierAdjustment(final TModule module) {
    Resource _eResource = null;
    if (module!=null) {
      _eResource=module.eResource();
    }
    URI _uRI = null;
    if (_eResource!=null) {
      _uRI=_eResource.getURI();
    }
    final URI resourceURI = _uRI;
    if ((resourceURI == null)) {
      return null;
    }
    final Optional<? extends IN4JSProject> project = this.n4jsCore.findProject(resourceURI);
    boolean _isPresent = project.isPresent();
    boolean _not = (!_isPresent);
    if (_not) {
      return null;
    }
    final ModuleLoader loader = project.get().getModuleLoader();
    if ((loader == null)) {
      return null;
    }
    final ModuleSpecifierAdjustment adjustment = N4JSLanguageConstants.MODULE_LOADER_PREFIXES.get(loader);
    return adjustment;
  }
  
  /**
   * SystemJS-wrapping of external JS-code which is not transpiled by ourselves.
   * Note: the code which will be wrapped usually is a non-stand-alone JS-snippet already referring to the parameters {@code require},
   * {@code exports} and {@code module} which are hereby introduced as formal parameters.
   */
  public static CharSequence wrapPlainJSCode(final CharSequence cs) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("(function(System) {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("System.registerDynamic([], true, function(require, exports, module) {");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append(cs, "\t\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("});");
    _builder.newLine();
    _builder.append("})(typeof module !== \'undefined\' && module.exports ? require(\'n4js-node\').System(require, module) : System);");
    _builder.newLine();
    return _builder;
  }
  
  /**
   * patch the statement with commonJS-support
   */
  private Statement doWrapInCJSpatch(final ExpressionStatement statement) {
    FormalParameter __Fpar = TranspilerBuilderBlocks._Fpar(this.steFor_System().getName());
    final ExpressionStatement ret = TranspilerBuilderBlocks._ExprStmnt(TranspilerBuilderBlocks._CallExpr(
      TranspilerBuilderBlocks._Parenthesis(
        TranspilerBuilderBlocks._FunExpr(false, null, new FormalParameter[] { __Fpar }, 
          TranspilerBuilderBlocks._ExprStmnt(TranspilerBuilderBlocks._StringLiteral("use strict")), statement)), 
      TranspilerBuilderBlocks._ConditionalExpr(
        TranspilerBuilderBlocks._BinaryLogicalExpr(
          TranspilerBuilderBlocks._EqualityExpr(TranspilerBuilderBlocks._UnaryExpr(UnaryOperator.TYPEOF, TranspilerBuilderBlocks._IdentRef(this.steFor_module())), EqualityOperator.NSAME, TranspilerBuilderBlocks._StringLiteralForSTE(this.steFor_undefined())), 
          BinaryLogicalOperator.AND, 
          TranspilerBuilderBlocks._PropertyAccessExpr(this.steFor_module(), this.steFor_exports())), 
        TranspilerBuilderBlocks._CallExpr(TranspilerBuilderBlocks._PropertyAccessExpr(TranspilerBuilderBlocks._CallExpr(TranspilerBuilderBlocks._IdentRef(this.steFor_require()), TranspilerBuilderBlocks._StringLiteral("n4js-node")), this.steFor_System()), TranspilerBuilderBlocks._IdentRef(this.steFor_require()), TranspilerBuilderBlocks._IdentRef(this.steFor_module())), 
        TranspilerBuilderBlocks._IdentRef(this.steFor_System()))));
    return ret;
  }
  
  /**
   * Given a destructuring binding, this method returns a destructuring assignment that performs the equivalent
   * destructuring operation.
   * <p>
   * For example, given an array destructuring binding (i.e. the code between the 'let' and the '=') as in
   * <pre>
   * let [a,b] = [1,2];
   * </pre>
   * this method will return the following assignment expression
   * <pre>
   * [a,b] = [1,2]
   * </pre>
   * and given an object destructuring binding (again, the code between the 'let' and the '=') as in
   * <pre>
   * let {prop1: x, prop2: y} = {prop1: 1, prop2: 2};
   * </pre>
   * this method will return the following assignment expression
   * <pre>
   * ({prop1: x, prop2: y} = {prop1: 1, prop2: 2})
   * </pre>
   * (wrapped in a parenthesis expression).
   */
  private ExpressionStatement convertDestructBindingToDestructAssignment(final VariableBinding binding) {
    final PrimaryExpression patternConverted = this.destructuringAssistant.convertBindingPatternToArrayOrObjectLiteral(binding.getPattern());
    final AssignmentExpression assignmentExpr = TranspilerBuilderBlocks._AssignmentExpr(patternConverted, binding.getExpression());
    Expression _xifexpression = null;
    if ((patternConverted instanceof ObjectLiteral)) {
      _xifexpression = TranspilerBuilderBlocks._Parenthesis(assignmentExpr);
    } else {
      _xifexpression = assignmentExpr;
    }
    final ExpressionStatement assignmentStmnt = TranspilerBuilderBlocks._ExprStmnt(_xifexpression);
    final Function1<VariableDeclaration, Boolean> _function = (VariableDeclaration it) -> {
      return Boolean.valueOf(this.getState().info.isToHoist(it));
    };
    boolean _exists = IterableExtensions.<VariableDeclaration>exists(binding.getVariableDeclarations(), _function);
    if (_exists) {
      final Consumer<VariableDeclaration> _function_1 = (VariableDeclaration it) -> {
        this.getState().info.markAsToHoist(it);
      };
      binding.getVariableDeclarations().forEach(_function_1);
      this.getState().info.markAsInitializerOfHoistedVariable(assignmentStmnt);
    }
    this.getState().tracer.copyTrace(binding, assignmentStmnt);
    return assignmentStmnt;
  }
  
  /**
   * Creates an export expression by the name of the given symbol table entry which exports a simple (identifier) reference
   * to the symbol table entry.
   */
  protected ParameterizedCallExpression createExportExpression(final SymbolTableEntry entry) {
    return TranspilerBuilderBlocks._N4ExportExpr(entry, TranspilerBuilderBlocks._IdentRef(entry), this.steFor_$n4Export());
  }
  
  private void inferExportCall(final Expression expr) {
    if (expr instanceof AssignmentExpression) {
      _inferExportCall((AssignmentExpression)expr);
      return;
    } else if (expr instanceof PostfixExpression) {
      _inferExportCall((PostfixExpression)expr);
      return;
    } else if (expr instanceof UnaryExpression) {
      _inferExportCall((UnaryExpression)expr);
      return;
    } else if (expr != null) {
      _inferExportCall(expr);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(expr).toString());
    }
  }
}
