/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.outline;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.List;
import org.apache.log4j.Logger;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.ExportedVariableStatement;
import org.eclipse.n4js.n4JS.FieldAccessor;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDefinition;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4EnumLiteral;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.ScriptElement;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.ts.types.MemberAccessModifier;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.util.MemberList;
import org.eclipse.n4js.ui.labeling.EObjectWithContext;
import org.eclipse.n4js.ui.labeling.N4JSLabelProvider;
import org.eclipse.n4js.ui.outline.N4JSEObjectNode;
import org.eclipse.n4js.ui.outline.N4JSOutlineModes;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.ui.editor.outline.IOutlineNode;
import org.eclipse.xtext.ui.editor.outline.IOutlineTreeProvider;
import org.eclipse.xtext.ui.editor.outline.impl.BackgroundOutlineTreeProvider;
import org.eclipse.xtext.ui.editor.outline.impl.DocumentRootNode;
import org.eclipse.xtext.ui.editor.outline.impl.EObjectNode;
import org.eclipse.xtext.ui.editor.outline.impl.OutlineMode;
import org.eclipse.xtext.util.CancelIndicator;
import org.eclipse.xtext.util.TextRegion;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Customization of the default outline structure.
 * We filter all null elements, which can only occur in case of syntax errors.
 * <p>
 * This outline runs in the background, e.g. in a non-UI thread, thus improving responsiveness.
 * That functionality requires that no operations are performed that must run in the UI thread.
 * In particular, {@link N4JSLabelProvider#doGetImage} shouldn't perform such operations.
 * A rule of thumb is "prefer {@link ImageDescriptor} over {@code Image}".
 * <p>
 * This outline tree provider supports different modes. There are two use cases:
 * <ul>
 * 	<li>Quick Outline View: In this case, modes are toggled via CMD/CTRL-O. This simply means that
 * 		{@link N4JSOutlineTreeProvider#getNextMode()} is called every time. These modes are defined in
 * 		{@link N4JSOutlineModes}. For that to work, the {@link IOutlineTreeProvider#ModeAware} protocol is used
 *  <li>Outline View: The normal outline view. In this case, toggle buttons are used to enable filtering. This
 * 		is achieved by the {@link N4JSShowInheritedMembersOutlineContribution}. After an object node has been
 * 		created, it is send to this contribution class which does the filtering.
 * </ul>
 * There is no direct way of detecting the current use case. Since the "normal" outline view does not call any
 * method of {@link IOutlineTreeProvider#ModeAware}, we initialize the corresponding field
 * {@link N4JSOutlineTreeProvider#modeAware} on demand. If it is null we can then conclude that we are in "normal"
 * outline view mode, other wise in the quick outline view.
 * 
 * @see http://www.eclipse.org/Xtext/documentation.html#outline
 */
@SuppressWarnings("all")
public class N4JSOutlineTreeProvider extends BackgroundOutlineTreeProvider implements IOutlineTreeProvider.ModeAware {
  private final static Logger logger = Logger.getLogger(N4JSOutlineTreeProvider.class);
  
  @Inject
  private ContainerTypesHelper containerTypesHelper;
  
  /**
   * This is used for cycling through the modes in the quick outline view. It is set on demand to detect whether
   * we run in quick outline view or in normal outline view.
   */
  private IOutlineTreeProvider.ModeAware modeAware = null;
  
  /**
   * casted access to the underlying label provider.
   */
  private N4JSLabelProvider getN4JSLabelProvider() {
    ILabelProvider _labelProvider = this.getLabelProvider();
    return ((N4JSLabelProvider) _labelProvider);
  }
  
  /**
   * This override captures for the duration of the invocation the {@link CancelIndicator} argument, for use during validation.
   * The superclass does the same (also captures it) but keeps it in private field.
   * All accesses to that field are mediated by {@link BackgroundOutlineTreeProvider#checkCanceled},
   * in our case besides checking-and-throwing as supported by that method we need to hand over the current cancel indicator.
   * <p>
   * For details on the protocol this method follows to hand it over see {@link N4JSLabelProvider#establishCancelIndicator}
   */
  @Override
  public IOutlineNode createRoot(final IXtextDocument document, final CancelIndicator cancelIndicator) {
    try {
      this.getN4JSLabelProvider().establishCancelIndicator(cancelIndicator);
      return super.createRoot(document, cancelIndicator);
    } finally {
      this.getN4JSLabelProvider().removeCancelIndicator();
    }
  }
  
  /**
   * Overridden to use dispatch methods ending in underscore {#createChildren_(...)}.
   * Dispatch call is wrapped in null/cancel check.
   */
  @Override
  public void createChildren(final IOutlineNode parentNode, final EObject modelElement) {
    this.checkCanceled();
    if (((modelElement != null) && parentNode.hasChildren())) {
      try {
        this.createChildren_(parentNode, modelElement);
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception ex = (Exception)_t;
          N4JSOutlineTreeProvider.logger.error("Error creating nodes for children", ex);
          throw new OperationCanceledException(("Canceled due to internal error: " + ex));
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
  }
  
  /**
   * First entry should not dispatch but specifically create the root node
   */
  protected void _createChildren_(final DocumentRootNode parentNode, final EObject modelElement) {
    super.createChildren(parentNode, modelElement);
  }
  
  protected void _createChildren_(final IOutlineNode parentNode, final EObject modelElement) {
    super.createChildren(parentNode, modelElement);
  }
  
  /**
   * only create entries on top level for elements listed in
   * isInstanceOfExpectedScriptChildren - so not exported functions
   * and not exported variables won't appear in the outline
   * for variable statements with one element only create one node
   */
  protected void _createChildren_(final IOutlineNode parentNode, final Script script) {
    EObjectNode node = null;
    Iterable<ScriptElement> _filterNull = IterableExtensions.<ScriptElement>filterNull(script.getScriptElements());
    for (final ScriptElement child : _filterNull) {
      {
        node = null;
        if ((child instanceof ExportDeclaration)) {
          final ExportableElement exportedElement = ((ExportDeclaration)child).getExportedElement();
          if ((exportedElement instanceof ExportedVariableStatement)) {
            int _size = ((ExportedVariableStatement)exportedElement).getVarDecl().size();
            boolean _equals = (_size == 1);
            if (_equals) {
              node = this.createNode(parentNode, IterableExtensions.<VariableDeclaration>head(((ExportedVariableStatement)exportedElement).getVarDecl()));
            } else {
              node = this.createNode(parentNode, exportedElement);
            }
          } else {
            if ((exportedElement != null)) {
              node = this.createNode(parentNode, exportedElement);
            }
          }
        } else {
          if ((this.isInstanceOfExpectedScriptChildren(child) && this.canCreateChildNode(child))) {
            node = this.createNode(parentNode, child);
            if ((node instanceof N4JSEObjectNode)) {
              ((N4JSEObjectNode)node).isLocal = true;
            }
          }
        }
      }
    }
  }
  
  /**
   * Create nodes for members (methods, fields) and field accessors (getter, setter).
   * If not turned off, also inherited (and consumed or polyfilled) members are shown.
   */
  protected void _createChildren_(final IOutlineNode parentNode, final N4ClassifierDefinition classifierDefinition) {
    Type _definedType = classifierDefinition.getDefinedType();
    final TClassifier tclassifier = ((TClassifier) _definedType);
    if (((tclassifier != null) && this.showInherited())) {
      final MemberList<TMember> members = this.containerTypesHelper.fromContext(tclassifier).members(tclassifier, false, true);
      Iterable<TMember> _filterNull = IterableExtensions.<TMember>filterNull(members);
      for (final TMember tchild : _filterNull) {
        {
          EObjectWithContext _eObjectWithContext = new EObjectWithContext(tchild, tclassifier);
          final EObjectNode node = this.createNodeForObjectWithContext(parentNode, _eObjectWithContext);
          if ((node instanceof N4JSEObjectNode)) {
            if (((tchild.getContainingType() != null) && (!Objects.equal(tchild.getContainingType(), tclassifier)))) {
              ((N4JSEObjectNode)node).isInherited = true;
            }
            ((N4JSEObjectNode)node).isMember = true;
            ((N4JSEObjectNode)node).isStatic = tchild.isStatic();
            ((N4JSEObjectNode)node).isPublic = (Objects.equal(tchild.getMemberAccessModifier(), MemberAccessModifier.PUBLIC) || 
              Objects.equal(tchild.getMemberAccessModifier(), MemberAccessModifier.PUBLIC_INTERNAL));
            ((N4JSEObjectNode)node).isConstructor = tchild.isConstructor();
            Resource _eResource = tchild.eResource();
            Resource _eResource_1 = tclassifier.eResource();
            boolean _equals = Objects.equal(_eResource, _eResource_1);
            if (_equals) {
              final ICompositeNode nodeModel = NodeModelUtils.getNode(tchild.getAstElement());
              if ((nodeModel != null)) {
                int _offset = nodeModel.getOffset();
                int _length = nodeModel.getLength();
                TextRegion _textRegion = new TextRegion(_offset, _length);
                ((N4JSEObjectNode)node).setTextRegion(_textRegion);
              }
            }
          }
        }
      }
    } else {
      Iterable<EObject> _filterNull_1 = IterableExtensions.<EObject>filterNull(classifierDefinition.eContents());
      for (final EObject child : _filterNull_1) {
        boolean _isInstanceOfExpectedClassifierChildren = this.isInstanceOfExpectedClassifierChildren(child);
        if (_isInstanceOfExpectedClassifierChildren) {
          this.createNode(parentNode, child);
        }
      }
    }
  }
  
  /**
   * Creates a node with context to be able to distinguish between owned and inherited/consumed members
   */
  public EObjectNode createNodeForObjectWithContext(final IOutlineNode parentNode, final EObjectWithContext objectWithContext) {
    this.checkCanceled();
    final Object text = this.getText(objectWithContext);
    final boolean isLeaf = this.isLeaf(objectWithContext.obj);
    if (((text == null) && isLeaf)) {
      return null;
    }
    final ImageDescriptor image = this.getImageDescriptor(objectWithContext);
    return this.getOutlineNodeFactory().createEObjectNode(parentNode, objectWithContext.obj, image, text, isLeaf);
  }
  
  /**
   * create nodes for literals
   */
  protected void _createChildren_(final IOutlineNode parentNode, final N4EnumDeclaration ed) {
    Iterable<N4EnumLiteral> _filterNull = IterableExtensions.<N4EnumLiteral>filterNull(ed.getLiterals());
    for (final N4EnumLiteral literal : _filterNull) {
      this.createNode(parentNode, literal);
    }
  }
  
  protected boolean _canCreateChildNode(final Object element) {
    return true;
  }
  
  protected boolean _canCreateChildNode(final N4ClassifierDefinition it) {
    Type _definedType = it.getDefinedType();
    return (null != _definedType);
  }
  
  /**
   * top level elements in outline view
   */
  private boolean isInstanceOfExpectedScriptChildren(final EObject child) {
    return this.isInstanceOfOneOfTheTypes(child, 
      ExportDeclaration.class, 
      N4ClassifierDeclaration.class, 
      N4EnumDeclaration.class, 
      FunctionDeclaration.class);
  }
  
  private boolean isInstanceOfExpectedClassifierChildren(final EObject child) {
    return this.isInstanceOfOneOfTheTypes(child, N4MemberDeclaration.class, FieldAccessor.class);
  }
  
  private boolean isInstanceOfOneOfTheTypes(final EObject eObject, final Class<?>... classes) {
    final Function1<Class<?>, Boolean> _function = (Class<?> it) -> {
      return Boolean.valueOf(it.isAssignableFrom(eObject.getClass()));
    };
    return IterableExtensions.<Class<?>>exists(((Iterable<Class<?>>)Conversions.doWrapArray(classes)), _function);
  }
  
  /**
   * Overridden to enable the dispatch-method mechanism of Xtend
   */
  protected boolean _isLeaf(final EObject modelElement) {
    return super.isLeaf(modelElement);
  }
  
  protected boolean _isLeaf(final Void _null) {
    return true;
  }
  
  protected boolean _isLeaf(final Script script) {
    final Function1<EObject, Boolean> _function = (EObject it) -> {
      return Boolean.valueOf(this.isInstanceOfExpectedScriptChildren(it));
    };
    boolean _exists = IterableExtensions.<EObject>exists(script.eContents(), _function);
    return (!_exists);
  }
  
  /**
   * Suppress + symbol in outline when classifier has no members
   * or we have a broken AST and the defined type of the classifier is not available yet.
   * 
   * This method directly works on AST, because the outline is called with the AST.
   */
  protected boolean _isLeaf(final N4ClassifierDefinition classifierDefinition) {
    Type _definedType = classifierDefinition.getDefinedType();
    final TClassifier tclassifier = ((TClassifier) _definedType);
    if ((tclassifier == null)) {
      return true;
    }
    boolean _showInherited = this.showInherited();
    if (_showInherited) {
      final MemberList<TMember> members = this.containerTypesHelper.fromContext(tclassifier).members(tclassifier, false, true);
      return IterableExtensions.isNullOrEmpty(members);
    } else {
      final Function1<EObject, Boolean> _function = (EObject it) -> {
        return Boolean.valueOf(this.isInstanceOfExpectedClassifierChildren(it));
      };
      boolean _exists = IterableExtensions.<EObject>exists(classifierDefinition.eContents(), _function);
      return (!_exists);
    }
  }
  
  /**
   * Type model members are always leaves.
   */
  protected boolean _isLeaf(final TMember member) {
    return true;
  }
  
  protected boolean _isLeaf(final N4FieldDeclaration md) {
    return true;
  }
  
  protected boolean _isLeaf(final FunctionOrFieldAccessor fa) {
    return true;
  }
  
  protected boolean _isLeaf(final ExportedVariableDeclaration vd) {
    return true;
  }
  
  protected boolean _isLeaf(final ImportDeclaration id) {
    int _size = id.getImportSpecifiers().size();
    return (_size == 1);
  }
  
  protected boolean showInherited() {
    if ((this.modeAware != null)) {
      OutlineMode _currentMode = this.modeAware.getCurrentMode();
      return Objects.equal(_currentMode, N4JSOutlineModes.SHOW_INHERITED_MODE);
    }
    return true;
  }
  
  @Override
  public List<OutlineMode> getOutlineModes() {
    return this.getOrCreateModeAware().getOutlineModes();
  }
  
  @Override
  public OutlineMode getCurrentMode() {
    return this.getOrCreateModeAware().getCurrentMode();
  }
  
  @Override
  public OutlineMode getNextMode() {
    return this.getOrCreateModeAware().getNextMode();
  }
  
  @Override
  public void setCurrentMode(final OutlineMode outlineMode) {
    this.getOrCreateModeAware().setCurrentMode(outlineMode);
  }
  
  /**
   * Create lazy in order to be able to distinguish between quick and non-quick outline mode.
   */
  private IOutlineTreeProvider.ModeAware getOrCreateModeAware() {
    if ((this.modeAware == null)) {
      N4JSOutlineModes _n4JSOutlineModes = new N4JSOutlineModes();
      this.modeAware = _n4JSOutlineModes;
    }
    return this.modeAware;
  }
  
  protected void createChildren_(final IOutlineNode parentNode, final EObject modelElement) {
    if (parentNode instanceof DocumentRootNode
         && modelElement != null) {
      _createChildren_((DocumentRootNode)parentNode, modelElement);
      return;
    } else if (parentNode != null
         && modelElement instanceof N4EnumDeclaration) {
      _createChildren_(parentNode, (N4EnumDeclaration)modelElement);
      return;
    } else if (parentNode != null
         && modelElement instanceof N4ClassifierDefinition) {
      _createChildren_(parentNode, (N4ClassifierDefinition)modelElement);
      return;
    } else if (parentNode != null
         && modelElement instanceof Script) {
      _createChildren_(parentNode, (Script)modelElement);
      return;
    } else if (parentNode != null
         && modelElement != null) {
      _createChildren_(parentNode, modelElement);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(parentNode, modelElement).toString());
    }
  }
  
  public boolean canCreateChildNode(final Object it) {
    if (it instanceof N4ClassifierDefinition) {
      return _canCreateChildNode((N4ClassifierDefinition)it);
    } else if (it != null) {
      return _canCreateChildNode(it);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(it).toString());
    }
  }
  
  protected boolean isLeaf(final EObject vd) {
    if (vd instanceof ExportedVariableDeclaration) {
      return _isLeaf((ExportedVariableDeclaration)vd);
    } else if (vd instanceof N4ClassifierDefinition) {
      return _isLeaf((N4ClassifierDefinition)vd);
    } else if (vd instanceof N4FieldDeclaration) {
      return _isLeaf((N4FieldDeclaration)vd);
    } else if (vd instanceof ImportDeclaration) {
      return _isLeaf((ImportDeclaration)vd);
    } else if (vd instanceof TMember) {
      return _isLeaf((TMember)vd);
    } else if (vd instanceof FunctionOrFieldAccessor) {
      return _isLeaf((FunctionOrFieldAccessor)vd);
    } else if (vd instanceof Script) {
      return _isLeaf((Script)vd);
    } else if (vd != null) {
      return _isLeaf(vd);
    } else if (vd == null) {
      return _isLeaf((Void)null);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(vd).toString());
    }
  }
}
