/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.quickfix;

import com.google.common.base.Objects;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.n4js.n4JS.N4Modifier;
import org.eclipse.xtext.validation.Issue;

/**
 * Utility methods for QuickFixes
 */
@SuppressWarnings("all")
public class QuickfixUtil {
  /**
   * Helper class to check file system access
   */
  public static class WorkspaceFileHandle {
    private IPath path;
    
    private IFile file;
    
    public static QuickfixUtil.WorkspaceFileHandle fileContainingObjectURI(final URI uri) {
      if ((uri == null)) {
        return null;
      }
      int _length = uri.toString().length();
      boolean _lessThan = (_length < 1);
      if (_lessThan) {
        return null;
      }
      URI fileURI = uri.deresolve(URI.createURI("platform:/resource/"));
      String[] fileFragments = fileURI.toString().split("#");
      String _get = fileFragments[0];
      Path _path = new Path(_get);
      return new QuickfixUtil.WorkspaceFileHandle(_path);
    }
    
    public WorkspaceFileHandle(final IPath path) {
      this.path = path;
      this.file = ResourcesPlugin.getWorkspace().getRoot().getFile(this.path);
    }
    
    public boolean getReadOnly() {
      return this.file.isReadOnly();
    }
    
    public boolean getIsDerived() {
      return this.file.isDerived();
    }
  }
  
  public static class IssueUserDataKeysExtension {
    public String getUserData(final Issue issue, final String key) {
      final String[] userData = issue.getData();
      if ((null == userData)) {
        return null;
      }
      final int userDataLength = userData.length;
      {
        int i = 0;
        boolean _while = (i < userDataLength);
        while (_while) {
          if ((userData[i].equals(key) && ((i + 1) < userDataLength))) {
            return userData[(i + 1)];
          }
          int _i = i;
          i = (_i + 2);
          _while = (i < userDataLength);
        }
      }
      return null;
    }
  }
  
  /**
   * Returns the eobject at the given uri.
   * @param set
   * 		Resource set containing the demanded resource
   * @param uri
   * 		String uri of the demanded object (partly resource uri partly object uri)
   */
  public static EObject getEObjectForUri(final ResourceSet set, final String uri) {
    final String[] fileFragments = uri.split("#");
    int _length = fileFragments.length;
    boolean _lessThan = (_length < 2);
    if (_lessThan) {
      return null;
    }
    final String fileURI = fileFragments[0];
    final String objectURI = fileFragments[1];
    final Resource resource = set.getResource(URI.createURI(fileURI), true);
    if ((resource != null)) {
      final EObject object = resource.getEObject(objectURI);
      return object;
    }
    return null;
  }
  
  /**
   * True if the objectUri containing resource is modifiable (i.e. file system write access)
   * 
   * @param objectUri
   * 		URI referring to eobject in resource
   */
  public static boolean isContainingResourceModifiable(final String objectUri) {
    URI uri = URI.createURI(objectUri, true);
    if ((uri == null)) {
      return false;
    }
    QuickfixUtil.WorkspaceFileHandle fileHandle = QuickfixUtil.WorkspaceFileHandle.fileContainingObjectURI(uri);
    if ((fileHandle == null)) {
      return false;
    }
    boolean _isDerived = fileHandle.getIsDerived();
    if (_isDerived) {
      return false;
    }
    boolean _readOnly = fileHandle.getReadOnly();
    if (_readOnly) {
      return false;
    }
    return true;
  }
  
  /**
   * Returns the N4Modifier for a given suggestion
   * 
   * @param suggestion
   * 	String encoded access modifier suggestion
   */
  public static N4Modifier modifierForSuggestion(final String suggestion) {
    String _upperCase = suggestion.toUpperCase();
    if (_upperCase != null) {
      switch (_upperCase) {
        case "PUBLIC":
          return N4Modifier.PUBLIC;
        case "PROTECTED":
          return N4Modifier.PROTECTED;
        case "PUBLICINTERNAL":
          return N4Modifier.PUBLIC;
        case "PROTECTEDINTERNAL":
          return N4Modifier.PROTECTED;
        case "PROJECT":
          return N4Modifier.PROJECT;
        default:
          return N4Modifier.UNDEFINED;
      }
    } else {
      return N4Modifier.UNDEFINED;
    }
  }
  
  /**
   * Returns if the suggestions asks for @Internal
   * 
   * @param suggestion
   * 	String encoded access modifier suggestion
   */
  public static boolean modifierSuggestionIsInternal(final String suggestion) {
    boolean _xblockexpression = false;
    {
      if ((Objects.equal(suggestion.toUpperCase(), "PUBLICINTERNAL") || Objects.equal(suggestion.toUpperCase(), "PROTECTEDINTERNAL"))) {
        return true;
      }
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns a human readable string describing the suggested access modifier
   * 
   * @param suggestion
   * 	String encoded access modifier suggestion
   */
  public static String readableStringForSuggestion(final String suggestion) {
    final N4Modifier modifier = QuickfixUtil.modifierForSuggestion(suggestion);
    final boolean isInternal = QuickfixUtil.modifierSuggestionIsInternal(suggestion);
    String output = "";
    if (modifier != null) {
      switch (modifier) {
        case PRIVATE:
          String _output = output;
          output = (_output + "private");
          break;
        case PROJECT:
          String _output_1 = output;
          output = (_output_1 + "project");
          break;
        case PROTECTED:
          String _output_2 = output;
          output = (_output_2 + "protected");
          break;
        case PUBLIC:
          String _output_3 = output;
          output = (_output_3 + "public");
          break;
        default:
          String _output_4 = output;
          output = (_output_4 + "undefined");
          break;
      }
    } else {
      String _output_4 = output;
      output = (_output_4 + "undefined");
    }
    if (isInternal) {
      String _output_5 = output;
      output = (_output_5 + ", @Internal");
    }
    return output;
  }
}
