/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js;

import com.google.inject.Inject;
import com.google.inject.Provider;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.n4js.N4JSInjectorProvider;
import org.eclipse.n4js.N4JSParseHelper;
import org.eclipse.n4js.N4JSTestHelper;
import org.eclipse.n4js.N4JSValidationTestHelper;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.resource.N4JSResource;
import org.eclipse.n4js.resource.UserdataMapper;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescription;
import org.eclipse.xtext.resource.IResourceDescriptions;
import org.eclipse.xtext.resource.XtextResourceSet;
import org.eclipse.xtext.resource.impl.ResourceDescriptionsProvider;
import org.eclipse.xtext.testing.InjectWith;
import org.eclipse.xtext.testing.XtextRunner;
import org.eclipse.xtext.testing.util.ParseHelper;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.junit.Assert;
import org.junit.runner.RunWith;

/**
 * Base class for simple N4JS unit tests. In more special cases, consider using base classes such as
 * {@code org.eclipse.n4js.tests.parser.AbstractParserTest}.
 */
@RunWith(XtextRunner.class)
@InjectWith(N4JSInjectorProvider.class)
@SuppressWarnings("all")
public class AbstractN4JSTest extends Assert {
  @Inject
  @Extension
  protected N4JSTestHelper _n4JSTestHelper;
  
  @Inject
  @Extension
  protected N4JSParseHelper _n4JSParseHelper;
  
  @Inject
  @Extension
  protected N4JSValidationTestHelper _n4JSValidationTestHelper;
  
  @Inject
  private ResourceDescriptionsProvider resourceDescriptionsProvider;
  
  @Inject
  private Provider<XtextResourceSet> resourceSetProvider;
  
  @Inject
  private IN4JSCore n4jsCore;
  
  /**
   * Creates a new N4JSResource for the given URI inside a newly created {@link  ResourceSet}.
   * 
   * The returned resource is not loaded yet.
   */
  protected N4JSResource createFromFile(final URI uri) {
    return IterableExtensions.<N4JSResource>head(this.createFromFiles(uri));
  }
  
  /**
   * Creates a new N4JSResource for each of the given URIs inside the same, newly created {@link ResourceSet}.
   * 
   * The returned resources are not loaded yet.
   */
  protected List<N4JSResource> createFromFiles(final URI... uris) {
    final XtextResourceSet resSet = this.resourceSetProvider.get();
    final Function<URI, N4JSResource> _function = (URI u) -> {
      final URI normalizedURI = resSet.getURIConverter().normalize(u);
      Resource _createResource = resSet.createResource(normalizedURI);
      return ((N4JSResource) _createResource);
    };
    final List<N4JSResource> resources = ((List<URI>)Conversions.doWrapArray(uris)).stream().<N4JSResource>map(_function).collect(Collectors.<N4JSResource>toList());
    return resources;
  }
  
  /**
   * Creates and loads a new N4JSResource inside a newly created resource set for the given URI. The resource returned
   * is loaded, but installation of derived state (i.e. types builder) has not yet been triggered.
   */
  protected N4JSResource loadFromFile(final URI uri) {
    try {
      final N4JSResource res = this.createFromFile(uri);
      res.load(CollectionLiterals.<Object, Object>emptyMap());
      return res;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Creates, loads, and installs derived state of a new N4JSResource inside a newly created resource set. The
   * resource returned is loaded and has its derived state installed (i.e. types builder completed) but is not yet
   * post-processed.
   * <p>
   * Same as methods such as {@link N4JSParseHelper#parseN4js(CharSequence)}, {@link ParseHelper#parse(CharSequence)},
   * but reads source from a file instead of a string.
   */
  protected N4JSResource parseFromFile(final URI uri) {
    final N4JSResource res = this.loadFromFile(uri);
    res.getContents();
    return res;
  }
  
  /**
   * Simulates re-loading the resource with the given URI
   * from the index.
   * 
   * @See #loadFromDescriptions(URI...)
   */
  protected N4JSResource loadFromDescription(final URI uri) {
    return IterableExtensions.<N4JSResource>head(this.loadFromDescriptions(uri));
  }
  
  /**
   * Simulates loading resources for the given URIs, serializing them
   * into the index, unloading them, reloading them from index and returning
   * the resulting reloaded resources.
   * 
   * During the execution this method also asserts that no validation
   * issues can be detected.
   * 
   * This method triggers the reconciliation of the {@link TModule} instances
   * that can be deserialized from the index.
   */
  protected List<N4JSResource> loadFromDescriptions(final URI... uris) {
    boolean _isEmpty = ((List<URI>)Conversions.doWrapArray(uris)).isEmpty();
    if (_isEmpty) {
      return Collections.<N4JSResource>unmodifiableList(CollectionLiterals.<N4JSResource>newArrayList());
    }
    final List<N4JSResource> resources = this.createFromFiles(uris);
    final Consumer<N4JSResource> _function = (N4JSResource res) -> {
      try {
        res.load(CollectionLiterals.<Object, Object>emptyMap());
        res.getContents();
        res.performPostProcessing();
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    };
    resources.forEach(_function);
    final ResourceSet resSet = IterableExtensions.<N4JSResource>head(resources).getResourceSet();
    final Function<N4JSResource, IResourceDescription> _function_1 = (N4JSResource res) -> {
      this._n4JSValidationTestHelper.assertNoErrors(res);
      final URI uri = res.getURI();
      final IResourceDescriptions resourceDescriptions = this.resourceDescriptionsProvider.getResourceDescriptions(res);
      Assert.assertFalse("index has not been filled", IterableExtensions.isEmpty(resourceDescriptions.getAllResourceDescriptions()));
      final IResourceDescription description = resourceDescriptions.getResourceDescription(uri);
      IterableExtensions.<IEObjectDescription>head(description.getExportedObjects()).getUserData(UserdataMapper.USERDATA_KEY_SERIALIZED_SCRIPT);
      Assert.assertTrue(res.isFullyInitialized());
      Assert.assertTrue(res.isFullyProcessed());
      return description;
    };
    final List<IResourceDescription> descriptions = resources.stream().<IResourceDescription>map(_function_1).collect(Collectors.<IResourceDescription>toList());
    final Consumer<N4JSResource> _function_2 = (N4JSResource res) -> {
      res.unload();
      resSet.getResources().remove(res);
    };
    resources.forEach(_function_2);
    final Function<IResourceDescription, N4JSResource> _function_3 = (IResourceDescription resDesc) -> {
      Resource _eResource = this.n4jsCore.loadModuleFromIndex(resSet, resDesc, false).eResource();
      final N4JSResource resFromIndex = ((N4JSResource) _eResource);
      Assert.assertTrue(resFromIndex.getScript().eIsProxy());
      Assert.assertFalse(resFromIndex.getModule().eIsProxy());
      return resFromIndex;
    };
    final List<N4JSResource> reloadedResources = descriptions.stream().<N4JSResource>map(_function_3).collect(Collectors.<N4JSResource>toList());
    return reloadedResources;
  }
}
