/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ts.utils;

import com.google.inject.Inject;
import java.util.Iterator;
import java.util.List;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.utils.SuperTypesList;
import org.eclipse.n4js.ts.utils.TypeCompareHelper;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Helper providing utility methods for type and type ref handling, needs to be injected. Static utility methods
 * can be found in {@link TypeUtils}.
 */
@SuppressWarnings("all")
public class TypeHelper {
  @Inject
  @Extension
  private TypeCompareHelper _typeCompareHelper;
  
  /**
   * Collects all declared super types of a type referenced by a type references, recognizing cyclic dependencies in
   * case of invalid type hierarchies. Type arguments are not considered during comparison, thus G<A> and G<B>
   * will result in a single reference in the result.
   * 
   * @param reflexive if true, type itself is also added to the list of super types
   * @return ordered list of super types, using a depth first search, starting with classes, then roles, and eventually interfaces.
   */
  public SuperTypesList<TypeRef> collectAllDeclaredSuperTypesTypeargsIgnored(final TypeRef ref, final boolean reflexive) {
    final SuperTypesList<TypeRef> allDeclaredSuperTypes = SuperTypesList.<TypeRef>newSuperTypesList(this._typeCompareHelper.getTypeRefComparator());
    allDeclaredSuperTypes.add(ref);
    this.collectAllDeclaredSuperTypesTypeargsIgnored(ref, allDeclaredSuperTypes);
    if ((!reflexive)) {
      allDeclaredSuperTypes.remove(ref);
    }
    return allDeclaredSuperTypes;
  }
  
  /**
   * Collects all declared super types of a type referenced by a type references, recognizing cyclic dependencies in
   * case of invalid type hierarchies. Type arguments are considered during comparison, thus G<A> and G<B>
   * will both be part of the result.
   * 
   * @param reflexive if true, type itself is also added to the list of super types
   * @return ordered list of super types, using a depth first search, starting with classes, then roles, and eventually interfaces.
   */
  public SuperTypesList<TypeRef> collectAllDeclaredSuperTypesWithTypeargs(final TypeRef ref, final boolean reflexive) {
    final SuperTypesList<TypeRef> allDeclaredSuperTypes = SuperTypesList.<TypeRef>newSuperTypesList(this._typeCompareHelper.getTypeRefComparator());
    allDeclaredSuperTypes.add(ref);
    this.collectAllDeclaredSuperTypesTypeargsIgnored(ref, allDeclaredSuperTypes);
    if ((!reflexive)) {
      allDeclaredSuperTypes.remove(ref);
    }
    return allDeclaredSuperTypes;
  }
  
  public SuperTypesList<Type> collectAllDeclaredSuperTypes(final Type type, final boolean reflexive) {
    final SuperTypesList<Type> allDeclaredSuperTypes = SuperTypesList.<Type>newSuperTypesList(this._typeCompareHelper.getTypeComparator());
    allDeclaredSuperTypes.add(type);
    this.collectAllDeclaredSuperTypes(type, allDeclaredSuperTypes);
    if ((!reflexive)) {
      allDeclaredSuperTypes.remove(type);
    }
    return allDeclaredSuperTypes;
  }
  
  /**
   * Collects all type references of given type reference and add these types to both given collections. This method requires the tree set to use the
   * {@link TypeCompareHelper#getTypeRefComparator()}. In most cases, you probably will call {@link #collectAllDeclaredSuperTypes(TypeRef, boolean)} instead of calling this method
   * directly. However, for some optimized methods, it may be usefull to call this method directly.
   * 
   * @param allDeclaredSuperTypes
   *            needs to be instantiated with correct comparator, see {@link collectAllDeclaredSuperTypes(TypeRef)}, that is the types are ordered by their
   * 				qualified name
   * @param allDeclaredSuperTypesOrdered ordered list of super types, using a depth first search, starting with classes, then roles, and eventually interfaces.
   */
  public void collectAllDeclaredSuperTypesTypeargsIgnored(final TypeRef ref, final SuperTypesList<TypeRef> allDeclaredSuperTypes) {
    if (((ref != null) && (ref.getDeclaredType() != null))) {
      Iterable<? extends ParameterizedTypeRef> _declaredSuperTypes = TypeUtils.declaredSuperTypes(ref.getDeclaredType());
      for (final ParameterizedTypeRef superTypeRef : _declaredSuperTypes) {
        boolean _add = allDeclaredSuperTypes.add(superTypeRef);
        if (_add) {
          this.collectAllDeclaredSuperTypesTypeargsIgnored(superTypeRef, allDeclaredSuperTypes);
        }
      }
    }
  }
  
  public void collectAllDeclaredSuperTypes(final Type type, final SuperTypesList<Type> allDeclaredSuperTypes) {
    if ((type != null)) {
      final Function1<ParameterizedTypeRef, Type> _function = (ParameterizedTypeRef it) -> {
        return it.getDeclaredType();
      };
      Iterable<Type> _map = IterableExtensions.map(TypeUtils.declaredSuperTypes(type), _function);
      for (final Type superType : _map) {
        boolean _add = allDeclaredSuperTypes.add(superType);
        if (_add) {
          this.collectAllDeclaredSuperTypes(superType, allDeclaredSuperTypes);
        }
      }
    }
  }
  
  /**
   * Removes the first occurrence of the given type from the iterable, whereby the type is found by name using the
   * {@link TypeCompareHelper#getTypeRefComparator()}. The iterator of the iterable needs to support the {@link Iterator#remove()} operation.
   * @return true if type has been found, false otherwise
   */
  public boolean removeTypeRef(final Iterable<TypeRef> typeRefs, final TypeRef toBeRemoved) {
    final Iterator<TypeRef> iter = typeRefs.iterator();
    while (iter.hasNext()) {
      int _compare = this._typeCompareHelper.compare(toBeRemoved, iter.next());
      boolean _tripleEquals = (_compare == 0);
      if (_tripleEquals) {
        iter.remove();
        return true;
      }
    }
    return false;
  }
  
  /**
   * Retains all types from list of refs.
   * <p>
   * This method is optimized for leastCommonSuperType and
   * assumes that all types in orderedRefs are ordered as returned by collecAllDeclaredSuperTypes().
   * The iterator returned by typeRefs must support the {@link Iterator#remove()} operation.
   */
  public void retainAllTypeRefs(final Iterable<TypeRef> typeRefs, final SuperTypesList<TypeRef> typesToBeRetained) {
    final Iterator<TypeRef> iter = typeRefs.iterator();
    while (iter.hasNext()) {
      boolean _contains = typesToBeRetained.contains(iter.next());
      boolean _not = (!_contains);
      if (_not) {
        iter.remove();
      }
    }
  }
  
  /**
   * Returns true if given iterable contains the type ref, using the {@link TypeCompareHelper#getTypeRefComparator()} for finding the type ref.
   */
  public boolean containsByType(final Iterable<TypeRef> typeRefs, final TypeRef typeRef) {
    final Function1<TypeRef, Boolean> _function = (TypeRef it) -> {
      int _compare = this._typeCompareHelper.compare(typeRef, it);
      return Boolean.valueOf((_compare == 0));
    };
    return IterableExtensions.<TypeRef>exists(typeRefs, _function);
  }
  
  /**
   * Returns the index of the type ref contained in the typeRefs collections, which either equals the given type ref
   * by means of the {@link TypeCompareHelper#getTypeRefComparator()}, or which is assignment compatible in case of primitive types.
   * 
   * @return index or -1, if type ref has not been found
   */
  public int findTypeRefOrAssignmentCompatible(final List<TypeRef> typeRefs, final TypeRef typeRef) {
    PrimitiveType _xifexpression = null;
    Type _declaredType = typeRef.getDeclaredType();
    if ((_declaredType instanceof PrimitiveType)) {
      Type _declaredType_1 = typeRef.getDeclaredType();
      _xifexpression = ((PrimitiveType) _declaredType_1).getAssignmentCompatible();
    } else {
      _xifexpression = null;
    }
    final PrimitiveType assignmentCompatible = _xifexpression;
    int i = 0;
    while ((i < ((Object[])Conversions.unwrapArray(typeRefs, Object.class)).length)) {
      {
        final TypeRef t = typeRefs.get(i);
        int _compare = this._typeCompareHelper.compare(typeRef, t);
        boolean _equals = (_compare == 0);
        if (_equals) {
          return i;
        }
        if ((assignmentCompatible != null)) {
          final Type type = t.getDeclaredType();
          if ((type instanceof PrimitiveType)) {
            if (((this._typeCompareHelper.getTypeComparator().compare(assignmentCompatible, type) == 0) || 
              (this._typeCompareHelper.getTypeComparator().compare(typeRef.getDeclaredType(), ((PrimitiveType)type).getAssignmentCompatible()) == 0))) {
              return i;
            }
          }
        }
        i = (i + 1);
      }
    }
    return (-1);
  }
}
