/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.function.Consumer;
import org.eclipse.n4js.utils.AbstractNLSProcessor;
import org.eclipse.n4js.utils.NLS;
import org.eclipse.xtend.lib.macro.TransformationContext;
import org.eclipse.xtend.lib.macro.declaration.AnnotationReference;
import org.eclipse.xtend.lib.macro.declaration.MutableClassDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableMethodDeclaration;
import org.eclipse.xtend.lib.macro.declaration.TypeReference;
import org.eclipse.xtend.lib.macro.declaration.Visibility;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IntegerRange;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * See annotation NLS
 */
@SuppressWarnings("all")
public class NLSProcessor extends AbstractNLSProcessor {
  private static List<String> SEVERITIES = Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("error", "warning", "info", "ignore"));
  
  @Override
  public Class<?> getAnnotationType() {
    return NLS.class;
  }
  
  @Override
  public void addMembers(final MutableClassDeclaration annotatedClass, final AnnotationReference nlsAnnotation, final TransformationContext context, final String propertyFileNameValue) {
    super.addMembers(annotatedClass, nlsAnnotation, context, propertyFileNameValue);
    this.addGetDefaultSeverityMethod(annotatedClass, nlsAnnotation, context);
  }
  
  @Override
  public void addGetStringMethod(final MutableClassDeclaration annotatedClass, final AnnotationReference nlsAnnotation, @Extension final TransformationContext context) {
    final String methodName = "getString";
    this.checkForExistentMethod(annotatedClass, methodName, context, nlsAnnotation, 1);
    final Procedure1<MutableMethodDeclaration> _function = (MutableMethodDeclaration it) -> {
      it.setVisibility(Visibility.PRIVATE);
      it.setStatic(true);
      it.setReturnType(context.getString());
      it.addParameter("key", context.getString());
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append("try {");
          _builder.newLine();
          _builder.append("\t");
          TypeReference _string = context.getString();
          _builder.append(_string, "\t");
          _builder.append(" value = ");
          String _simpleName = annotatedClass.findDeclaredField(AbstractNLSProcessor.RESOURCE_BUNDLE_FIELD).getSimpleName();
          _builder.append(_simpleName, "\t");
          _builder.append(".getString(key);");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          TypeReference _string_1 = context.getString();
          _builder.append(_string_1, "\t");
          _builder.append("[] parts = value.split(\";;;\");");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          _builder.append("return parts[1];");
          _builder.newLine();
          _builder.append("} catch (");
          _builder.append(MissingResourceException.class);
          _builder.append(" e) {");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          _builder.append("return \'!\' + key + \'!\';");
          _builder.newLine();
          _builder.append("}");
          _builder.newLine();
        }
      };
      it.setBody(_client);
    };
    annotatedClass.addMethod(methodName, _function);
  }
  
  private void addGetDefaultSeverityMethod(final MutableClassDeclaration annotatedClass, final AnnotationReference nlsAnnotation, @Extension final TransformationContext context) {
    final String methodName = "getDefaultSeverity";
    this.checkForExistentMethod(annotatedClass, methodName, context, nlsAnnotation, 1);
    final Procedure1<MutableMethodDeclaration> _function = (MutableMethodDeclaration it) -> {
      it.setVisibility(Visibility.PUBLIC);
      it.setStatic(true);
      it.setReturnType(context.newTypeReference(Severity.class));
      it.addParameter("key", context.getString());
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append("try {");
          _builder.newLine();
          _builder.append("\t");
          TypeReference _string = context.getString();
          _builder.append(_string, "\t");
          _builder.append(" value = ");
          String _simpleName = annotatedClass.findDeclaredField(AbstractNLSProcessor.RESOURCE_BUNDLE_FIELD).getSimpleName();
          _builder.append(_simpleName, "\t");
          _builder.append(".getString(key);");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          TypeReference _string_1 = context.getString();
          _builder.append(_string_1, "\t");
          _builder.append("[] parts = value.split(\";;;\");");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          TypeReference _string_2 = context.getString();
          _builder.append(_string_2, "\t");
          _builder.append(" defaultSeverity = parts[0];");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          _builder.append("return ");
          _builder.append(Severity.class, "\t");
          _builder.append(".valueOf(defaultSeverity.toUpperCase());");
          _builder.newLineIfNotEmpty();
          _builder.append("} catch (");
          _builder.append(MissingResourceException.class);
          _builder.append(" e) {");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          _builder.append("return null;");
          _builder.newLine();
          _builder.append("}");
          _builder.newLine();
        }
      };
      it.setBody(_client);
    };
    annotatedClass.addMethod(methodName, _function);
  }
  
  @Override
  public void addMethod(final Map.Entry<Object, Object> entry, final MutableClassDeclaration annotatedClass, final AnnotationReference nlsAnnotation, @Extension final TransformationContext context) {
    Object _value = entry.getValue();
    final String value = ((String) _value);
    final String[] parts = value.split(";;;");
    int _length = parts.length;
    boolean _notEquals = (_length != 2);
    if (_notEquals) {
      Object _key = entry.getKey();
      String _plus = ("Value for " + _key);
      String _plus_1 = (_plus + " in properties file doesn\'t follow pattern \'defaultSeverity;;;message\'");
      context.addError(nlsAnnotation, _plus_1);
      return;
    }
    final String defaultSeverity = parts[0];
    boolean _contains = NLSProcessor.SEVERITIES.contains(defaultSeverity);
    boolean _not = (!_contains);
    if (_not) {
      String _join = IterableExtensions.join(NLSProcessor.SEVERITIES, ", ");
      String _plus_2 = ((defaultSeverity + " is not a valid severity (which would be ") + _join);
      String _plus_3 = (_plus_2 + ").");
      context.addError(nlsAnnotation, _plus_3);
      return;
    }
    final String message = parts[1];
    final int wildcardCount = this.getWildcardCount(message);
    Iterable<String> _xifexpression = null;
    if ((wildcardCount > 0)) {
      final Function1<Integer, String> _function = (Integer it) -> {
        return ("param" + it);
      };
      _xifexpression = IterableExtensions.<Integer, String>map(new IntegerRange(0, (wildcardCount - 1)), _function);
    } else {
      _xifexpression = CollectionLiterals.<String>newArrayList();
    }
    final Iterable<String> params = _xifexpression;
    Object _key_1 = entry.getKey();
    final String getMessageForMethodName = ("getMessageFor" + _key_1);
    this.checkForExistentMethod(annotatedClass, getMessageForMethodName, context, nlsAnnotation, IterableExtensions.size(params));
    final Procedure1<MutableMethodDeclaration> _function_1 = (MutableMethodDeclaration it) -> {
      it.setVisibility(Visibility.PUBLIC);
      it.setStatic(true);
      it.setReturnType(context.getString());
      final Consumer<String> _function_2 = (String param) -> {
        it.addParameter(param, context.getObject());
      };
      params.forEach(_function_2);
      it.setDocComment(message);
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append("return ");
          _builder.append(AbstractNLSProcessor.nlsClass);
          _builder.append(".bind(getString(");
          Object _key = entry.getKey();
          _builder.append(_key);
          _builder.append("), new Object [] { ");
          String _join = IterableExtensions.join(params, ", ");
          _builder.append(_join);
          _builder.append(" });");
        }
      };
      it.setBody(_client);
    };
    annotatedClass.addMethod(getMessageForMethodName, _function_1);
  }
}
