/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.jsdoc2spec.adoc;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.jsdoc.N4JSDocletParser;
import org.eclipse.n4js.jsdoc.dom.Composite;
import org.eclipse.n4js.jsdoc.dom.ContentNode;
import org.eclipse.n4js.jsdoc.dom.Doclet;
import org.eclipse.n4js.jsdoc.dom.InlineTag;
import org.eclipse.n4js.jsdoc.dom.LineTag;
import org.eclipse.n4js.jsdoc.dom.Literal;
import org.eclipse.n4js.jsdoc.dom.SimpleTypeReference;
import org.eclipse.n4js.jsdoc.dom.TagValue;
import org.eclipse.n4js.jsdoc.dom.Text;
import org.eclipse.n4js.jsdoc.dom.VariableReference;
import org.eclipse.n4js.jsdoc.tags.DefaultLineTagDefinition;
import org.eclipse.n4js.jsdoc2spec.KeyUtils;
import org.eclipse.n4js.jsdoc2spec.RepoRelativePath;
import org.eclipse.n4js.jsdoc2spec.SpecTestInfo;
import org.eclipse.n4js.jsdoc2spec.adoc.Html2ADocConverter;
import org.eclipse.n4js.jsdoc2spec.adoc.RepoRelativePathHolder;
import org.eclipse.n4js.jsdoc2spec.adoc.SourceEntry;
import org.eclipse.n4js.jsdoc2spec.adoc.SourceEntryFactory;
import org.eclipse.n4js.jsdoc2spec.adoc.SpecIdentifiableElementSection;
import org.eclipse.n4js.jsdoc2spec.adoc.SpecRequirementSection;
import org.eclipse.n4js.jsdoc2spec.adoc.SpecSection;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.FieldAccessor;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.SyntaxRelatedTElement;
import org.eclipse.n4js.ts.types.TAnnotation;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMemberWithAccessModifier;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TN4Classifier;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.util.AllSuperTypesCollector;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.n4js.validation.ValidatorMessageHelper;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Print AsciiDoc code of specification JSDoc. Start and end markers are printed by client.
 * 
 * Needs to be injectd.
 */
@SuppressWarnings("all")
public class ADocSerializer {
  @Inject
  @Extension
  private Html2ADocConverter _html2ADocConverter;
  
  @Inject
  private ValidatorMessageHelper validatorMessageHelper;
  
  @Inject
  private N4JSElementKeywordProvider keywordProvider;
  
  @Inject
  private RepoRelativePathHolder repoPathHolder;
  
  public CharSequence process(final SpecRequirementSection spec, final Map<String, SpecSection> specsByKey) {
    final StringBuilder strb = new StringBuilder();
    this.appendSpecElementPost(strb, spec, specsByKey);
    return strb;
  }
  
  public CharSequence process(final SpecIdentifiableElementSection spec, final Map<String, SpecSection> specsByKey) {
    final StringBuilder strb = new StringBuilder();
    this.appendSpecElementPre(strb, spec);
    this.appendSpec(strb, spec);
    this.appendSpecElementPost(strb, spec, specsByKey);
    return strb;
  }
  
  private StringBuilder appendSpecElementPost(final StringBuilder strb, final SpecRequirementSection spec, final Map<String, SpecSection> map) {
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(spec.getTestInfosForType());
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      final Function1<SpecTestInfo, String> _function = (SpecTestInfo it) -> {
        return it.testTitle;
      };
      final Map<String, List<SpecTestInfo>> groupdTests = IterableExtensions.<String, SpecTestInfo>groupBy(spec.getTestInfosForType(), _function);
      this.<String>appendApiConstraints(strb, groupdTests);
    }
    return strb;
  }
  
  private StringBuilder appendSpec(final StringBuilder strb, final SpecIdentifiableElementSection spec) {
    strb.append("\n");
    boolean addedTaskLinks = false;
    final EList<LineTag> taskTags = spec.getDoclet().lineTags(N4JSDocletParser.TAG_TASK.getTitle());
    for (final LineTag tag : taskTags) {
      {
        final String taskID = N4JSDocletParser.TAG_TASK.getValue(tag, "");
        boolean _isEmpty = taskID.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          boolean _startsWith = taskID.startsWith("*");
          if (_startsWith) {
            this.appendTaskLink(strb, taskID.substring(1));
          } else {
            this.appendTaskLink(strb, taskID);
          }
          strb.append(" ");
          addedTaskLinks = true;
        }
      }
    }
    if (addedTaskLinks) {
      strb.append("\n\n");
    }
    this.appendSpecDescription(strb, spec);
    return strb;
  }
  
  private StringBuilder appendSpecDescription(final StringBuilder strb, final SpecIdentifiableElementSection spec) {
    final Doclet doclet = spec.getDoclet();
    final boolean bSpecFromDescr = doclet.hasLineTag(N4JSDocletParser.TAG_SPECFROMDESCR.getTitle());
    final String reqID = this.getReqId(doclet);
    final EList<LineTag> specTags = doclet.lineTags(N4JSDocletParser.TAG_SPEC.getTitle());
    if (((specTags.isEmpty() && (!bSpecFromDescr)) && reqID.isEmpty())) {
      return strb;
    }
    if ((!((spec.idElement instanceof TN4Classifier) || (spec.idElement instanceof TEnum)))) {
      strb.append("==== Description\n\n");
    }
    boolean _isEmpty = reqID.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      strb.append((("See req:" + reqID) + "[].\n"));
    }
    this.appendContents(strb, doclet);
    for (final LineTag tag : specTags) {
      this.appendContents(strb, tag.getValueByKey(DefaultLineTagDefinition.CONTENTS));
    }
    strb.append("\n");
    return strb;
  }
  
  private StringBuilder appendSpecDescriptions(final StringBuilder strb, final Iterable<Doclet> doclets) {
    for (final Doclet doclet : doclets) {
      {
        boolean bSpecFromDescr = doclet.hasLineTag(N4JSDocletParser.TAG_SPECFROMDESCR.getTitle());
        final EList<LineTag> specTags = doclet.lineTags(N4JSDocletParser.TAG_SPEC.getTitle());
        if (((!specTags.isEmpty()) || bSpecFromDescr)) {
          this.appendContents(strb, doclet);
          for (final LineTag tag : specTags) {
            this.appendContents(strb, tag.getValueByKey(DefaultLineTagDefinition.CONTENTS));
          }
        }
      }
    }
    return strb;
  }
  
  private StringBuilder appendContents(final StringBuilder strb, final Composite composite) {
    EList<ContentNode> _contents = composite.getContents();
    for (final ContentNode c : _contents) {
      strb.append(this.processContent(c));
    }
    boolean _isEmpty = composite.getContents().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      strb.append("\n");
    }
    return strb;
  }
  
  private CharSequence _processContent(final ContentNode node) {
    return null;
  }
  
  private CharSequence _processContent(final Text node) {
    return this._html2ADocConverter.transformHTML(node.getText());
  }
  
  private CharSequence _processContent(final Literal node) {
    return this._html2ADocConverter.transformHTML(node.getValue());
  }
  
  private CharSequence _processContent(final SimpleTypeReference node) {
    return this._html2ADocConverter.passThenMonospace(this._html2ADocConverter.transformHTML(node.getTypeName()));
  }
  
  private CharSequence _processContent(final VariableReference node) {
    return this._html2ADocConverter.passThenMonospace(this._html2ADocConverter.transformHTML(node.getVariableName()));
  }
  
  private CharSequence _processContent(final InlineTag node) {
    String _switchResult = null;
    String _title = node.getTitle().getTitle();
    boolean _matched = false;
    String _title_1 = N4JSDocletParser.TAG_CODE.getTitle();
    if (Objects.equal(_title, _title_1)) {
      _matched=true;
      _switchResult = this._html2ADocConverter.passThenMonospace(this._html2ADocConverter.transformHTML(N4JSDocletParser.TAG_CODE.getValue(node, "")));
    }
    if (!_matched) {
      String _title_2 = N4JSDocletParser.TAG_LINK.getTitle();
      if (Objects.equal(_title, _title_2)) {
        _matched=true;
        _switchResult = this._html2ADocConverter.passThenMonospace(this._html2ADocConverter.transformHTML(N4JSDocletParser.TAG_LINK.getValue(node, "")));
      }
    }
    if (!_matched) {
      {
        final StringBuilder strb = new StringBuilder();
        final Consumer<TagValue> _function = (TagValue it) -> {
          this.appendContents(strb, it);
        };
        node.getValues().forEach(_function);
        return strb;
      }
    }
    return _switchResult;
  }
  
  private StringBuilder appendSpecElementPre(final StringBuilder strb, final SpecIdentifiableElementSection spec) {
    return this.appendElementCodePre(strb, spec.getIdentifiableElement(), spec);
  }
  
  /**
   * E.g. classes
   */
  private StringBuilder _appendElementCodePre(final StringBuilder strb, final IdentifiableElement element, final SpecIdentifiableElementSection spec) {
    boolean _hasTodo = this.hasTodo(spec.getDoclet());
    if (_hasTodo) {
      strb.append(this.getTodoLink(spec.getDoclet()));
    }
    return strb;
  }
  
  private StringBuilder _appendElementCodePre(final StringBuilder strb, final TMember element, final SpecIdentifiableElementSection spec) {
    return this.appendMemberOrVarOrFuncPre(strb, 
      this.validatorMessageHelper.shortDescription(element), 
      this.validatorMessageHelper.shortQualifiedName(element), 
      element.getMemberAsString(), element, spec);
  }
  
  private StringBuilder _appendElementCodePre(final StringBuilder strb, final TMethod element, final SpecIdentifiableElementSection spec) {
    return this.appendMemberOrVarOrFuncPre(strb, 
      this.validatorMessageHelper.shortDescription(((TMember) element)), 
      this.validatorMessageHelper.shortQualifiedName(((TMember) element)), 
      element.getMemberAsString(), element, spec);
  }
  
  private StringBuilder _appendElementCodePre(final StringBuilder strb, final TFunction element, final SpecIdentifiableElementSection spec) {
    return this.appendMemberOrVarOrFuncPre(strb, 
      this.validatorMessageHelper.shortDescription(element), 
      this.validatorMessageHelper.shortQualifiedName(element), 
      element.getName(), element, spec);
  }
  
  private StringBuilder _appendElementCodePre(final StringBuilder strb, final TVariable element, final SpecIdentifiableElementSection spec) {
    return this.appendMemberOrVarOrFuncPre(strb, 
      this.validatorMessageHelper.shortDescription(element), 
      this.validatorMessageHelper.shortQualifiedName(element), 
      element.getName(), element, spec);
  }
  
  private StringBuilder appendMemberOrVarOrFuncPre(final StringBuilder strb, final String shortDescr, final String shortQN, final String reqName, final SyntaxRelatedTElement element, final SpecIdentifiableElementSection spec) {
    final boolean isIntegratedFromPolyfill = (!Objects.equal(spec.sourceEntry.trueFolder, spec.sourceEntry.folder));
    final String trueSrcFolder = ((spec.sourceEntry.repository + ":") + spec.sourceEntry.trueFolder);
    StringConcatenation _builder = new StringConcatenation();
    _builder.newLine();
    _builder.append("[[gsec:spec_");
    String _adocCompatibleAnchorID = spec.sourceEntry.getAdocCompatibleAnchorID();
    _builder.append(_adocCompatibleAnchorID);
    _builder.append("]]");
    _builder.newLineIfNotEmpty();
    _builder.append("[role=memberdoc]");
    _builder.newLine();
    _builder.append("=== ");
    String _pass = this._html2ADocConverter.pass(StringExtensions.toFirstUpper(shortDescr));
    _builder.append(_pass);
    _builder.newLineIfNotEmpty();
    {
      boolean _hasTodo = this.hasTodo(spec.getDoclet());
      if (_hasTodo) {
        String _todoLink = this.getTodoLink(spec.getDoclet());
        _builder.append(_todoLink);
        _builder.newLineIfNotEmpty();
      }
    }
    {
      if (isIntegratedFromPolyfill) {
        _builder.newLine();
        _builder.append("[.small]#(Integrated from static polyfill aware class in: ");
        _builder.append(trueSrcFolder);
        _builder.append(")#");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append("==== Signature");
    _builder.newLine();
    _builder.newLine();
    CharSequence _codeLink = this.codeLink(element);
    _builder.append(_codeLink);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    strb.append(_builder);
    return strb;
  }
  
  private CharSequence _codeLink(final TMember member) {
    return this.doCodeLink(member, this.fullSignature(member));
  }
  
  private CharSequence _codeLink(final TMethod method) {
    return this.doCodeLink(method, this.fullSignature(method));
  }
  
  private CharSequence _codeLink(final TFunction func) {
    return this.doCodeLink(func, this.fullSignature(func));
  }
  
  private CharSequence _codeLink(final TVariable tvar) {
    return this.doCodeLink(tvar, this.fullSignature(tvar));
  }
  
  private String fullSignature(final TMember member) {
    final StringBuilder strb = new StringBuilder();
    final Function1<TAnnotation, Boolean> _function = (TAnnotation it) -> {
      String _name = it.getName();
      return Boolean.valueOf((!Objects.equal(_name, AnnotationDefinition.INTERNAL.name)));
    };
    Iterable<TAnnotation> _filter = IterableExtensions.<TAnnotation>filter(member.getAnnotations(), _function);
    for (final TAnnotation a : _filter) {
      String _annotationAsString = a.getAnnotationAsString();
      String _plus = (_annotationAsString + " ");
      strb.append(_plus);
    }
    String _keyword = this.keywordProvider.keyword(member.getMemberAccessModifier());
    String _plus_1 = (_keyword + " ");
    strb.append(_plus_1);
    boolean _isAbstract = member.isAbstract();
    if (_isAbstract) {
      strb.append("@abstract ");
    }
    strb.append(member.getMemberAsString());
    return strb.toString();
  }
  
  private String fullSignature(final TMethod method) {
    return this.validatorMessageHelper.fullFunctionSignature(method);
  }
  
  private String fullSignature(final TFunction func) {
    return this.validatorMessageHelper.fullFunctionSignature(func);
  }
  
  private String fullSignature(final TVariable tvar) {
    TypeRef _typeRef = tvar.getTypeRef();
    boolean _tripleEquals = (_typeRef == null);
    if (_tripleEquals) {
      return tvar.getName();
    }
    StringConcatenation _builder = new StringConcatenation();
    String _name = tvar.getName();
    _builder.append(_name);
    _builder.append(": ");
    String _typeRefAsString = tvar.getTypeRef().getTypeRefAsString();
    _builder.append(_typeRefAsString);
    return _builder.toString();
  }
  
  private CharSequence doCodeLink(final IdentifiableElement element, final String signature) {
    final RepoRelativePath rrp = this.repoPathHolder.get(element);
    final StringBuilder strb = new StringBuilder();
    if ((rrp != null)) {
      final SourceEntry se = SourceEntryFactory.create(this.repoPathHolder, rrp, element);
      this.appendSourceLink(strb, se, this._html2ADocConverter.passThenMonospace(signature));
    }
    return strb.toString();
  }
  
  private StringBuilder appendSpecElementPost(final StringBuilder strb, final SpecIdentifiableElementSection spec, final Map<String, SpecSection> map) {
    return this.appendElementPost(strb, spec.getIdentifiableElement(), spec, map);
  }
  
  private StringBuilder _appendElementPost(final StringBuilder strb, final IdentifiableElement element, final SpecIdentifiableElementSection specRegion, final Map<String, SpecSection> specsByKey) {
    if ((element instanceof ContainerType<?>)) {
      Map<TMember, SortedSet<SpecTestInfo>> testsForInherited = specRegion.getTestInfosForInheritedMember();
      if (((testsForInherited == null) || testsForInherited.isEmpty())) {
        return strb;
      }
      final String typeName = ((ContainerType<?>)element).getName();
      final List<TClassifier> superTypes = AllSuperTypesCollector.collect(((ContainerType<?>)element));
      final Function1<Map.Entry<TMember, SortedSet<SpecTestInfo>>, Boolean> _function = (Map.Entry<TMember, SortedSet<SpecTestInfo>> it) -> {
        return Boolean.valueOf(((it.getValue() != null) && (!it.getValue().isEmpty())));
      };
      final Iterable<Map.Entry<TMember, SortedSet<SpecTestInfo>>> tests = IterableExtensions.<Map.Entry<TMember, SortedSet<SpecTestInfo>>>filter(testsForInherited.entrySet(), _function);
      final Function1<Map.Entry<TMember, SortedSet<SpecTestInfo>>, String> _function_1 = (Map.Entry<TMember, SortedSet<SpecTestInfo>> it) -> {
        return this.getFormattedID(it, superTypes);
      };
      final List<Map.Entry<TMember, SortedSet<SpecTestInfo>>> sortedTests = IterableExtensions.<Map.Entry<TMember, SortedSet<SpecTestInfo>>, String>sortBy(tests, _function_1);
      for (final Map.Entry<TMember, SortedSet<SpecTestInfo>> tmemberSpecs : sortedTests) {
        {
          final String shortDescr = this.validatorMessageHelper.shortDescription(tmemberSpecs.getKey());
          String _shortQualifiedName = this.validatorMessageHelper.shortQualifiedName(tmemberSpecs.getKey());
          final String secSpecLink = ((typeName + ".") + _shortQualifiedName);
          final String secSpecLinkEsc = SourceEntry.getEscapedAdocAnchorString(secSpecLink);
          final String description = this.validatorMessageHelper.description(tmemberSpecs.getKey().getContainingType());
          final String shortQualName = this.validatorMessageHelper.shortQualifiedName(tmemberSpecs.getKey());
          StringConcatenation _builder = new StringConcatenation();
          _builder.newLine();
          _builder.append("[[gsec:spec_");
          _builder.append(secSpecLinkEsc);
          _builder.append("]]");
          _builder.newLineIfNotEmpty();
          _builder.append("[role=memberdoc]");
          _builder.newLine();
          _builder.append("=== ");
          String _pass = this._html2ADocConverter.pass(StringExtensions.toFirstUpper(shortDescr));
          _builder.append(_pass);
          _builder.newLineIfNotEmpty();
          _builder.newLine();
          _builder.append("Inherited from");
          _builder.newLine();
          String _pass_1 = this._html2ADocConverter.pass(description);
          _builder.append(_pass_1);
          _builder.newLineIfNotEmpty();
          {
            boolean _isInSpec = this.isInSpec(tmemberSpecs.getKey(), specsByKey);
            if (_isInSpec) {
              _builder.append("<<gsec:spec_");
              String _pass_2 = this._html2ADocConverter.pass(shortQualName);
              _builder.append(_pass_2);
              _builder.append(">>");
              _builder.newLineIfNotEmpty();
            }
          }
          _builder.newLine();
          strb.append(_builder);
          this.appendConstraints(strb, tmemberSpecs.getKey(), specRegion, tmemberSpecs.getValue(), false);
        }
      }
    }
    return strb;
  }
  
  private boolean isInSpec(final TMember member, final Map<String, SpecSection> specsByKey) {
    if ((member == null)) {
      return false;
    }
    return specsByKey.containsKey(KeyUtils.getSpecKey(this.repoPathHolder, member));
  }
  
  private String getFormattedID(final Map.Entry<TMember, SortedSet<SpecTestInfo>> entry, final List<TClassifier> superTypes) {
    final int index = superTypes.indexOf(entry.getKey().getContainingType());
    String _format = String.format("%04d", Integer.valueOf(index));
    String _name = entry.getKey().getName();
    return (_format + _name);
  }
  
  private StringBuilder _appendElementPost(final StringBuilder strb, final TMember element, final SpecIdentifiableElementSection specRegion, final Map<String, SpecSection> specsByKey) {
    SortedSet<SpecTestInfo> _testInfosForMember = specRegion.getTestInfosForMember();
    boolean _hasReqId = this.hasReqId(specRegion.getDoclet());
    boolean _not = (!_hasReqId);
    this.appendConstraints(strb, element, specRegion, _testInfosForMember, _not);
    return strb;
  }
  
  private StringBuilder _appendElementPost(final StringBuilder strb, final TMethod element, final SpecIdentifiableElementSection specRegion, final Map<String, SpecSection> specsByKey) {
    SortedSet<SpecTestInfo> _testInfosForMember = specRegion.getTestInfosForMember();
    boolean _hasReqId = this.hasReqId(specRegion.getDoclet());
    boolean _not = (!_hasReqId);
    this.appendConstraints(strb, element, specRegion, _testInfosForMember, _not);
    return strb;
  }
  
  private StringBuilder _appendElementPost(final StringBuilder strb, final TFunction element, final SpecIdentifiableElementSection specRegion, final Map<String, SpecSection> specsByKey) {
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(specRegion.getTestInfosForType());
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      final Function1<SpecTestInfo, String> _function = (SpecTestInfo it) -> {
        return it.testTitle;
      };
      final Map<String, List<SpecTestInfo>> groupdTests = IterableExtensions.<String, SpecTestInfo>groupBy(specRegion.getTestInfosForType(), _function);
      strb.append("==== Semantics\n");
      this.<String>appendApiConstraints(strb, groupdTests);
    } else {
      final String reqID = this.getReqId(specRegion.getDoclet());
      boolean _isEmpty = reqID.isEmpty();
      if (_isEmpty) {
        String _passThenMonospace = this._html2ADocConverter.passThenMonospace(element.getName());
        String _plus = ("Add tests specifying semantics for " + _passThenMonospace);
        String _pass = this._html2ADocConverter.pass(element.getName());
        String _plus_1 = ("test function " + _pass);
        final String todoLink = this.getTodoLink(_plus, _plus_1);
        StringConcatenation _builder = new StringConcatenation();
        _builder.newLine();
        _builder.append("==== Semantics");
        _builder.newLine();
        _builder.append(todoLink);
        _builder.newLineIfNotEmpty();
        strb.append(_builder);
      } else {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("% tests see ");
        _builder_1.append(reqID);
        strb.append(_builder_1);
      }
    }
    return strb;
  }
  
  private StringBuilder _appendElementPost(final StringBuilder strb, final TVariable element, final SpecSection specRegion, final Map<String, SpecSection> specsByKey) {
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(specRegion.getTestInfosForType());
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      final Function1<SpecTestInfo, String> _function = (SpecTestInfo it) -> {
        return it.testTitle;
      };
      final Map<String, List<SpecTestInfo>> groupdTests = IterableExtensions.<String, SpecTestInfo>groupBy(specRegion.getTestInfosForType(), _function);
      strb.append("==== Semantics\n");
      this.<String>appendApiConstraints(strb, groupdTests);
    }
    return strb;
  }
  
  private StringBuilder appendConstraints(final StringBuilder strb, final TMember element, final SpecIdentifiableElementSection specRegion, final Set<SpecTestInfo> specTestInfos, final boolean addTodo) {
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(specTestInfos);
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      final Function1<SpecTestInfo, String> _function = (SpecTestInfo it) -> {
        return it.testTitle;
      };
      final Map<String, List<SpecTestInfo>> groupdTests = IterableExtensions.<String, SpecTestInfo>groupBy(specTestInfos, _function);
      strb.append("==== Semantics\n");
      this.<String>appendApiConstraints(strb, groupdTests);
    } else {
      if (addTodo) {
        boolean _elementMayNeedsTest = this.elementMayNeedsTest(element, specRegion);
        if (_elementMayNeedsTest) {
          String _passThenMonospace = this._html2ADocConverter.passThenMonospace(element.getMemberAsString());
          String _plus = ("Add tests specifying semantics for " + _passThenMonospace);
          String _name = element.getContainingType().getName();
          String _plus_1 = (_name + ".");
          String _name_1 = element.getName();
          String _plus_2 = (_plus_1 + _name_1);
          String _pass = this._html2ADocConverter.pass(_plus_2);
          String _plus_3 = ("test " + _pass);
          final String todoLink = this.getTodoLink(_plus, _plus_3);
          StringConcatenation _builder = new StringConcatenation();
          _builder.newLine();
          _builder.append("==== Semantics");
          _builder.newLine();
          _builder.append(todoLink);
          _builder.newLineIfNotEmpty();
          strb.append(_builder);
        }
      }
    }
    return strb;
  }
  
  private boolean elementMayNeedsTest(final TMember element, final SpecIdentifiableElementSection spec) {
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(spec.getTestInfosForType());
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      return true;
    }
    if (((element instanceof TMethod) || (element instanceof FieldAccessor))) {
      ContainerType<?> _containingType = element.getContainingType();
      if ((_containingType instanceof TInterface)) {
        if ((element instanceof TMemberWithAccessModifier)) {
          boolean _isHasNoBody = ((TMemberWithAccessModifier)element).isHasNoBody();
          return (!_isHasNoBody);
        }
      }
      boolean _isAbstract = element.isAbstract();
      return (!_isAbstract);
    }
    return false;
  }
  
  /**
   * List of tests in apiConstraint macros.
   */
  private <T extends Object> StringBuilder appendApiConstraints(final StringBuilder strb, final Map<T, ? extends Collection<SpecTestInfo>> groupdTests) {
    final Function1<Map.Entry<T, ? extends Collection<SpecTestInfo>>, String> _function = (Map.Entry<T, ? extends Collection<SpecTestInfo>> it) -> {
      return it.getKey().toString();
    };
    List<? extends Map.Entry<T, ? extends Collection<SpecTestInfo>>> _sortBy = IterableExtensions.sortBy(groupdTests.entrySet(), _function);
    for (final Map.Entry<T, ? extends Collection<SpecTestInfo>> group : _sortBy) {
      {
        strb.append("\n");
        strb.append(". *");
        final String key = group.getKey().toString();
        final String keyWithoutPrecedingNumber = this.removePrecedingNumber(key);
        strb.append(this._html2ADocConverter.pass(keyWithoutPrecedingNumber));
        strb.append("* (");
        final Iterator<SpecTestInfo> iter = group.getValue().iterator();
        while (iter.hasNext()) {
          {
            final SpecTestInfo testSpec = iter.next();
            strb.append(this.nfgitTest(testSpec));
            boolean _hasNext = iter.hasNext();
            if (_hasNext) {
              strb.append(", \n");
            }
          }
        }
        strb.append(")\n");
        final Function1<SpecTestInfo, Boolean> _function_1 = (SpecTestInfo it) -> {
          return Boolean.valueOf((it.doclet != null));
        };
        final Function1<SpecTestInfo, Doclet> _function_2 = (SpecTestInfo it) -> {
          return it.doclet;
        };
        final Iterable<Doclet> doclets = IterableExtensions.<SpecTestInfo, Doclet>map(IterableExtensions.<SpecTestInfo>filter(group.getValue(), _function_1), _function_2);
        final StringBuilder strbTmp = new StringBuilder();
        this.appendSpecDescriptions(strbTmp, doclets);
        int _length = strbTmp.length();
        boolean _greaterThan = (_length > 0);
        if (_greaterThan) {
          strb.append("+\n");
          strb.append("[.generatedApiConstraint]\n");
          strb.append("====\n\n");
          strb.append(strbTmp);
          strb.append("\n====\n");
        }
      }
    }
    return strb;
  }
  
  private CharSequence nfgitTest(final SpecTestInfo testInfo) {
    final StringBuilder strb = new StringBuilder();
    if ((testInfo.rrp == null)) {
      CharSequence _testModuleSpec = testInfo.testModuleSpec();
      String _plus = (_testModuleSpec + ".");
      strb.append(this.small(_plus));
      String _testMethodTypeName = testInfo.testMethodTypeName();
      String _plus_1 = (_testMethodTypeName + ".");
      String _testMethodName = testInfo.testMethodName();
      String _plus_2 = (_plus_1 + _testMethodName);
      strb.append(_plus_2);
    } else {
      final SourceEntry pc = SourceEntryFactory.create(testInfo);
      String _xifexpression = null;
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(testInfo.testCase);
      if (_isNullOrEmpty) {
        _xifexpression = "Test";
      } else {
        String _xblockexpression = null;
        {
          String formattedCase = this.removePrecedingNumber(testInfo.testCase);
          boolean _isNullOrEmpty_1 = StringExtensions.isNullOrEmpty(formattedCase);
          if (_isNullOrEmpty_1) {
            formattedCase = testInfo.testCase;
          }
          _xblockexpression = this._html2ADocConverter.pass(formattedCase);
        }
        _xifexpression = _xblockexpression;
      }
      final String strCase = _xifexpression;
      final StringBuilder strbTmp = new StringBuilder();
      this.appendSourceLink(strbTmp, pc, strCase);
      strb.append(this.small(strbTmp));
    }
    return strb.toString();
  }
  
  /**
   * Reminder: Escaping the caption using the method {@link Html2ADocConverter#pass} is recommended.
   */
  private StringBuilder appendSourceLink(final StringBuilder strb, final SourceEntry pc, final String caption) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("srclnk:++");
    String _pQN = pc.toPQN();
    _builder.append(_pQN);
    _builder.append("++[");
    _builder.append(caption);
    _builder.append("]");
    strb.append(_builder);
    return strb;
  }
  
  /**
   * Returns req id, may be an empty string but never null.
   */
  private String getReqId(final Doclet doclet) {
    return N4JSDocletParser.TAG_REQID.getValue(doclet, "");
  }
  
  /**
   * Returns true, if spec contains a reference to a requirement id.
   */
  private boolean hasReqId(final Doclet doclet) {
    boolean _isEmpty = this.getReqId(doclet).isEmpty();
    return (!_isEmpty);
  }
  
  /**
   * Returns true, if spec contains a reference to a todo.
   */
  private boolean hasTodo(final Doclet doclet) {
    boolean _isEmpty = this.getTodo(doclet).isEmpty();
    return (!_isEmpty);
  }
  
  /**
   * Returns todo, may be an empty string but never null.
   */
  private String getTodo(final Doclet doclet) {
    return N4JSDocletParser.TAG_TODO.getValue(doclet, "");
  }
  
  private String getTodoLink(final String todoText, final String sideText) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.newLine();
    _builder.append("[TODO");
    {
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(sideText);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        _builder.append(", title=\"");
        _builder.append(sideText);
        _builder.append("\"");
      }
    }
    _builder.append("]");
    _builder.newLineIfNotEmpty();
    _builder.append("--");
    _builder.newLine();
    _builder.append(todoText);
    _builder.newLineIfNotEmpty();
    _builder.append("--");
    _builder.newLine();
    _builder.newLine();
    final String todo = _builder.toString();
    return todo;
  }
  
  private String getTodoLink(final Doclet doclet) {
    return this.getTodoLink(this.getTodo(doclet), "");
  }
  
  private StringBuilder appendTaskLink(final StringBuilder strb, final String taskID) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("task:");
    _builder.append(taskID);
    _builder.append("[]");
    strb.append(_builder);
    return strb;
  }
  
  private String small(final CharSequence smallString) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("[.small]#");
    _builder.append(smallString);
    _builder.append("#");
    return _builder.toString();
  }
  
  private String removePrecedingNumber(final String key) {
    for (int i = 0; (i < key.length()); i++) {
      {
        final String stringAt = Character.toString(key.charAt(i));
        boolean _contains = "0123456789 ".contains(stringAt);
        boolean _not = (!_contains);
        if (_not) {
          return key.substring(i);
        }
      }
    }
    return "";
  }
  
  private CharSequence processContent(final ContentNode node) {
    if (node instanceof InlineTag) {
      return _processContent((InlineTag)node);
    } else if (node instanceof Literal) {
      return _processContent((Literal)node);
    } else if (node instanceof SimpleTypeReference) {
      return _processContent((SimpleTypeReference)node);
    } else if (node instanceof Text) {
      return _processContent((Text)node);
    } else if (node instanceof VariableReference) {
      return _processContent((VariableReference)node);
    } else if (node != null) {
      return _processContent(node);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(node).toString());
    }
  }
  
  private StringBuilder appendElementCodePre(final StringBuilder strb, final IdentifiableElement element, final SpecIdentifiableElementSection spec) {
    if (element instanceof TMethod) {
      return _appendElementCodePre(strb, (TMethod)element, spec);
    } else if (element instanceof TFunction) {
      return _appendElementCodePre(strb, (TFunction)element, spec);
    } else if (element instanceof TVariable) {
      return _appendElementCodePre(strb, (TVariable)element, spec);
    } else if (element instanceof TMember) {
      return _appendElementCodePre(strb, (TMember)element, spec);
    } else if (element != null) {
      return _appendElementCodePre(strb, element, spec);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(strb, element, spec).toString());
    }
  }
  
  private CharSequence codeLink(final EObject method) {
    if (method instanceof TMethod) {
      return _codeLink((TMethod)method);
    } else if (method instanceof TFunction) {
      return _codeLink((TFunction)method);
    } else if (method instanceof TVariable) {
      return _codeLink((TVariable)method);
    } else if (method instanceof TMember) {
      return _codeLink((TMember)method);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(method).toString());
    }
  }
  
  private StringBuilder appendElementPost(final StringBuilder strb, final IdentifiableElement element, final SpecSection specRegion, final Map<String, SpecSection> specsByKey) {
    if (element instanceof TMethod
         && specRegion instanceof SpecIdentifiableElementSection) {
      return _appendElementPost(strb, (TMethod)element, (SpecIdentifiableElementSection)specRegion, specsByKey);
    } else if (element instanceof TFunction
         && specRegion instanceof SpecIdentifiableElementSection) {
      return _appendElementPost(strb, (TFunction)element, (SpecIdentifiableElementSection)specRegion, specsByKey);
    } else if (element instanceof TVariable
         && specRegion != null) {
      return _appendElementPost(strb, (TVariable)element, specRegion, specsByKey);
    } else if (element instanceof TMember
         && specRegion instanceof SpecIdentifiableElementSection) {
      return _appendElementPost(strb, (TMember)element, (SpecIdentifiableElementSection)specRegion, specsByKey);
    } else if (element != null
         && specRegion instanceof SpecIdentifiableElementSection) {
      return _appendElementPost(strb, element, (SpecIdentifiableElementSection)specRegion, specsByKey);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(strb, element, specRegion, specsByKey).toString());
    }
  }
}
