/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.n4idl.migrations;

import java.util.Iterator;
import java.util.List;
import org.eclipse.n4js.n4idl.migrations.AndSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.ArrayTypeSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.ConstantSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.OrSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.SwitchConditionIterator;
import org.eclipse.n4js.n4idl.migrations.TypeSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.TypeTypeCondition;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * A {@link SwitchCondition} represents a runtime condition that matches a given
 * compile-time {@link TypeRef} at runtime.
 * 
 * See sub-classes for concrete conditions (e.g. type, and, or, etc.).
 */
@SuppressWarnings("all")
public abstract class SwitchCondition implements Iterable<SwitchCondition> {
  /**
   * Returns a new {@link OrSwitchCondition} of the given left-hand side and right-hand side.
   * 
   * @param operands The OR operands. At least 2.
   */
  public static SwitchCondition or(final List<? extends SwitchCondition> operands) {
    int _size = operands.size();
    boolean _lessThan = (_size < 1);
    if (_lessThan) {
      throw new IllegalArgumentException("Cannot create or-switch-condition with less than one operand.");
    } else {
      int _size_1 = operands.size();
      boolean _equals = (_size_1 == 1);
      if (_equals) {
        return operands.get(0);
      }
    }
    SwitchCondition _get = operands.get(0);
    SwitchCondition _get_1 = operands.get(1);
    Iterable<? extends SwitchCondition> _drop = IterableExtensions.drop(operands, 2);
    return new OrSwitchCondition(_get, _get_1, _drop);
  }
  
  /**
   * Returns a new {@link AndSwitchCondition} of the given operands.
   * 
   * @param operands The AND operands. At least 2.
   */
  public static SwitchCondition and(final List<? extends SwitchCondition> operands) {
    int _size = operands.size();
    boolean _lessThan = (_size < 1);
    if (_lessThan) {
      throw new IllegalArgumentException("Cannot create and-switch-condition with less than one operand.");
    } else {
      int _size_1 = operands.size();
      boolean _equals = (_size_1 == 1);
      if (_equals) {
        return operands.get(0);
      }
    }
    SwitchCondition _get = operands.get(0);
    SwitchCondition _get_1 = operands.get(1);
    Iterable<? extends SwitchCondition> _drop = IterableExtensions.drop(operands, 2);
    return new AndSwitchCondition(_get, _get_1, _drop);
  }
  
  /**
   * Returns a new {@link TypeSwitchCondition} for the given type.
   */
  public static TypeSwitchCondition instanceOf(final Type type) {
    return new TypeSwitchCondition(type);
  }
  
  /**
   * Returns a new {@link TypeTypeCondition} for the given type.
   */
  public static TypeTypeCondition type(final Type type) {
    return new TypeTypeCondition(type);
  }
  
  /**
   * Returns a new {@link ArrayTypeSwitchCondition} for the given element type.
   */
  public static ArrayTypeSwitchCondition arrayOf(final SwitchCondition elementTypeCondition) {
    return new ArrayTypeSwitchCondition(elementTypeCondition);
  }
  
  /**
   * Returns a constant switch condition, which will always evaluate to {@code true}.
   */
  public static ConstantSwitchCondition trueCondition() {
    return ConstantSwitchCondition.TRUE;
  }
  
  /**
   * Returns a constant switch condition, which will always evaluate to {@code false}.
   */
  public static ConstantSwitchCondition falseCondition() {
    return ConstantSwitchCondition.FALSE;
  }
  
  protected static String getTypeDescription(final Type type) {
    String _name = type.getName();
    String _xifexpression = null;
    int _version = type.getVersion();
    boolean _notEquals = (_version != 0);
    if (_notEquals) {
      int _version_1 = type.getVersion();
      _xifexpression = ("#" + Integer.valueOf(_version_1));
    } else {
      _xifexpression = "";
    }
    return (_name + _xifexpression);
  }
  
  /**
   * Returns a string-representation of this condition, testing the given value identifier.
   * 
   * Example: {@code getConditionAsString("v")} return {@code "v instanceof A#1" }
   */
  public abstract String getConditionAsString(final String valueIdentifier);
  
  public abstract Iterable<? extends SwitchCondition> subConditions();
  
  @Override
  public String toString() {
    return this.getConditionAsString("v");
  }
  
  @Override
  public Iterator<SwitchCondition> iterator() {
    return new SwitchConditionIterator(this);
  }
}
