/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.n4idl.scoping;

import java.util.Objects;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4idl.scoping.N4IDLVersionableFilter;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TVersionable;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * An implementation of {@link IScope} that considers versioned objects.
 */
@SuppressWarnings("all")
public class N4IDLVersionAwareScope implements IScope {
  private final IScope delegate;
  
  private final N4IDLVersionableFilter filter;
  
  /**
   * Creates a new instance that filters the elements from the given delegate scope using the given context version.
   * 
   * @param delegate
   *            the delegate to query for elements
   * @param contextVersion
   * 			  the context version to consider
   * @param qualifiedNameComputer
   * 			.. the QualifiedNameComputer implementation to use
   */
  public N4IDLVersionAwareScope(final IScope delegate, final int contextVersion) {
    if ((contextVersion <= 0)) {
      throw new IllegalArgumentException("Context version must be a positive integer");
    }
    this.delegate = Objects.<IScope>requireNonNull(delegate);
    N4IDLVersionableFilter _n4IDLVersionableFilter = new N4IDLVersionableFilter(contextVersion);
    this.filter = _n4IDLVersionableFilter;
  }
  
  @Override
  public Iterable<IEObjectDescription> getAllElements() {
    return this.delegate.getAllElements();
  }
  
  @Override
  public Iterable<IEObjectDescription> getElements(final QualifiedName name) {
    return this.filter.filterElements(this.delegate.getElements(name));
  }
  
  @Override
  public Iterable<IEObjectDescription> getElements(final EObject object) {
    return this.filter.filterElements(this.delegate.getElements(object));
  }
  
  @Override
  public IEObjectDescription getSingleElement(final QualifiedName name) {
    return this.selectElement(this.delegate.getElements(name));
  }
  
  @Override
  public IEObjectDescription getSingleElement(final EObject object) {
    return this.selectElement(this.delegate.getElements(object));
  }
  
  /**
   * Selects the first description of an element that satisfies either of the following conditions:
   * 
   * <ul>
   * <li>The element is not an instance of {@link TClassifier}, i.e., it is not versionable.</li>
   * <li>The element is an instance of {@link TClassifier} and its version is equal to the upper limit of the
   * requested version range.</li>
   * </ul>
   * <p>
   * If no element satisfies these conditions, then the given iterable only contains descriptions for instances of
   * {@link TClassifier}. In that case, the element with the maximal version number that is still contained in the
   * requested version range is selected and its description is returned.
   * </p>
   * <p>
   * If no element is contained in the requested version range, then <code>null</code> is returned.
   * </p>
   */
  private IEObjectDescription selectElement(final Iterable<IEObjectDescription> descriptions) {
    final IEObjectDescription description = IterableExtensions.<IEObjectDescription>head(this.filter.filterElements(descriptions));
    if ((null == description)) {
      return null;
    }
    final EObject element = description.getEObjectOrProxy();
    if ((element instanceof TVersionable)) {
      int _version = ((TVersionable)element).getVersion();
      int _contextVersion = this.filter.getContextVersion();
      boolean _notEquals = (_version != _contextVersion);
      if (_notEquals) {
        return description;
      }
    }
    return description;
  }
  
  @Override
  public String toString() {
    int _contextVersion = this.filter.getContextVersion();
    String _plus = ("N4IDLVersionAwareScope[contextVersion = " + Integer.valueOf(_contextVersion));
    String _plus_1 = (_plus + "] -> ");
    String _string = this.delegate.toString();
    return (_plus_1 + _string);
  }
}
