/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesbuilder;

import com.google.inject.Inject;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesbuilder.N4JSTypesBuilderHelper;
import org.eclipse.xtext.xbase.lib.Extension;

@SuppressWarnings("all")
class N4JSFormalParameterTypesBuilder {
  @Inject
  @Extension
  private N4JSTypesBuilderHelper _n4JSTypesBuilderHelper;
  
  boolean relinkFormalParameter(final FormalParameter astFormalParameter, final TFunction functionType, final boolean preLinkingPhase, final int idx) {
    final TFormalParameter formalParameterType = functionType.getFpars().get(idx);
    this._n4JSTypesBuilderHelper.ensureEqualName(astFormalParameter, formalParameterType);
    formalParameterType.setAstElement(astFormalParameter);
    astFormalParameter.setDefinedTypeElement(formalParameterType);
    return true;
  }
  
  TFormalParameter createFormalParameter(final FormalParameter n4FormalParameter, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    return this.createFormalParameter(n4FormalParameter, null, builtInTypeScope, preLinkingPhase);
  }
  
  /**
   * Creates a TFormalParameter for the given FormalParameter from the AST.
   * 
   * @param defaultTypeRef  will be used in case there is no declared type for the formal parameter;
   *                        this may be <code>null</code> and in this case <code>any</code> will be
   *                        the formal parameter's actual type.
   */
  TFormalParameter createFormalParameter(final FormalParameter astFormalParameter, final TypeRef defaultTypeRef, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    final TFormalParameter formalParameterType = TypesFactory.eINSTANCE.createTFormalParameter();
    formalParameterType.setName(astFormalParameter.getName());
    formalParameterType.setVariadic(astFormalParameter.isVariadic());
    formalParameterType.setAstInitializer(null);
    formalParameterType.setHasInitializerAssignment(astFormalParameter.isHasInitializerAssignment());
    this.setFormalParameterType(formalParameterType, astFormalParameter, defaultTypeRef, builtInTypeScope, preLinkingPhase);
    this._n4JSTypesBuilderHelper.copyAnnotations(formalParameterType, astFormalParameter, preLinkingPhase);
    formalParameterType.setAstElement(astFormalParameter);
    astFormalParameter.setDefinedTypeElement(formalParameterType);
    return formalParameterType;
  }
  
  /**
   * @param formalParameterType the type system related parameter type to be set
   * @param astFormalParameter the AST related parameter which is to be copied to the former
   */
  private void setFormalParameterType(final TFormalParameter formalParameterType, final FormalParameter astFormalParameter, final TypeRef defaultTypeRef, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    if ((!preLinkingPhase)) {
      TypeRef _elvis = null;
      TypeRef _copyWithProxies = TypeUtils.<TypeRef>copyWithProxies(astFormalParameter.getDeclaredTypeRef());
      if (_copyWithProxies != null) {
        _elvis = _copyWithProxies;
      } else {
        TypeRef _defaultParameterType = this.getDefaultParameterType(defaultTypeRef, astFormalParameter, builtInTypeScope);
        _elvis = _defaultParameterType;
      }
      formalParameterType.setTypeRef(_elvis);
    }
  }
  
  private TypeRef getDefaultParameterType(final TypeRef defaultTypeRef, final FormalParameter astFormalParameter, final BuiltInTypeScope builtInTypeScope) {
    TypeRef _xifexpression = null;
    Expression _initializer = astFormalParameter.getInitializer();
    boolean _tripleNotEquals = (_initializer != null);
    if (_tripleNotEquals) {
      _xifexpression = TypeUtils.createDeferredTypeRef();
    } else {
      TypeRef _xifexpression_1 = null;
      if ((defaultTypeRef == null)) {
        _xifexpression_1 = builtInTypeScope.getAnyTypeRef();
      } else {
        _xifexpression_1 = TypeUtils.<TypeRef>copy(defaultTypeRef);
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
}
