/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.BreakStatement;
import org.eclipse.n4js.n4JS.ContinueStatement;
import org.eclipse.n4js.n4JS.DebuggerStatement;
import org.eclipse.n4js.n4JS.DoStatement;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ForStatement;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.GenericDeclaration;
import org.eclipse.n4js.n4JS.IfStatement;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4JSFeatureUtils;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.ReturnStatement;
import org.eclipse.n4js.n4JS.SwitchStatement;
import org.eclipse.n4js.n4JS.ThrowStatement;
import org.eclipse.n4js.n4JS.TryStatement;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.n4JS.WhileStatement;
import org.eclipse.n4js.n4JS.WithStatement;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.services.N4JSGrammarAccess;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.Wildcard;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.MemberType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ts.types.util.Variance;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.Result;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelper;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.UtilN4;
import org.eclipse.n4js.validation.AbstractMessageAdjustingN4JSValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.n4js.validation.ValidatorMessageHelper;
import org.eclipse.xtext.nodemodel.BidiIterable;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.service.OperationCanceledManager;
import org.eclipse.xtext.validation.AbstractDeclarativeValidator;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Common base class for all N4JS validators. Provides some convenience methods and
 * internal validations. The latter are applied to AST nodes of different types and
 * therefore have to be reused in different concrete subclasses (for example,
 * {@link #internalCheckTypeArguments(Type,List,EObject,EStructuralFeature)}
 * is called from N4JSTypeValidator and N4JSExpressionValidator).
 */
@SuppressWarnings("all")
public abstract class AbstractN4JSDeclarativeValidator extends AbstractMessageAdjustingN4JSValidator {
  @Inject
  @Extension
  protected N4JSElementKeywordProvider keywordProvider;
  
  @Inject
  @Extension
  protected ValidatorMessageHelper validatorMessageHelper;
  
  @Inject
  private N4JSGrammarAccess grammarAccess;
  
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  @Inject
  private IN4JSCore n4jsCore;
  
  @Inject
  private OperationCanceledManager operationCanceledManager;
  
  /**
   * @since 2.6
   */
  @Override
  public AbstractMessageAdjustingN4JSValidator.MethodWrapperCancelable createMethodWrapper(final AbstractDeclarativeValidator instanceToUse, final Method method) {
    abstract class __AbstractN4JSDeclarativeValidator_1 extends AbstractMessageAdjustingN4JSValidator.MethodWrapperCancelable {
      __AbstractN4JSDeclarativeValidator_1(final AbstractDeclarativeValidator instance, final Method m) {
        super(instance, m);
      }
      
      /**
       * Reasons to skip invocation of a validation method:
       * <ul>
       * <li>no resource associated to the current object</li>
       * <li>the resource in question shouldn't be validated</li>
       * </ul>
       */
      abstract boolean shouldInvoke(final AbstractDeclarativeValidator.State state);
    }
    
    return new __AbstractN4JSDeclarativeValidator_1(instanceToUse, method) {
      @Override
      public void invoke(final AbstractDeclarativeValidator.State state) {
        AbstractN4JSDeclarativeValidator.this.operationCanceledManager.checkCanceled(this.getCancelIndicator(state));
        boolean _shouldInvoke = this.shouldInvoke(state);
        if (_shouldInvoke) {
          super.invoke(state);
        }
      }
      
      /**
       * Reasons to skip invocation of a validation method:
       * <ul>
       * <li>no resource associated to the current object</li>
       * <li>the resource in question shouldn't be validated</li>
       * </ul>
       */
      boolean shouldInvoke(final AbstractDeclarativeValidator.State state) {
        EObject _currentObject = state.currentObject;
        Resource _eResource = null;
        if (_currentObject!=null) {
          _eResource=_currentObject.eResource();
        }
        boolean _tripleNotEquals = (_eResource != null);
        if (_tripleNotEquals) {
          boolean _isNoValidate = AbstractN4JSDeclarativeValidator.this.n4jsCore.isNoValidate(state.currentObject.eResource().getURI());
          boolean _not = (!_isNoValidate);
          if (_not) {
            return true;
          }
        }
        return false;
      }
      
      @Override
      public void handleInvocationTargetException(final Throwable targetException, final AbstractDeclarativeValidator.State state) {
        super.handleInvocationTargetException(targetException, state);
        if ((targetException instanceof NullPointerException)) {
          Exceptions.sneakyThrow(targetException);
        }
      }
    };
  }
  
  /**
   * Checks for (1) correct number of type arguments, (2) correct type of each type argument, and (3) consistency of
   * use-site and definition-site variance (in case of wildcards).
   * <p>
   * Usually 'parameterizedElement' will be a TClass, TInterface, TMethod or TFunction.
   * However, it might also be a TField, TGetter or TSetter to also support the corner cases
   * of an invalid parameterized property access to a field, getter, setter in this method.
   * It may be <code>null</code> if no type model element is available as the source of the
   * type variables given in 'typeVars' (e.g. in case of a function type expression).
   * 
   * @param typeVars  the type variables the provided type arguments are intended for.
   * @param typeArgs  the type arguments to check.
   * @param allowAutoInference  if true, it will be legal to provide no arguments, even if there are several
   *                            type variables. Intended for cases where inference of type arguments is supported.
   * @param parameterizedElement  the element defining the type variables in 'typeVars' or <code>null</code>
   *                              if not available (e.g. in case of a FunctionTypeExpression). Will be used only
   *                              for error reporting (to show a keyword and name, e.g. "for method myMethod").
   * @param source   the AST node where the parameterized access occurs;
   *                 used to derive the region of the error.
   * @param feature  the feature in 'source' where the parameterized access occurs;
   *                 used to derive the region of the error.
   */
  protected void internalCheckTypeArguments(final List<? extends TypeVariable> typeVars, final List<? extends TypeArgument> typeArgs, final boolean allowAutoInference, final IdentifiableElement parameterizedElement, final EObject source, final EStructuralFeature feature) {
    final int typeParameterCount = typeVars.size();
    final int typeArgumentCount = typeArgs.size();
    if ((allowAutoInference && (typeArgumentCount == 0))) {
      return;
    }
    if ((typeParameterCount != typeArgumentCount)) {
      if (((parameterizedElement != null) && (parameterizedElement.getName() != null))) {
        final String message = IssueCodes.getMessageForEXP_WRONG_NUMBER_OF_TYPEARGS_FOR_ELEMENT(this.keywordProvider.keyword(parameterizedElement), 
          parameterizedElement.getName(), Integer.valueOf(typeParameterCount), Integer.valueOf(typeArgumentCount));
        this.addIssue(message, source, feature, IssueCodes.EXP_WRONG_NUMBER_OF_TYPEARGS_FOR_ELEMENT);
      } else {
        final String message_1 = IssueCodes.getMessageForEXP_WRONG_NUMBER_OF_TYPEARGS(Integer.valueOf(typeParameterCount), Integer.valueOf(typeArgumentCount));
        this.addIssue(message_1, source, feature, IssueCodes.EXP_WRONG_NUMBER_OF_TYPEARGS);
      }
      return;
    }
    final int minTypeVariables = Math.min(typeParameterCount, typeArgumentCount);
    if ((minTypeVariables != 0)) {
      final RuleEnvironment G_subst = RuleEnvironmentExtensions.newRuleEnvironment(source);
      if ((source instanceof ParameterizedPropertyAccessExpression)) {
        final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(source);
        final TypeRef targetTypeRef = this.ts.type(G, ((ParameterizedPropertyAccessExpression)source).getTarget());
        this.tsh.addSubstitutions(G_subst, targetTypeRef);
      }
      int _size = typeArgs.size();
      ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, _size, true);
      for (final int i : _doubleDotLessThan) {
        RuleEnvironmentExtensions.addTypeMapping(G_subst, typeVars.get(i), typeArgs.get(i));
      }
      ExclusiveRange _doubleDotLessThan_1 = new ExclusiveRange(0, minTypeVariables, true);
      for (final int i_1 : _doubleDotLessThan_1) {
        {
          final TypeVariable typeParameter = typeVars.get(i_1);
          final TypeArgument typeArgument = typeArgs.get(i_1);
          if ((typeArgument instanceof Wildcard)) {
            final Variance defSiteVariance = typeParameter.getVariance();
            Variance _xifexpression = null;
            TypeRef _declaredUpperBound = ((Wildcard)typeArgument).getDeclaredUpperBound();
            boolean _tripleNotEquals = (_declaredUpperBound != null);
            if (_tripleNotEquals) {
              _xifexpression = Variance.CO;
            } else {
              Variance _xifexpression_1 = null;
              TypeRef _declaredLowerBound = ((Wildcard)typeArgument).getDeclaredLowerBound();
              boolean _tripleNotEquals_1 = (_declaredLowerBound != null);
              if (_tripleNotEquals_1) {
                _xifexpression_1 = Variance.CONTRA;
              }
              _xifexpression = _xifexpression_1;
            }
            final Variance useSiteVariance = _xifexpression;
            if ((((defSiteVariance != Variance.INV) && (useSiteVariance != null)) && (useSiteVariance != defSiteVariance))) {
              boolean _isUsingInOutNotation = ((Wildcard)typeArgument).isUsingInOutNotation();
              if (_isUsingInOutNotation) {
                final String message_2 = IssueCodes.getMessageForEXP_INCONSISTENT_VARIANCE_OF_TYPE_ARG_IN_OUT(
                  useSiteVariance.getDescriptiveStringNoun(true), 
                  defSiteVariance.getDescriptiveStringNoun(true));
                this.addIssue(message_2, typeArgument, IssueCodes.EXP_INCONSISTENT_VARIANCE_OF_TYPE_ARG_IN_OUT);
              } else {
                String _xifexpression_2 = null;
                if ((useSiteVariance == Variance.CO)) {
                  _xifexpression_2 = "upper";
                } else {
                  _xifexpression_2 = "lower";
                }
                final String message_3 = IssueCodes.getMessageForEXP_INCONSISTENT_VARIANCE_OF_TYPE_ARG(_xifexpression_2, 
                  defSiteVariance.getDescriptiveString(true));
                this.addIssue(message_3, typeArgument, IssueCodes.EXP_INCONSISTENT_VARIANCE_OF_TYPE_ARG);
              }
            }
          }
          final boolean isExceptionCase = TypeUtils.isVoid(typeArgument);
          if ((!isExceptionCase)) {
            TypeRef _elvis = null;
            TypeRef _declaredUpperBound_1 = typeParameter.getDeclaredUpperBound();
            if (_declaredUpperBound_1 != null) {
              _elvis = _declaredUpperBound_1;
            } else {
              TypeRef _typeVariableImplicitUpperBound = N4JSLanguageUtils.getTypeVariableImplicitUpperBound(G_subst);
              _elvis = _typeVariableImplicitUpperBound;
            }
            final TypeRef upperBound = _elvis;
            final TypeRef substituted = this.ts.substTypeVariables(G_subst, upperBound);
            final Result result = this.ts.subtype(G_subst, typeArgument, substituted);
            boolean _isFailure = result.isFailure();
            if (_isFailure) {
              this.createTypeError(result, typeArgument);
            }
          }
        }
      }
    }
  }
  
  /**
   * Check whether every type parameter of a generic function or method declaration is actually used by that
   * declaration.
   * 
   * @param genericFunctionDeclaration the generic function or method declaration to check.
   */
  protected <T extends GenericDeclaration & FunctionDefinition> void internalCheckNoUnusedTypeParameters(final T genericFunctionOrMethod) {
    Type _definedType = genericFunctionOrMethod.getDefinedType();
    boolean _tripleEquals = (_definedType == null);
    if (_tripleEquals) {
      return;
    }
    Type _definedType_1 = genericFunctionOrMethod.getDefinedType();
    final TFunction functionType = ((TFunction) _definedType_1);
    this.internalCheckNoUnusedTypeParameters(genericFunctionOrMethod, genericFunctionOrMethod.getTypeVars(), functionType.getTypeVars());
  }
  
  /**
   * Check whether every type parameter of a generic function type expression is actually used by the
   * declared function type.
   * 
   * @param functionTypeExp the generic function type to check.
   */
  protected void internalCheckNoUnusedTypeParameters(final FunctionTypeExpression functionTypeExp) {
    TFunction _declaredType = functionTypeExp.getDeclaredType();
    boolean _tripleEquals = (_declaredType == null);
    if (_tripleEquals) {
      return;
    }
    final TFunction declaredType = functionTypeExp.getDeclaredType();
    this.internalCheckNoUnusedTypeParameters(functionTypeExp, functionTypeExp.getOwnedTypeVars(), declaredType.getTypeVars());
  }
  
  /**
   * Check whether every type variable that is present in two lists is references anywhere in a subtree.
   * Thereby the first list contains the AST nodes for the type variables whereas the second list contains
   * the corresponding elements from the type model. The AST nodes are required for attaching the generated
   * issues to the correct nodes, whereas the type model elements are needed for the actual usage check.
   * 
   * We assume that each pair of an actual type variable (AST node) and a declared type variable (type model
   * element) that share an index in both lists have the same name. If the lists are not of equal length,
   * then this assumption only applies to the elements up to, but not including, the length of the shorter
   * list. In other words, we assume that one of the two given lists is a prefix of the other (given that
   * we consider actual and declared type variables to be equal if they have the same name), and we only
   * check this prefix.
   * 
   * Finally we add an issue for each type variable that is found to be unreferenced.
   * 
   * @param root the root of the subtree to search for references to the given type variables
   * @param actualTypeVars the actual type variables from the AST
   * @param declaredTypeVars the declared type variables from the type model
   */
  private void internalCheckNoUnusedTypeParameters(final EObject root, final EList<TypeVariable> actualTypeVars, final EList<TypeVariable> declaredTypeVars) {
    final int typeVarCount = Math.min(actualTypeVars.size(), declaredTypeVars.size());
    if ((typeVarCount == 1)) {
      final TypeVariable actualTypeVar = actualTypeVars.get(0);
      final TypeVariable declaredTypeVar = declaredTypeVars.get(0);
      boolean _isOrContainsRefToTypeVar = TypeUtils.isOrContainsRefToTypeVar(root, declaredTypeVar);
      boolean _not = (!_isOrContainsRefToTypeVar);
      if (_not) {
        this.addIssue(IssueCodes.getMessageForFUN_UNUSED_GENERIC_TYPE_PARAM(actualTypeVar.getName()), actualTypeVar, TypesPackage.Literals.IDENTIFIABLE_ELEMENT__NAME, IssueCodes.FUN_UNUSED_GENERIC_TYPE_PARAM);
      }
    } else {
      if ((typeVarCount > 1)) {
        final Set<TypeVariable> referencedTypeVars = TypeUtils.getReferencedTypeVars(root);
        for (int i = 0; (i < typeVarCount); i++) {
          {
            final TypeVariable actualTypeVar_1 = actualTypeVars.get(i);
            final TypeVariable declaredTypeVar_1 = declaredTypeVars.get(i);
            boolean _contains = referencedTypeVars.contains(declaredTypeVar_1);
            boolean _not_1 = (!_contains);
            if (_not_1) {
              this.addIssue(IssueCodes.getMessageForFUN_UNUSED_GENERIC_TYPE_PARAM(actualTypeVar_1.getName()), actualTypeVar_1, TypesPackage.Literals.IDENTIFIABLE_ELEMENT__NAME, IssueCodes.FUN_UNUSED_GENERIC_TYPE_PARAM);
            }
          }
        }
      }
    }
  }
  
  /**
   * Finds first inheritance cycle in the inheritance hierarchy above 'classifier'. Consequential errors are
   * omitted, that is, if a class X extends A, and if A extends B and B extends A, then no error is reported
   * for X as it is not part of the cycle itself.
   * 
   * @return a non-empty list of TClassifiers forming a cyclic path in the inheritance hierarchy above
   *         'classifier' or <code>null</code> if no such cycle exists. Never returns a list with a length
   *         below 2.
   */
  protected List<TClassifier> findCyclicInheritance(final TClassifier classifier) {
    final Function<TClassifier, Iterable<TClassifier>> _function = (TClassifier it) -> {
      Iterable<TClassifier> _xifexpression = null;
      if ((it instanceof TClass)) {
        List<TClassifier> _xifexpression_1 = null;
        ParameterizedTypeRef _superClassRef = ((TClass)it).getSuperClassRef();
        Type _declaredType = null;
        if (_superClassRef!=null) {
          _declaredType=_superClassRef.getDeclaredType();
        }
        if ((_declaredType instanceof TClass)) {
          Type _declaredType_1 = ((TClass)it).getSuperClassRef().getDeclaredType();
          _xifexpression_1 = Collections.<TClassifier>unmodifiableList(CollectionLiterals.<TClassifier>newArrayList(((TClassifier) _declaredType_1)));
        } else {
          _xifexpression_1 = Collections.<TClassifier>unmodifiableList(CollectionLiterals.<TClassifier>newArrayList());
        }
        _xifexpression = _xifexpression_1;
      } else {
        final Function1<ParameterizedTypeRef, Type> _function_1 = (ParameterizedTypeRef it_1) -> {
          return it_1.getDeclaredType();
        };
        _xifexpression = Iterables.<TClassifier>filter(IterableExtensions.<ParameterizedTypeRef, Type>map(it.getSuperClassifierRefs(), _function_1), TClassifier.class);
      }
      return _xifexpression;
    };
    return UtilN4.<TClassifier>findCycleInDirectedGraph(classifier, _function);
  }
  
  /**
   * Convenience method, returns true if given member is a constructor (method).
   */
  public static boolean isConstructor(final TMember tMember) {
    boolean _xifexpression = false;
    if ((tMember instanceof TMethod)) {
      _xifexpression = ((TMethod)tMember).isConstructor();
    } else {
      _xifexpression = false;
    }
    return _xifexpression;
  }
  
  public boolean isAbstract(final TClassifier classifier) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (classifier instanceof TClass) {
      _matched=true;
      _switchResult = ((TClass)classifier).isAbstract();
    }
    if (!_matched) {
      if (classifier instanceof TInterface) {
        _matched=true;
        _switchResult = true;
      }
    }
    if (!_matched) {
      _switchResult = false;
    }
    return _switchResult;
  }
  
  public boolean isDeclaredAbstract(final TMember member) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (member instanceof TMethod) {
      _matched=true;
      _switchResult = ((TMethod)member).isDeclaredAbstract();
    }
    if (!_matched) {
      if (member instanceof TGetter) {
        _matched=true;
        _switchResult = ((TGetter)member).isDeclaredAbstract();
      }
    }
    if (!_matched) {
      if (member instanceof TSetter) {
        _matched=true;
        _switchResult = ((TSetter)member).isDeclaredAbstract();
      }
    }
    if (!_matched) {
      _switchResult = false;
    }
    return _switchResult;
  }
  
  public Block getBody(final N4MemberDeclaration n4MemberDeclaration) {
    Block _switchResult = null;
    boolean _matched = false;
    if (n4MemberDeclaration instanceof N4MethodDeclaration) {
      _matched=true;
      _switchResult = ((N4MethodDeclaration)n4MemberDeclaration).getBody();
    }
    if (!_matched) {
      if (n4MemberDeclaration instanceof N4GetterDeclaration) {
        _matched=true;
        _switchResult = ((N4GetterDeclaration)n4MemberDeclaration).getBody();
      }
    }
    if (!_matched) {
      if (n4MemberDeclaration instanceof N4SetterDeclaration) {
        _matched=true;
        _switchResult = ((N4SetterDeclaration)n4MemberDeclaration).getBody();
      }
    }
    if (!_matched) {
      _switchResult = null;
    }
    return _switchResult;
  }
  
  public boolean hasAnnotation(final AnnotableElement elem, final String annotationName) {
    final Function1<Annotation, Boolean> _function = (Annotation it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equal(_name, annotationName));
    };
    return IterableExtensions.<Annotation>exists(elem.getAnnotations(), _function);
  }
  
  /**
   * Provides default implementations for error, warning and info creation in order to throw exceptions as these methods
   * are not to be called in N4JS. Instead, addIssue is to be used, the severity is retrieved from the message file, see
   * NLSProcessor for details.
   */
  @Override
  public void error(final String message, final EObject source, final EStructuralFeature feature, final int index, final String code, final String... issueData) {
    throw new UnsupportedOperationException("Don\'t use error(...) anymore, but addIssue");
  }
  
  @Override
  public void warning(final String message, final EObject source, final EStructuralFeature feature, final int index, final String code, final String... issueData) {
    throw new UnsupportedOperationException("Don\'t use warning(...) anymore, but addIssue");
  }
  
  @Override
  public void info(final String message, final EObject source, final EStructuralFeature feature, final int index, final String code, final String... issueData) {
    throw new UnsupportedOperationException("Don\'t use info(...) anymore, but addIssue");
  }
  
  /**
   * Finds the closes name feature, either of the object itself, or of a sibling or parent.
   * This is basically used for error markers.
   */
  public Pair<? extends EObject, ? extends EStructuralFeature> findNameFeature(final EObject eo) {
    if ((eo instanceof NamedElement)) {
      final EStructuralFeature attribute = N4JSFeatureUtils.attributeOfNameFeature(((NamedElement)eo));
      if ((attribute != null)) {
        return Pair.<NamedElement, EStructuralFeature>of(((NamedElement)eo), attribute);
      }
    }
    Pair<? extends EObject, ? extends EStructuralFeature> _xifexpression = null;
    if ((eo instanceof ExportDeclaration)) {
      _xifexpression = this.findNameFeature(((ExportDeclaration)eo).getExportedElement());
    } else {
      Pair<? extends EObject, ? extends EStructuralFeature> _xifexpression_1 = null;
      if ((eo instanceof IdentifiableElement)) {
        _xifexpression_1 = Pair.<IdentifiableElement, EAttribute>of(((IdentifiableElement)eo), TypesPackage.Literals.IDENTIFIABLE_ELEMENT__NAME);
      } else {
        Pair<? extends EObject, ? extends EStructuralFeature> _xifexpression_2 = null;
        if ((eo instanceof VariableStatement)) {
          Pair<? extends EObject, ? extends EStructuralFeature> _xblockexpression = null;
          {
            final VariableDeclaration varDecl = IterableExtensions.<VariableDeclaration>head(((VariableStatement)eo).getVarDecl());
            _xblockexpression = this.findNameFeature(varDecl);
          }
          _xifexpression_2 = _xblockexpression;
        } else {
          _xifexpression_2 = null;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    Pair<? extends EObject, ? extends EStructuralFeature> eObjectToNameFeature = _xifexpression;
    return eObjectToNameFeature;
  }
  
  /**
   * Returns range which can be used for error markers, this is usually only a fall back if no
   * name feature can be used.
   */
  public Pair<Integer, Integer> findOffsetAndLength(final EObject eo) {
    Pair<Integer, Integer> _xblockexpression = null;
    {
      final ICompositeNode node = NodeModelUtils.findActualNodeFor(eo);
      INode _xifexpression = null;
      if ((eo instanceof Block)) {
        _xifexpression = this.findChildNode(node, this.grammarAccess.getBlockAccess().getLeftCurlyBracketKeyword_0_0_1());
      } else {
        INode _xifexpression_1 = null;
        if ((eo instanceof IfStatement)) {
          _xifexpression_1 = this.findChildNode(node, this.grammarAccess.getIfStatementAccess().getIfKeyword_0());
        } else {
          INode _xifexpression_2 = null;
          if ((eo instanceof DoStatement)) {
            _xifexpression_2 = this.findChildNode(node, this.grammarAccess.getDoStatementAccess().getDoKeyword_0());
          } else {
            INode _xifexpression_3 = null;
            if ((eo instanceof WhileStatement)) {
              _xifexpression_3 = this.findChildNode(node, this.grammarAccess.getWhileStatementAccess().getWhileKeyword_0());
            } else {
              INode _xifexpression_4 = null;
              if ((eo instanceof ForStatement)) {
                _xifexpression_4 = this.findChildNode(node, this.grammarAccess.getForStatementAccess().getForKeyword_1());
              } else {
                INode _xifexpression_5 = null;
                if ((eo instanceof ContinueStatement)) {
                  _xifexpression_5 = this.findChildNode(node, this.grammarAccess.getContinueStatementAccess().getContinueKeyword_1());
                } else {
                  INode _xifexpression_6 = null;
                  if ((eo instanceof BreakStatement)) {
                    _xifexpression_6 = this.findChildNode(node, this.grammarAccess.getBreakStatementAccess().getBreakKeyword_1());
                  } else {
                    INode _xifexpression_7 = null;
                    if ((eo instanceof ReturnStatement)) {
                      _xifexpression_7 = this.findChildNode(node, this.grammarAccess.getReturnStatementAccess().getReturnKeyword_1());
                    } else {
                      INode _xifexpression_8 = null;
                      if ((eo instanceof WithStatement)) {
                        _xifexpression_8 = this.findChildNode(node, this.grammarAccess.getWithStatementAccess().getWithKeyword_0());
                      } else {
                        INode _xifexpression_9 = null;
                        if ((eo instanceof SwitchStatement)) {
                          _xifexpression_9 = this.findChildNode(node, this.grammarAccess.getSwitchStatementAccess().getSwitchKeyword_0());
                        } else {
                          INode _xifexpression_10 = null;
                          if ((eo instanceof ThrowStatement)) {
                            _xifexpression_10 = this.findChildNode(node, this.grammarAccess.getThrowStatementAccess().getThrowKeyword_0());
                          } else {
                            INode _xifexpression_11 = null;
                            if ((eo instanceof TryStatement)) {
                              _xifexpression_11 = this.findChildNode(node, this.grammarAccess.getTryStatementAccess().getTryKeyword_0());
                            } else {
                              INode _xifexpression_12 = null;
                              if ((eo instanceof DebuggerStatement)) {
                                _xifexpression_12 = this.findChildNode(node, this.grammarAccess.getDebuggerStatementAccess().getDebuggerKeyword_1());
                              }
                              _xifexpression_11 = _xifexpression_12;
                            }
                            _xifexpression_10 = _xifexpression_11;
                          }
                          _xifexpression_9 = _xifexpression_10;
                        }
                        _xifexpression_8 = _xifexpression_9;
                      }
                      _xifexpression_7 = _xifexpression_8;
                    }
                    _xifexpression_6 = _xifexpression_7;
                  }
                  _xifexpression_5 = _xifexpression_6;
                }
                _xifexpression_4 = _xifexpression_5;
              }
              _xifexpression_3 = _xifexpression_4;
            }
            _xifexpression_2 = _xifexpression_3;
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      final INode childNode = _xifexpression;
      Pair<Integer, Integer> _xifexpression_13 = null;
      if ((childNode != null)) {
        int _offset = childNode.getOffset();
        int _length = childNode.getLength();
        _xifexpression_13 = Pair.<Integer, Integer>of(Integer.valueOf(_offset), Integer.valueOf(_length));
      } else {
        int _offset_1 = node.getOffset();
        int _length_1 = node.getLength();
        _xifexpression_13 = Pair.<Integer, Integer>of(Integer.valueOf(_offset_1), Integer.valueOf(_length_1));
      }
      final Pair<Integer, Integer> offsetAndLength = _xifexpression_13;
      _xblockexpression = offsetAndLength;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the offset and length for a list feature.
   * 
   * That is the offset of the first list element and the length
   * from the first feature node to the last feature node (cf. {@link NodeModelUtils#findNodeForFeature}).
   */
  public Pair<Integer, Integer> findListFeatureOffsetAndLength(final EObject obj, final EStructuralFeature feature) {
    final List<INode> nodes = NodeModelUtils.findNodesForFeature(obj, feature);
    boolean _isEmpty = nodes.isEmpty();
    if (_isEmpty) {
      return Pair.<Integer, Integer>of(Integer.valueOf(0), Integer.valueOf(0));
    }
    final int start = IterableExtensions.<INode>head(nodes).getOffset();
    int _offset = IterableExtensions.<INode>last(nodes).getOffset();
    int _length = IterableExtensions.<INode>last(nodes).getLength();
    final int end = (_offset + _length);
    return Pair.<Integer, Integer>of(Integer.valueOf(start), Integer.valueOf((end - start)));
  }
  
  /**
   * Adds an issues, just as {@link #addIssue} but marks all nodes associated with the multi-value feature {@code feature}
   * as an error.
   * 
   * For multi-value features (such as list attributes), this results in all list elements to be marked as error.
   */
  protected void addIssueToMultiValueFeature(final String message, final EObject source, final EStructuralFeature feature, final String issueCode, final String... issueData) {
    final Pair<Integer, Integer> offsetAndLength = this.findListFeatureOffsetAndLength(source, feature);
    this.addIssue(message, source, (offsetAndLength.getKey()).intValue(), (offsetAndLength.getValue()).intValue(), issueCode, issueData);
  }
  
  private INode findChildNode(final ICompositeNode compositeNode, final EObject grammarElement) {
    BidiIterable<INode> _children = compositeNode.getChildren();
    for (final INode childNode : _children) {
      {
        EObject _grammarElement = childNode.getGrammarElement();
        boolean _equals = Objects.equal(_grammarElement, grammarElement);
        if (_equals) {
          return childNode;
        }
        if ((childNode instanceof ICompositeNode)) {
          final INode foundNode = this.findChildNode(((ICompositeNode)childNode), grammarElement);
          if ((foundNode != null)) {
            return foundNode;
          }
        }
      }
    }
    return null;
  }
  
  /**
   * Assumes that members have same name.
   */
  protected boolean isFieldAccessorPair(final TMember member, final TMember member2) {
    final MemberType mt1 = member.getMemberType();
    final MemberType mt2 = member2.getMemberType();
    return (((mt1 == MemberType.GETTER) && (mt2 == MemberType.SETTER)) || ((mt1 == MemberType.SETTER) && (mt2 == MemberType.GETTER)));
  }
  
  /**
   * Assumes that members have same name.
   */
  protected boolean isFieldAccessorPair(final Iterable<TMember> members) {
    final Iterator<TMember> iter = members.iterator();
    boolean _hasNext = iter.hasNext();
    boolean _not = (!_hasNext);
    if (_not) {
      return false;
    }
    final TMember m1 = iter.next();
    boolean _hasNext_1 = iter.hasNext();
    boolean _not_1 = (!_hasNext_1);
    if (_not_1) {
      return false;
    }
    final TMember m2 = iter.next();
    boolean _hasNext_2 = iter.hasNext();
    if (_hasNext_2) {
      return false;
    }
    return this.isFieldAccessorPair(m1, m2);
  }
}
