/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.common.base.Optional;
import com.google.inject.Inject;
import java.math.BigDecimal;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4TypeDeclaration;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.ScriptElement;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.n4JS.VariableStatementKeyword;
import org.eclipse.n4js.n4JS.VersionedElement;
import org.eclipse.n4js.n4idl.versioning.VersionHelper;
import org.eclipse.n4js.n4idl.versioning.VersionUtils;
import org.eclipse.n4js.ts.typeRefs.TypeRefsPackage;
import org.eclipse.n4js.ts.typeRefs.VersionedReference;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariant;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;

/**
 * Validate the use of versions (versioned types, type version requests) in N4IDL.
 */
@SuppressWarnings("all")
public class N4IDLValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  private VersionHelper versionHelper;
  
  @Inject
  private JavaScriptVariantHelper variantHelper;
  
  @Inject
  private N4JSElementKeywordProvider elementKeywordProvider;
  
  /**
   * NEEDED
   * 
   * When removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator.
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  /**
   * Check that the requested version does not exceed the maximal version of the current context.
   */
  @Check
  public void checkVersion(final VersionedReference ref) {
    if ((ref == null)) {
      return;
    }
    final EObject context = ref.eContainer();
    if ((context != null)) {
      int _xifexpression = (int) 0;
      BigDecimal _requestedVersion = ref.getRequestedVersion();
      boolean _tripleEquals = (_requestedVersion == null);
      if (_tripleEquals) {
        _xifexpression = 0;
      } else {
        _xifexpression = ref.getRequestedVersion().intValue();
      }
      final int requestedVersion = _xifexpression;
      final Optional<Integer> maxVersion = this.versionHelper.computeMaximumVersion(context);
      boolean _isPresent = maxVersion.isPresent();
      boolean _not = (!_isPresent);
      if (_not) {
        return;
      }
      Integer _get = maxVersion.get();
      boolean _greaterThan = (requestedVersion > (_get).intValue());
      if (_greaterThan) {
        final String message = IssueCodes.getMessageForIDL_INVALID_VERSION(Integer.valueOf(requestedVersion), maxVersion.get());
        this.addIssue(message, context, 
          ref.eContainingFeature(), IssueCodes.IDL_INVALID_VERSION);
      }
    }
  }
  
  /**
   * Checks that an explicit type version request is valid in the current context.
   */
  @Check
  public void checkExplicitVersionDeclaration(final VersionedReference ref) {
    boolean _allowVersionedTypes = this.variantHelper.allowVersionedTypes(ref);
    boolean _not = (!_allowVersionedTypes);
    if (_not) {
      return;
    }
    boolean _isVersionAwareContext = VersionUtils.isVersionAwareContext(ref);
    boolean _not_1 = (!_isVersionAwareContext);
    if (_not_1) {
      final String message = IssueCodes.getMessageForIDL_EXPLICIT_VERSION_DECLARATION_NOT_ALLOWED();
      this.addIssue(message, ref, TypeRefsPackage.Literals.VERSIONED_REFERENCE__REQUESTED_VERSION, 
        IssueCodes.IDL_EXPLICIT_VERSION_DECLARATION_NOT_ALLOWED);
    }
  }
  
  /**
   * Checks that type declarations in language variants that allow versioned types
   * always explicitly declare a type version.
   */
  @Check
  public void checkTypeDeclaration(final N4TypeDeclaration n4TypeDeclaration) {
    boolean _allowVersionedTypes = this.variantHelper.allowVersionedTypes(n4TypeDeclaration);
    boolean _not = (!_allowVersionedTypes);
    if (_not) {
      return;
    }
    if ((n4TypeDeclaration instanceof VersionedElement)) {
      if (((!((VersionedElement)n4TypeDeclaration).hasDeclaredVersion()) && (!VersionUtils.hasVersionAwarenessAnnotation(n4TypeDeclaration)))) {
        final String message = IssueCodes.getMessageForIDL_VERSIONED_ELEMENT_MISSING_VERSION(
          this.elementKeywordProvider.keyword(n4TypeDeclaration), n4TypeDeclaration.getName());
        this.addIssue(message, n4TypeDeclaration, N4JSPackage.Literals.N4_TYPE_DECLARATION__NAME, 
          IssueCodes.IDL_VERSIONED_ELEMENT_MISSING_VERSION);
      }
      if ((((VersionedElement)n4TypeDeclaration).hasDeclaredVersion() && VersionUtils.hasVersionAwarenessAnnotation(n4TypeDeclaration))) {
        this.addIssue(IssueCodes.getMessageForIDL_VERSION_AWARE_CLASSIFIER_MUST_NOT_DECLARE_VERSION(), n4TypeDeclaration, 
          N4JSPackage.Literals.N4_TYPE_DECLARATION__NAME, 
          IssueCodes.IDL_VERSION_AWARE_CLASSIFIER_MUST_NOT_DECLARE_VERSION);
      }
    }
  }
  
  /**
   * Adds an issue in case of missing support for type versions in
   * the current JavaScript variant.
   * 
   * This validation only applies to {@link VersionedElement}s (e.g. classifier, enum declarations).
   */
  @Check
  public void checkVersionedElements(final VersionedElement versionedElement) {
    boolean _allowVersionedTypes = this.variantHelper.allowVersionedTypes(versionedElement);
    boolean _not = (!_allowVersionedTypes);
    if (_not) {
      boolean _isVersioned = VersionUtils.isVersioned(versionedElement);
      if (_isVersioned) {
        this.addIssueForUnsupportedVersionedTypes(versionedElement);
      }
    }
  }
  
  /**
   * Adds an issue in case of missing support for type versions in
   * the current JavaScript variant.
   * 
   * This validation only applies to {@link VersionedReference}s (e.g. TypeRef, IdentifierRef).
   */
  @Check
  public void checkVersionedReferencesSupported(final VersionedReference versionedReference) {
    if (((!this.variantHelper.allowVersionedTypes(versionedReference)) && 
      versionedReference.hasRequestedVersion())) {
      this.addIssueForUnsupportedVersionedTypes(versionedReference);
    }
  }
  
  /**
   * Adds an issue for the element, indicating that versioned types
   * are not supported in the current JavaScript variant.
   */
  private void addIssueForUnsupportedVersionedTypes(final EObject element) {
    final String variantName = this.variantHelper.getVariantName(element);
    this.addIssue(IssueCodes.getMessageForIDL_VERSIONED_TYPES_NOT_SUPPORTED(variantName), element, 
      IssueCodes.IDL_VERSIONED_TYPES_NOT_SUPPORTED);
  }
  
  /**
   * Checks whether the current {@link JavaScriptVariant} allows for top-level statements and issues errors
   * accordingly.
   */
  @Check
  public void checkTopLevelElements(final Script script) {
    boolean _allowTopLevelStatements = this.variantHelper.allowTopLevelStatements(script);
    boolean _not = (!_allowTopLevelStatements);
    if (_not) {
      final Consumer<ScriptElement> _function = (ScriptElement it) -> {
        this.handleScriptElement(it);
      };
      script.getScriptElements().stream().forEach(_function);
    }
  }
  
  /**
   * Adds an issue for the given element if it is considered a top-level statement (also see {@link #isStatement}).
   */
  private void handleScriptElement(final ScriptElement element) {
    if ((this.isStatement(element) && (element.eContainer().eContainer() == null))) {
      boolean _isConstVariableDeclaration = this.isConstVariableDeclaration(element);
      if (_isConstVariableDeclaration) {
        return;
      }
      final String variantName = this.variantHelper.getVariantName(element);
      this.addIssue(IssueCodes.getMessageForAST_TOP_LEVEL_STATEMENTS(variantName), element, 
        IssueCodes.AST_TOP_LEVEL_STATEMENTS);
    }
  }
  
  /**
   * Returns {@code true} iff the given element is a variable declaration using the {@code const} keyword.
   */
  private boolean isConstVariableDeclaration(final EObject element) {
    return ((element instanceof VariableStatement) && Objects.equal(((VariableStatement) element).getVarStmtKeyword(), VariableStatementKeyword.CONST));
  }
  
  /**
   * Returns {@code true} if the given element is a considered a statement.
   */
  private boolean isStatement(final EObject element) {
    return ((element instanceof Statement) && (!(element instanceof FunctionDeclaration)));
  }
}
